#!/usr/bin/env python3
#%%

import pandas
import cropsar

# - cloudfree_s2_df: Sentinel-2 input data with cloud-masking already applied,
#   and structured as a DataFrame containing one or more fields:
# 
#                 date-1   date-2   date-3  ....
#        field-1   ...      ...      ...
#        field-2   ...      ...      ...
#         ...      ...      ...      ...
#      
#   this should have sufficient data before and after the date range of
#   interest (depending on the model, use cropsar.prepare_date_range to
#   generate the extended date range)
#
# - s1_df: Sentinel-1 input data for VV, VH, and optionally angle,
#   structured as as multilevel DataFrame containing multiple fields:
#      
#                         date-1   date-2   date-3  ....
#        field-1  vh       ...      ...      ...
#                 vv       ...      ...      ...
#               [ angle    ...      ...      ... ]
#        field-2  vh       ...      ...      ...
#                 vv       ...      ...      ...
#               [ angle    ...      ...      ... ]
#         ...     ...      ...      ...      ...
#
#   this should have sufficient data before and after the date range of
#   interest (depending on the model, use cropsar.prepare_date_range to
#   generate the extended date range)
#
# - date_range: a [start, end] tuple/list
#
#   this should be the actual date range of interest (so not the extended
#   date range generated by cropsar.prepare_date_range)
#
# - s2_data_layer: any of ['S2_FAPAR', 'S2_FCOVER', 'S2_NDVI']
#
# - s1_var: any of ['gamma', 'sigma'], use of 'sigma' is not recommended
#
# - minimal_input_checks: by enabling this cropsar will be more forgiving
#     withrespect to input data, but may generate unreliable results

def run_cropsar_dataframes(cloudfree_s2_df, s1_df, date_range=None, s2_data_layer='S2_FAPAR', s1_var='gamma', minimal_input_checks=False):

    start = pandas.to_datetime(date_range[0]).strftime('%Y-%m-%d')
    end   = pandas.to_datetime(date_range[1]).strftime('%Y-%m-%d')

    q10 = pandas.DataFrame()
    q50 = pandas.DataFrame()
    q90 = pandas.DataFrame()

    for field_index in s1_df.index.get_level_values(0).unique():

        s2_row = pandas.DataFrame(cloudfree_s2_df.loc[field_index].T)
        s1_row = pandas.DataFrame(s1_df.loc[field_index].T)

        result = cropsar.generate_timeseries(s1_data_df=s1_row,
                                             s2_data_df=s2_row,
                                             s2_scene_dfs_per_border=None,
                                             s2_data_layer=s2_data_layer,
                                             start_date=start,
                                             end_date=end,
                                             params=None,
                                             s1_var=s1_var,
                                             field_id=field_index,
                                             minimal_input_checks=minimal_input_checks)

        q10[field_index] = result['cropsar'].q10
        q50[field_index] = result['cropsar'].q50
        q90[field_index] = result['cropsar'].q90
    
    return q50, q10, q90



# This is a simple program to collect some test-data and structure
# it so we can pass it to run_cropsar_dataframes...

if __name__ == '__main__':
    
    import shapely
    import matplotlib.pyplot as plt

    # For some field, collect input data from the MEP
    # timeseries-service and generate a cropsar timeseries

    product = 'S2_FAPAR'
    start = '2019-01-01'
    end   = '2019-12-31'

    geometry = shapely.geometry.shape({
        'type': 'Polygon',
        'coordinates': ((
            (646683.9409017033, 5683354.097000904),
            (646673.4757396372, 5683407.323430645),
            (646754.1554968565, 5683476.354438913),
            (646720.4389642378, 5683365.705344125),
            (646683.9409017033, 5683354.097000904)
        ),)
    })

    crs = 'epsg:32631'

    analysis = cropsar.retrieve_cropsar_analysis(product,
                                                 start,
                                                 end, 
                                                 geometry,
                                                 crs)

    cropsar_ref = analysis['cropsar'].q50
    vh = analysis['sources']['s1-data'].vh
    vv = analysis['sources']['s1-data'].vv
    data = analysis['sources']['s2-data'].data

    # In case of openeo, cloud masking is already applied
    # to the S2 source data.  However, to simulate this 
    # we can't simply take the analysis['clean']['s2-data']
    # because cropsar performs some additional cleaning steps.
    # So check the flags in the cleaned data to see exactly
    # which samples have to be removed.

    flag = analysis['clean']['s2-data'].flag

    data[(flag == 21) | (flag == 22)] = None

    # At this point we have all cropsar inputs and outputs
    # for one field, now simply duplicate this data to
    # build multi-field dataframes for S1 and S2

    s1_dfs = {}
    s2_dfs = {}

    for field_id in ['field-1', 'field-2', 'field-3']:
        s1_dfs[(field_id, 'vh')] = vh.to_dict()
        s1_dfs[(field_id, 'vv')] = vv.to_dict()
        s2_dfs[field_id] = data.to_dict()

    s1_df = pandas.DataFrame.from_dict(s1_dfs, orient='index')
    s2_df = pandas.DataFrame.from_dict(s2_dfs, orient='index')

    # Now call the run_cropsar_dataframes() function for all
    # fields at once

    cropsar_dfs = run_cropsar_dataframes(cloudfree_s2_df=s2_df,
                                         s1_df=s1_df,
                                         date_range=(start, end),
                                         s2_data_layer=product,
                                         minimal_input_checks=False)
    
    cropsar_df = cropsar_dfs[0]

    # Plot the result for the first (duplicated) field and the
    # cropsar timeseries we acquired earlier. 

    fig = plt.figure()
    ax = fig.subplots(1, 1)
    ax.set_title('CropSAR retrieve_analysis vs run_cropsar_dataframes')
    cropsar_ref.plot(ax=ax, label='retrieve_analysis')
    cropsar_df['field-1'].plot(ax=ax, label='run_cropsar_dataframes field-1')
    cropsar_df['field-2'].plot(ax=ax, label='run_cropsar_dataframes field-2')
    cropsar_df['field-3'].plot(ax=ax, label='run_cropsar_dataframes field-3')
    ax.legend()

    plt.show()



#import json
#import logging
#import shutil
#import tempfile
#from pathlib import Path
#from typing import List
#
#import numpy
#import pandas
#import pandas as pd
#from openeo.metadata import CollectionMetadata
#from openeo.rest.datacube import DataCube
#from shapely.geometry import MultiPolygon, GeometryCollection
#
#from cropsar import prepare_date_range
#from cropsar.preprocessing.cloud_mask_openeo import evaluate
#
#
#_log = logging.getLogger(__name__)
#
#
#class InProcessSession:
#    """Not a real session but straight function calls."""
#
#    def __init__(self):
#        self.imagecollection = self.load_collection
#
#    def load_collection(self, collection_id, bands=None):
#        return DataCube.load_collection(collection_id, connection=self, bands=bands)
#
#    def collection_metadata(self, collection_id):
#        import openeogeotrellis.layercatalog
#        catalog = openeogeotrellis.layercatalog.get_layer_catalog()
#        return CollectionMetadata(catalog.get_collection_metadata(collection_id))
#
#    def list_output_formats(self) -> dict:
#        return {
#            "JSON": {
#              "gis_data_types": [
#                "raster"
#              ],
#              "parameters": {}
#            },
#            "NetCDF": {
#              "gis_data_types": [
#                "other",
#                "raster"
#              ],
#              "parameters": {},
#              "title": "Network Common Data Form"
#            }
#          }
#
#
#session = InProcessSession()
#
##available collections: http://openeo.vgt.vito.be/openeo/0.4.0/collections
## Relevant for CropSAR:
## S1_GRD_SIGMA0_ASCENDING
## S1_GRD_SIGMA0_DESCENDING
## S2_FAPAR_V102_WEBMERCATOR2
## S2_FAPAR_SCENECLASSIFICATION_V102_PYRAMID
#
#
#def preprocess_file(in_file, out_file,buffer=-15,size_threshold=1000,skip_overlaps=True):
#    import geopandas as gpd
#    #openeohelpers == https://git.vito.be/projects/BIGGEO/repos/openeo-python-helpers/browse
#
#
#    def buffer_geometry(df, distance, intermediate_crs='epsg:32631', **kwargs):
#        # We use an intermediate CRS to be able to express distance in more
#        # convenient units.  By default this CRS is UTM zone 31N, so units
#        # are expressed in meter.
#        #
#        # Optional kwargs are passed to shapely's object.buffer() function.
#        #
#        # For more information, see:
#        #
#        #  https://shapely.readthedocs.io/en/latest/manual.html#object.buffer
#
#        intermediate_df = df.to_crs(intermediate_crs)
#        intermediate_df.geometry = intermediate_df.buffer(distance, **kwargs)
#
#        return intermediate_df.to_crs(df.crs)
#
#    fields = gpd.read_file(in_file)
#
#    buffered_parcels = buffer_geometry(fields, buffer)
#
#    buffered_parcels = buffered_parcels[buffered_parcels.area > size_threshold]
#    buffered_parcels = buffered_parcels.to_crs('epsg:4326')
#
#    if not skip_overlaps:
#        from openeohelpers import filter_overlaps
#        filtered = filter_overlaps(buffered_parcels)
#    else:
#        filtered = buffered_parcels
#
#    filtered.to_file(out_file, driver="GeoJSON")
#    minx, miny, maxx, maxy = filtered.geometry.total_bounds
#    return (minx, miny, maxx, maxy)
#
#
#
#def retrieve_timeseries(collection,parcels_file, start, end, output_file: Path = Path("timeseries.json"),env=None):
#    datacube = session.load_collection(collection)
#
#    time_series = datacube\
#        .filter_temporal(start, end)\
#        .polygonal_mean_timeseries(parcels_file)\
#        .save_result(format="json")
#
#    evaluate(time_series, output_file,env=env)
#
#
#from cropsar.preprocessing.cloud_mask_openeo import retrieve_timeseries as retrieve_clean_fapar
#def retrieve_all_timeseries(parcels_file, start, end, output_dir: Path,use_gamma=True,params={},env=None):
#    """
#    Retrieve all input timeseries.
#    :param parcels_file:
#    :param start:
#    :param end:
#    :return:
#    """
#
#    print("Start preprocessing of cropsar geometries!")
#    prepared_geometry_list = _prepare_cropsar_geometries(parcels_file)
#    print("Done preprocessing geometries.")
#
#    start_date = pd.to_datetime(start).date()
#    end_date = pd.to_datetime(end).date()
#
#    current_date = start_date
#    while current_date < end_date:
#        next_date = current_date + pd.DateOffset(months=5)
#        if next_date > end_date:
#            next_date = end_date
#            upper = next_date
#        else:
#            upper = next_date - pd.DateOffset(days=1)
#
#        print(current_date)
#        print(upper)
#
#        postfix = "_%s_%s.json" % (current_date.strftime('%Y%m%d'), upper.strftime('%Y%m%d'))
#
#        if use_gamma:
#            asc_output_file = output_dir / ("S1_GAMMA0" + postfix)
#            print("retrieving Gamma0 timeseries for: %s to %s " % (str(current_date),str(upper)))
#            retrieve_timeseries("TERRASCOPE_S1_GAMMA0_V1", prepared_geometry_list, current_date, upper,
#                                output_file=asc_output_file,env=env)
#        else:
#            asc_output_file = output_dir / ("S1_ASCENDING" + postfix)
#            desc_output_file = output_dir / ("S1_DESCENDING" + postfix)
#            retrieve_timeseries("S1_GRD_SIGMA0_ASCENDING", prepared_geometry_list, current_date, upper, output_file=asc_output_file,env=env)
#            retrieve_timeseries("S1_GRD_SIGMA0_DESCENDING", prepared_geometry_list, current_date, upper, output_file=desc_output_file,env=env)
#
#        fapar_output_file = output_dir / ("FAPAR_CLEAN" + postfix)
#        print("retrieving FAPAR timeseries for: %s to %s " % (str(current_date), str(upper)))
#        retrieve_clean_fapar(prepared_geometry_list, current_date, upper, session, output_file=fapar_output_file,params=params,env=env)
#
#        current_date = next_date
#
#
#def _prepare_cropsar_geometries(parcels_file) -> GeometryCollection:
#    from openeo_driver.delayed_vector import DelayedVector
#
#    from cropsar._crs import auto_utm_crs_for_geometry
#    from cropsar._generate import _get_param
#    import shapely.geometry
#    parcel_reader = DelayedVector(parcels_file)
#    gdf = parcel_reader.as_geodataframe()
#    bbox = shapely.geometry.box(*gdf.total_bounds)
#    crs = gdf.crs
#    print("Found this CRS in the input data: %s." % str(crs))
#    #https://geopandas.org/projections.html#upgrading-to-geopandas-0-7-with-pyproj-2-2-and-proj-6
#    if not isinstance(crs,dict):
#        crs = crs.to_dict()
#    crs_utm = auto_utm_crs_for_geometry(bbox, crs)
#
#    border = -_get_param({}, 's2_clean', 'inarrowfieldbordersinmeter')
#    buffer_args = {'cap_style': 1, 'join_style': 2, 'resolution': 4}
#    if crs == crs_utm:
#        gdf_buffer = gdf.buffer(border,**buffer_args)
#        print("Total number of polygons: %s with area: %s" % (str(gdf.length), str(gdf.area.sum())))
#    else:
#        gdf_utm = gdf.to_crs(crs_utm)
#        gdf_utm = gdf_utm.buffer(border,**buffer_args)
#        print("Total number of polygons: %s with area: %s" % (str(gdf_utm.length), str(gdf_utm.area.sum())))
#        gdf_buffer = gdf_utm.to_crs(gdf.crs)
#
#    empty_geoms = gdf.geometry.loc[gdf_buffer.geometry.is_empty]
#    if len(empty_geoms)>0:
#        gdf_buffer.geometry.loc[gdf_buffer.geometry.is_empty] = empty_geoms
#
#    return GeometryCollection(gdf_buffer.geometry.array)
#
#
#def retrieve_cropsar(parcels_file, start:str, end:str,use_gamma=False, working_directory=None, params={},env=None) -> List[Path]:
#    """
#    Retrieve cropsar curves for all polygons in a given file. This method is invoked by the openEO cropsar process.
#
#    :param parcels_file: shp or geojson file with polygons
#    :param start: start date string, formatted as '%Y-%m-%d'
#    :param end: start date string, formatted as '%Y-%m-%d'
#    :return: list of paths containing cropsar results
#    """
#    if working_directory is None:
#        input_timeseries_dir = Path(tempfile.mkdtemp(prefix="in_", dir=Path.cwd()))
#    else:
#        input_timeseries_dir = Path(working_directory)
#
#    print("Running CropSAR process on: %s, start date: %s, end date: %s " % (str(parcels_file),str(start),str(end)))
#    print("Working directory: %s" % str(input_timeseries_dir))
#
#    start,end = prepare_date_range(start,end)
#    
#    retrieve_all_timeseries(parcels_file, start, end, input_timeseries_dir,use_gamma,params=params,env=env)
#
#    if use_gamma:
#        gamma_input_glob = str(input_timeseries_dir / "S1_GAMMA0*.json")
#        asc_input_glob = desc_input_glob = None
#    else:
#        asc_input_glob = str(input_timeseries_dir / "S1_ASCENDING_*.json")
#        desc_input_glob = str(input_timeseries_dir / "S1_DESCENDING_*.json")
#        gamma_input_glob = None
#
#    fapar_input_glob = str(input_timeseries_dir / "FAPAR_CLEAN_*.json")
#
#    if working_directory is None:
#        output_cropsar_dir = Path(tempfile.mkdtemp(prefix="out_", dir=Path.cwd()))
#    else:
#        output_cropsar_dir = Path(working_directory)
#
#    try:
#        return run_cropsar_on_files(asc_input_glob, desc_input_glob, fapar_input_glob, output_cropsar_dir,gamma0_glob=gamma_input_glob,include_inputs=params.get('include_inputs',False))
#    finally:
#        if working_directory is None:
#            shutil.rmtree(input_timeseries_dir, ignore_errors=True)
#
#
#def run_cropsar_on_files(s1_asc_glob: str, s1_desc_glob: str, S2_FAPAR_clean_glob: str, output_dir: Path = Path.cwd(), gamma0_glob = None, include_inputs = False) -> List[Path]:
#    from openeo.rest.conversions import timeseries_json_to_pandas
#
#    #import cropsar.service.api as api
#    import pandas as pd
#    import glob
#
#    def load_ts_json(glob_pattern):
#
#        filenames = glob.glob(glob_pattern)
#
#        df = pd.DataFrame()
#
#        for file in filenames:
#            with open(file, "r") as f:
#                ts_json = json.load(f)
#            if len(ts_json) > 0:
#                ts_dataframe = timeseries_json_to_pandas(ts_json, auto_collapse=False)
#                ts_dataframe.index = pd.to_datetime(ts_dataframe.index).date
#                df = df.append(ts_dataframe)
#            else:
#                print("Skipping file with empty json: %s" % file)
#        return df.sort_index().T
#
#    if gamma0_glob is None:
#        df_S1_asc = load_ts_json(s1_asc_glob)
#        df_S1_desc = load_ts_json(s1_desc_glob)
#    else:
#        df_gamma0 = load_ts_json(gamma0_glob)
#        df_S1_asc = df_gamma0
#        df_S1_desc = None
#
#    df_S2_clean = load_ts_json(S2_FAPAR_clean_glob)
#
#    #scaling for S2 fapar
#    df_S2_clean *= 0.005
#
#    cropsar_result = run_cropsar_dataframes(df_S2_clean, df_S1_asc, df_S1_desc, output_dir)
#    if include_inputs:
#        cropsar_result.extend(_write(df_S2_clean,"S2_FAPAR",output_dir))
#        if df_S1_desc is None:
#            cropsar_result.extend(_write(df_S1_asc,"S1_GAMMA0",output_dir))
#        else:
#            cropsar_result.extend(_write(df_S1_desc, "S1_SIGMA0_DESCENDING", output_dir))
#            cropsar_result.extend(_write(df_S1_asc, "S1_SIGMA0_ASCENDING", output_dir))
#
#    return cropsar_result
#
#def _write(df,name, output_dir:Path) -> List:
#    out_file = output_dir / ("%s.csv"%name)
#    df.to_csv(out_file, date_format='%Y-%m-%d')
#    return [out_file]
#
#def run_cropsar_dataframes(df_S2_clean, df_S1_asc, df_S1_desc, output_dir=None, scale=0.0005, offset=29, date_range=None,spark = False):
#    import cropsar
#
#    if date_range:
#        # Use flexible `pd.to_datetime` to parse the date range
#        start, end = [pd.to_datetime(d).date() for d in date_range]
#    else:
#        # This is original behavior, but probably gives wrong results. See https://jira.vito.be/browse/EP-4198
#        start = df_S1_asc.columns.min()
#        end = df_S1_asc.columns.max()
#        _log.warning(f"Extracted start and end date from dataframes, which includes margin: {start} {end}")
#    # s2_data_layer = api._map_ts_product("S2_FAPAR")  # For CropSAR RNN model scalers
#    s2_data_layer = "S2_FAPAR"
#    cropsar_df = pd.DataFrame()
#    cropsar_df_q10 = pd.DataFrame()
#    cropsar_df_q90 = pd.DataFrame()
#
#    def outlier_filter(cleanfielddatatimeseries, start_date, end_date):
#
#        from s2_clean import smooth
#        from cropsar._generate import _get_param
#
#        params = None
#        localminimamaxdip = _get_param(params, 's2_clean', 'localminimamaxdip')
#        localminimamaxdif = _get_param(params, 's2_clean', 'localminimamaxdif')
#        localminimamaxgap = _get_param(params, 's2_clean', 'localminimamaxgap')
#        localminimamaxpas = _get_param(params, 's2_clean', 'localminimamaxpas')
#
#        cleanfieldfulldatatimeseries = pandas.Series(
#            index=pandas.date_range(start=start_date, end=end_date), dtype=float)
#        cleanfielddatatimeseries.index = pd.to_datetime(cleanfielddatatimeseries.index)
#        ts_intersection = cleanfielddatatimeseries[start_date:end_date]
#        cleanfieldfulldatatimeseries.loc[ts_intersection.index] = ts_intersection
#        smooth.flaglocalminima(cleanfieldfulldatatimeseries.values, maxdipvalueornumpycube=localminimamaxdip,
#                               maxdifvalueornumpycube=localminimamaxdif,
#                               maxgap=localminimamaxgap, maxpasses=localminimamaxpas)
#        dippedindices = cleanfieldfulldatatimeseries.loc[cleanfieldfulldatatimeseries.isnull()].index.intersection(
#            cleanfielddatatimeseries.loc[cleanfielddatatimeseries.notnull()].index)
#        cleanfielddatatimeseries.loc[dippedindices] = numpy.nan
#        cleanfielddatatimeseries.dropna(inplace=True)
#        return cleanfielddatatimeseries
#
#    if spark:
#        # os.environ['PYSPARK_PYTHON'] = '/usr/bin/python3.6'
#        from pyspark import SparkContext
#
#        spark = SparkContext.getOrCreate()
#        my_list = [{'fieldid': field_index,
#                    's2_data': pd.DataFrame(df_S2_clean.loc[field_index].T),
#                    's1_asc': df_S1_asc.loc[field_index].T,
#                    's1_desc': df_S1_desc.loc[field_index].T if df_S1_desc is not None else None,
#
#                    'date_start': start.strftime('%Y-%m-%d'),
#                    'date_end': end.strftime('%Y-%m-%d')
#                    } for field_index in df_S1_asc.index.get_level_values(0).unique()]
#
#        def do_cropsar(item):
#            ascending_row = item['s1_asc']
#            descending_row = item['s1_desc']
#            s2_row = item['s2_data']
#
#            logging.basicConfig(level=logging.DEBUG)
#
#            combined_orbits_row = _create_combined_sigma0(ascending_row, descending_row, scale, offset)
#
#            s2_row.columns = ['clean']
#            s2_row['clean'] = outlier_filter(s2_row['clean'], item['date_start'], item['date_end'])
#            result = cropsar.generate_timeseries(combined_orbits_row, s2_row, None, s2_data_layer,
#                                                 item['date_start'], item['date_end'],
#                                                 s1_var='sigma' if descending_row is not None else 'gamma')
#            return (item['fieldid'], result['cropsar'])
#
#        cropsar_ts = spark.parallelize(my_list, max(len(my_list) // 10,2)).map(do_cropsar).collectAsMap()
#
#        for field_index, result in cropsar_ts.items():
#            cropsar_df[field_index] = result.q50
#            cropsar_df_q10[field_index] = result.q10
#            cropsar_df_q90[field_index] = result.q90
#
#    else:
#
#        for field_index in df_S1_asc.index.get_level_values(0).unique():
#            s2_row = pd.DataFrame(df_S2_clean.loc[field_index].T)
#            combined_orbits_row = _create_combined_sigma0(df_S1_asc.loc[field_index].T, df_S1_desc.loc[
#                field_index].T if df_S1_desc is not None else None, scale, offset)
#
#            s2_row.columns = ['clean']
#            s2_row['clean'] = outlier_filter(s2_row['clean'], start, end)
#
#            result = cropsar.generate_timeseries(combined_orbits_row, s2_row, None, s2_data_layer,
#                                                 start.strftime('%Y-%m-%d'), end.strftime('%Y-%m-%d'),
#                                                 s1_var='sigma' if df_S1_desc is not None else 'gamma')
#
#            cropsar_df[field_index] = result['cropsar'].q50
#            cropsar_df_q10[field_index] = result['cropsar'].q10
#            cropsar_df_q90[field_index] = result['cropsar'].q90
#            
#    if output_dir == None:
#        return cropsar_df, cropsar_df_q10, cropsar_df_q90
#    else:
#
#        q50_json_output_file = output_dir / "cropsar.json"
#        q10_json_output_file = output_dir / "cropsar_q10.json"
#        q90_json_output_file = output_dir / "cropsar_q90.json"
#
#        cropsar_df.to_json(q50_json_output_file, date_format='%Y-%m-%d')
#        cropsar_df_q10.to_json(q10_json_output_file, date_format='%Y-%m-%d')
#        cropsar_df_q90.to_json(q90_json_output_file, date_format='%Y-%m-%d')
#
#        q50_csv_output_file = output_dir / "cropsar.csv"
#        q10_csv_output_file = output_dir / "cropsar_q10.csv"
#        q90_csv_output_file = output_dir / "cropsar_q90.csv"
#
#        cropsar_df.to_csv(q50_csv_output_file, date_format='%Y-%m-%d')
#        cropsar_df_q10.to_csv(q10_csv_output_file, date_format='%Y-%m-%d')
#        cropsar_df_q90.to_csv(q90_csv_output_file, date_format='%Y-%m-%d')
#
#        return [
#            q50_csv_output_file,
#            q10_csv_output_file,
#            q90_csv_output_file,
#            q50_json_output_file,
#            q10_json_output_file,
#            q90_json_output_file
#        ]
#
#
#def _create_combined_sigma0(ascending_row, descending_row, scale=0.0005, offset=29):
#    """
#
#    @param ascending_row:  ascending data or already combined gamma0 data
#    @param descending_row: descending data or None if data is already combined
#    @return:
#    """
#    if descending_row is None:
#
#        ascending_row.columns = ['vh','vv']
#        ascending_row['angle'] = 0.0
#        return ascending_row
#    else:
#        ascending_row.columns = ['vh', 'vv', 'angle']
#        descending_row.columns = ['vh', 'vv', 'angle']
#        ascending_row['angle'] = scale * ascending_row['angle'] + offset
#        descending_row['angle'] = scale * descending_row['angle'] + offset
#        combined_orbits_row = pd.concat((ascending_row, descending_row)).groupby(level=0).mean()
#        return combined_orbits_row
#
#
#def get_result(job_id:str,session):
#    from openeo.rest.job import RESTJob
#    job=RESTJob(id,session)
#    job.download_results("out.json")
#
#
#if __name__ == '__main__':
#    import fire
#
#    fire.Fire({"cropsar_on_files": run_cropsar_on_files,"retrieve_all_timeseries": retrieve_all_timeseries})
#
##    start = '2019-02-01'
##    end = '2019-12-31'
#
##    parcels_file = "/data/CapSat/DLV/EPR/DLV2018_maize.gpkg"
##    parcels_file_processed = "/data/users/Public/driesj/hpg_flanders_v3_filtered.geojson"
#
#    # IMPORTANT: make sure parcels files do not overlap
#    #minx, miny, maxx, maxy = preprocess_file(parcels_file, parcels_file_processed)
#
##    retrieve_all_timeseries(parcels_file_processed,start,end)
#
# %%
