######################################################################################################
# biomassproduction.py
# ---------------------------
# Purpose
#       Build OpenEO biomass production service
#
######################################################################################################

#imports
from pathlib import Path

import pandas as pd
from openeo.api.process import Parameter
import shapely
import openeo
import geopandas as gpd
import logging
import json
import os
import pandas as pd
from openeo.rest.udp import build_process_dict
from nextland_services.helper_functions import get_sentinel_datacubes_for_cropsar, preprocessing_datacube_for_udf, load_udf
from nextland_services.constants import *




_log = logging.getLogger("biomass_production")

#Generate results paths
OUTPUT_DIR = Path(__file__).with_suffix('.output')
OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
OUTPUT_DIR_GRAPH = RESOURCES
BASEPATH = Path(__file__).parent

#Write content to named file
def fwrite(fname, fcontent):
    f = open(fname, 'w')
    f.write(str(fcontent))
    f.close()


def run_udf_poly(polygon,time_range,eoconn, bbox = False):

    timeseries_input = load_biomass_timeseries_input_poly(polygon,time_range,eoconn, bbox, biopar_fixed= True)

    dmp_code = load_udf("biomass_udf.py")
    udf =  timeseries_input.process("run_udf", data = timeseries_input, udf = dmp_code, runtime = "Python", context = dict({'date': time_range}))
    TS =  udf.execute()#.send_job().start_and_wait().get_result().load_json()

    with open(os.path.join(OUTPUT_DIR, 'biomass_result_poly_adapt.json'), 'w') as file:
        json.dump(TS, file)


def biomass_build_graph_poly(polygon,time_range,eoconn):

    #Define service parameters
    time_range = date_parameter("Left-closed temporal interval, i.e. an array with exactly two elements:\n\n1. The first element is the start of the temporal interval. The specified instance in time is **included** in the interval.\n2. The second element is the end of the temporal interval. The specified instance in time is **excluded** from the interval.\n\nThe specified temporal strings follow [RFC 3339](https://www.rfc-editor.org/rfc/rfc3339.html). Also supports open intervals by setting one of the boundaries to `null`, but never both.")
    polygon = polygon_param(description=" A polygon object for which the Biomass production will be determined.")
    timeseries_input = load_biomass_timeseries_input_poly(polygon,time_range,eoconn, biopar_fixed= True)

    #Load UDF
    dmp_code = load_udf("biomass_udf.py")
    udf = timeseries_input.process("run_udf", data=timeseries_input, udf=dmp_code, runtime="Python",
                                   context = {'date': {'from_parameter': 'date'}})


    #Build service dict
    from helpers import load_markdown
    biom_dict = build_process_dict(
            process_id="BIOMASS",
            process_graph=udf,
            description= load_markdown("biomass.md"),
            parameters=[time_range,polygon]
        )

    #Write service graph to json file
    fwrite(os.path.join(OUTPUT_DIR_GRAPH, 'biomass_graph.json'), json.dumps(biom_dict, indent=4))
    '''with open("biomass_graph_poly.json", "w") as f:
        json.dump(spec, f, indent=2)'''


def run_udf_bbox(bbox,time_range,eoconn):

    timeseries_input = load_biomass_timeseries_input_bbox(bbox,time_range,eoconn, bbox = True)

    dmp_code = load_udf("biomass_udf.py")
    udf = timeseries_input.process("run_udf", data=timeseries_input, udf=dmp_code, runtime="Python")

    udf.download(os.path.join(OUTPUT_DIR, 'biomass_result_bbox.json'), format="JSON")


def biomass_build_graph_bbox(bbox,time_range,eoconn):

    #Define service parameters
    time_range = Parameter(
        name="time_range", description="A time range for which to compute the Biomass production.",
        schema={"type": "temporal-intervals", "subtype": "date-time"}
    )
    bbox = Parameter(
        name="bbox", schema="object"
    )

    timeseries_input = load_biomass_timeseries_input_bbox(bbox,time_range,eoconn)

    #Load UDF
    dmp_code = load_udf("biomass_udf.py")
    udf = timeseries_input.process("run_udf", data=timeseries_input, udf=dmp_code, runtime="Python")


    #Build service dict
    biom_dict = build_process_dict(
        process_id="biomass",
        process_graph=udf,
        parameters=[time_range,bbox]
    )

    #Write service graph to json file
    fwrite(os.path.join(OUTPUT_DIR, 'biomass_graph_bbox.json'), json.dumps(biom_dict, indent=4))
    '''with open("biomass_graph_bbox.json", "w") as f:
        json.dump(spec, f, indent=2)'''

def get_input_datacube_udf(eoconn, time_range, geo, bbox, biopar_fixed):
    gamma0, fapar_masked = get_sentinel_datacubes_for_cropsar(eoconn, biopar_fixed  = biopar_fixed)
    meteo = eoconn.load_collection("AGERA5")
    meteo._pg.arguments['featureflags'] = {'tilesize': 1}
    merged_cube = gamma0.merge_cubes(fapar_masked).merge_cubes(meteo.resample_cube_spatial(gamma0))
    input_datacube = preprocessing_datacube_for_udf(merged_cube, time_range, geo, bbox)
    return input_datacube

def load_biomass_timeseries_input_poly(polygon,time_range,eoconn, bbox = False, biopar_fixed = False):
    input_datacube_udf = get_input_datacube_udf(eoconn, time_range, polygon, bbox, biopar_fixed)
    return input_datacube_udf

def load_biomass_timeseries_input_bbox(bbox,time_range,eoconn):
    input_datacube_udf = get_input_datacube_udf(eoconn, time_range, bbox, bbox = True)
    return input_datacube_udf

# def test_load_s2_data():
#     """
#     Tests downloading cropsar inputs, to verify if that works as expected.
#     The resulting NetCDF should contain 3 bands: FAPAR, GAMMA0_VV, GAMMA0_VH
#     @return:
#     """
#     time_range = "2020-05-06", "2020-05-30"
#     eoconn = openeo.connect('https://openeo.vito.be').authenticate_basic('bontek','bontek123')
#     bbox = {
#         "west": 5.15183687210083,
#         "east": 5.153381824493408,
#         "south": 51.18192559252128,
#         "north": 51.18469636040683,
#         "crs": "EPSG:4326"
#     }
#     polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
#                                         [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])
#     s2f = get_sentinel_bands_for_cropsar(eoconn, time_range, polygon, biomass= True)
#     s2f = s2f.send_job().start_and_wait().get_result().load_json()
#     with open(OUTPUT_DIR/"cropsar_inputs.json", 'w') as f:
#         json.dump(s2f, f)
#     #s2f.download(OUTPUT_DIR/"cropsar_inputs.json",format="json")


def test_load_timeseries_data():
    time_range = "2019-01-01", "2019-12-31"

    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    '''polygon = geojson.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                            [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])'''

    gpd_shp = gpd.read_file(r'/data/sigma/Nextland/BDB/In_situ/Carbon_sequestration_WGS84.shp')
    js_shp = json.loads(gpd_shp.to_json())
    polygon = js_shp['features'][0].get('geometry')

    #eoconn = openeo.connect('https://openeo-dev.vito.be').authenticate_basic()
    eoconn = openeo.connect('https://openeo.vito.be').authenticate_basic('bontek', 'bontek123')
    timeseries_input_poly = load_biomass_timeseries_input_poly(polygon,time_range,eoconn, biopar_fixed = True)
    timeseries_input_poly = timeseries_input_poly.send_job().start_and_wait().get_result().load_json()
    with open(OUTPUT_DIR/"biomass_inputs_poly_BDB_0.json", 'w') as f:
             json.dump(timeseries_input_poly, f)

    #timeseries_input_poly.download(OUTPUT_DIR/"biomass_inputs_poly.json",format="json")
    '''timeseries_input_bbox = load_biomass_timeseries_input_bbox(bbox,time_range,eoconn)
    timeseries_input_bbox.download(OUTPUT_DIR/"biomass_inputs_bbox.json",format="json")'''
    # js = ""
    # with open(OUTPUT_DIR/"biomass_inputs_poly_2.json") as jsf:
    #     js = json.load(jsf)
    # fwrite(os.path.join(OUTPUT_DIR, 'biomass_inputs_poly_2_indented.json'), json.dumps(js, indent=4))
    '''js = ""
    with open(OUTPUT_DIR/"biomass_inputs_bbox.json") as jsf:
        js = json.load(jsf)
    fwrite(os.path.join(OUTPUT_DIR, 'biomass_inputs_bbox_indented.json'), json.dumps(js, indent=4))'''

def test_load_meteo_data():
    eoconn = openeo.connect('https://openeo.vito.be').authenticate_basic('bontek', 'bontek123')
    time_range = "2020-09-01", "2020-09-30"

    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    meteo = eoconn.load_collection("AGERA5", bands  = ['dewpoint-temperature'])
    meteo._pg.arguments['featureflags'] = {'tilesize': 1}
    meteo = meteo.resample_spatial(resolution= 50.0, projection=32631)

    met = meteo.filter_temporal(time_range).polygonal_mean_timeseries(polygon)
    #met = met.send_job().start_and_wait().get_result().load_json()
    # with open(OUTPUT_DIR/"meteo.json", 'w') as f:
    #      json.dump(met, f)
    met.download(OUTPUT_DIR/"meteo2.json",format="json")


def test_debug_timeseries_udf():
    from openeo_udf.api.udf_data import UdfData
    from openeo_udf.api.structured_data import StructuredData
    from nextland_services.biomass_udf import udf_biomass
    time_range = '2018-11-19','2018-12-15'
    with open(OUTPUT_DIR/"biomass_inputs_poly_BDB_0.json","r") as f:
        input_dict = json.load(f)
        udfdata = UdfData(structured_data_list=[StructuredData(description="timeseries input", data=input_dict, type="dict")])
        udfdata.user_context =  {"date":time_range}
        res = udf_biomass(udfdata)
        fwrite(os.path.join(OUTPUT_DIR, 'biomass_testudf_result.res'), res.to_dict())


def test_run_udfs():
    logging.basicConfig(level=logging.INFO)
    _log = logging.getLogger("biomass_production")
    _log.info("Run UDF (bbox and polygon)")
    eoconn = openeo.connect("https://openeo.vito.be").authenticate_basic('bontek','bontek123')

    time_range = "2020-05-06", "2020-05-30"

    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    # eoconn = openeo.connect('https://openeo-dev.vito.be').authenticate_basic()
    eoconn = openeo.connect('https://openeo.vito.be').authenticate_basic('bontek','bontek123')
    # shapes.append(Path(BASEPATH).joinpath('data','D0911317_sup_depurada_WGS84.shp'))


    gpd_shp = gpd.read_file(Path(BASEPATH).joinpath('data', 'parcel-in-retie.shp'))
    js_shp = json.loads(gpd_shp.to_json())

    run_udf_poly(js_shp,time_range,eoconn)
    #run_udf_bbox(bbox,time_range,eoconn)


def test_build_biomass_graphs():
    logging.basicConfig(level=logging.INFO)
    _log = logging.getLogger("biomass_production")
    _log.info("Create Graph Biomass Production Service")
    eoconn = openeo.connect("https://openeo.vito.be").authenticate_basic('bontek','bontek123')

    time_range = "2020-05-06", "2020-05-30"

    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    '''poly = geojson.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                            [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])'''
    biomass_build_graph_poly(polygon=polygon,time_range=time_range,eoconn=eoconn)
    #biomass_build_graph_bbox(bbox=bbox,time_range=time_range, eoconn=eoconn)



def test_biomass_call():
    _log.info("Biomass (UDP call)")

    eoconn = openeo.connect("https://openeo.vito.be").authenticate_basic('bontek','bontek123')

    time_range = "2021-01-01", "2021-12-15"

    # bbox = {
    #     "west": 5.15183687210083,
    #     "east": 5.153381824493408,
    #     "south": 51.18192559252128,
    #     "north": 51.18469636040683,
    #     "crs": "EPSG:4326"
    # }
    #
    # poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])
    #gpd_shp = gpd.read_file(r'/data/sigma/Nextland/BDB/In_situ/Carbon_sequestration_WGS84.shp')
    #gpd_shp = gpd.read_file(r'/data/sigma/Nextland/QdC/In_situ/Parcels_QDC_merged.shp')
    #outdir_nextland = r'/data/sigma/Nextland/QdC/Products/DMP/'
    gpd_shp = gpd.read_file(r'/data/sigma/SES/Drought_detection/In_situ/2021/GPSLocation_Field_SoF_Satellite_buffer_20M.shp')
    outdir_nextland = r'//data/sigma/SES/Drought_detection/dmp/2021/'
    os.makedirs(outdir_nextland, exist_ok= True)
    js_shp = json.loads(gpd_shp.to_json())
    for poly in js_shp['features']:
        polyid = poly['properties']['Name'] #ID_LBST202
        polyg = poly['geometry']

        biom = eoconn.datacube_from_process("BIOMASS", date=time_range,
                                            polygon=polyg,namespace="nextland") #

        '''polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                            [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])
    
        poly = geojson.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                            [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])'''
        #biom = eoconn.datacube_from_process("biomass", namespace="nextland", time_range=time_range, bbox=bbox)
        #biom = eoconn.datacube_from_process("BIOMASS", namespace="nextland", time_range=time_range, polygon=shapely.geometry.mapping(poly))
        #biom.download(os.path.join(OUTPUT_DIR, "biom_res_BDB.json"), format="JSON")
        if not os.path.exists(os.path.join(outdir_nextland,"DMP_{}_2021.csv".format(str(polyid)))):
            try:
                BIOM_result = biom.send_job().start_and_wait().get_result().load_json()
            except Exception as e:
                print('CODE FAILED FOR FIELD: {} -> skip and continue for other fields'.format(str(polyid)))
                print('ERROR IS: {}'.format(print(e)))
                continue
            from openeo.rest.conversions import timeseries_json_to_pandas
            df_DMP = pd.DataFrame.from_dict(BIOM_result)
            df_DMP.to_csv(os.path.join(outdir_nextland,"DMP_{}_2021.csv".format(str(polyid))), index = True)
            # with open(outdir_nextland/"DMP_2017_now_{}.json".format(polyid), 'w') as f:
            #     json.dump(BIOM_result, f)


def save_udp():
    eoconn = openeo.connect("https://openeo.vito.be").authenticate_basic('bontek','bontek123')

    time_range = date_parameter("Left-closed temporal interval, i.e. an array with exactly two elements:\n\n1. The first element is the start of the temporal interval. The specified instance in time is **included** in the interval.\n2. The second element is the end of the temporal interval. The specified instance in time is **excluded** from the interval.\n\nThe specified temporal strings follow [RFC 3339](https://www.rfc-editor.org/rfc/rfc3339.html). Also supports open intervals by setting one of the boundaries to `null`, but never both.")
    polygon = polygon_param(description=" A polygon object for which the Biomass production will be determined.")

    with open(os.path.join(OUTPUT_DIR_GRAPH,'biomass_graph.json'),'r') as file:
        graph = json.load(file)
    udp = eoconn.save_user_defined_process(
        'BIOMASS',
        graph["process_graph"],
        description= "Calculates the biomass curve for each field over the given time range.",
        parameters=[time_range, polygon], public=True)

def test_use_shape():

    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic('bontek','bontek123')
    time_range = "2020-01-01", "2020-12-31"

    import geopandas as gpd

    shapes = []

    shapes.append(Path(BASEPATH).joinpath('data','parcel-in-retie.shp'))
    #shapes.append(Path(BASEPATH).joinpath('data','D0911317_sup_depurada_WGS84.shp'))

    for shp in shapes:
        gpd_shp = gpd.read_file(shp)
        js_shp = json.loads(gpd_shp.to_json())
        shp_nm = Path(shp).stem

        for poly in js_shp['features']:
            polyid = poly['id']
            polyg = poly['geometry']

            biom = eoconn.datacube_from_process("BIOMASS", date=time_range,
                                                polygon=polyg) #namespace="nextland"
            BIOM_result = biom.send_job().start_and_wait().get_result().load_json()
            with open(OUTPUT_DIR/"biomass_reference_result_DMP.json", 'w') as f:
                json.dump(BIOM_result, f)
            # fwrite(os.path.join(OUTPUT_DIR, 'biom_CALL_%s_ID-%s_0001.json' % (shp_nm, polyid)),
            #        biom.to_json())
            # biom.download(os.path.join(OUTPUT_DIR, 'biom_%s_ID-%s_0001.tiff' % (shp_nm, polyid)),
            #               format="GTIFF")
            # biom.download(os.path.join(OUTPUT_DIR, 'biom_%s_ID-%s_0001.nc' % (shp_nm, polyid)),
            #               format="NetCDF")
            # biom.download(os.path.join(OUTPUT_DIR, 'biom_%s_ID-%s_0001.json' % (shp_nm, polyid)),
            #               format="JSON")
#test_load_meteo_data()
#test_load_timeseries_data()
#test_debug_timeseries_udf()
#test_build_biomass_graphs()
#save_udp()
#test_biomass_call()
#test_use_shape()



