#!/usr/bin/env python3

import logging
import collections.abc

from concurrent.futures import ThreadPoolExecutor

import requests
import pandas as pd
import shapely.ops
import shapely.geometry

from ._crs import *
from ._generate import *
from ._version import __version__


log = logging.getLogger(__name__)


def _map_ts_product(product):

    v2_products = {
        'S2_FAPAR':                  'TERRASCOPE_S2_FAPAR_V2',
        'S2_FCOVER':                 'TERRASCOPE_S2_FCOVER_V2',
        'S2_NDVI':                   'TERRASCOPE_S2_NDVI_V2',
        'S2_SCENECLASSIFICATION':    'S2_SCENECLASSIFICATION_V200_FILE'
    }

    return v2_products.get(product, product)


def _map_fis_product(product):
    return product


def _to_crs(geometry, from_crs, to_crs):
    geometry = geometry_to_crs(geometry, from_crs, to_crs)

    return geometry, to_crs


def _get_ts_timeseries(product, start, end, geometry, crs, user=None,
                       referrer=False):

    geometry, crs = _to_crs(geometry, crs, 'epsg:4326')

    # Uses HTTP request:
    #
    # https://proba-v-mep.esa.int/api/timeseries/apidocs/resource_TimeSeriesService.html

    endpoint = 'https://proba-v-mep.esa.int/api/timeseries/v1.0/ts'

    geometry = shapely.geometry.mapping(geometry)

    # Determine the actual product name

    product_layer = _map_ts_product(product)

    # Perform the HTTP post request

    url = endpoint + '/' + product_layer + '/geometry'

    if product_layer.startswith('S1_GRD_'):
        url += '/multiband'
    elif product_layer.startswith('S2_SCENECLASSIFICATION'):
        url += '/histogram'

    if referrer:
        headers = {'Referer': referrer}
    else:
        headers = {}

    response = requests.post(url=url,
                            json=geometry,
                            params={'startDate': start, 'endDate': end},
                            headers=headers)

    #print(url)
    #print(response.text)

    # Raise an exception if an error occurred

    response.raise_for_status()

    # Otherwise, parse the response body as JSON

    results = response.json()['results']

    # Convert to a timeseries dict

    if product_layer.startswith('S1_GRD_SIGMA0'):
        return {r['date']: {'vh':    r['result'][0]['average'],
                            'vv':    r['result'][1]['average'],
                            'angle': r['result'][2]['average']}
                for r in results}
    elif product_layer.startswith('S1_GRD_GAMMA0'):
            return {r['date']: {'vh': r['result'][0]['average'],
                                'vv': r['result'][1]['average'],
                                'angle': 0.0}
                    for r in results}
    elif product_layer.startswith('S2_SCENECLASSIFICATION'):
        return {r['date']: {s['value']: s['count'] for s in r['result']}
                for r in results}
    else:
        return {r['date']: {'data': r['result']['average']}
                for r in results}


def _get_fis_timeseries(product, start, end, geometry, crs, user=None,
                        referrer=None):

    geometry, crs = _to_crs(geometry, crs, 'epsg:4326')

    # Uses HTTP request:
    #
    # https://www.sentinel-hub.com/develop/documentation/api/fis-request
    #

    endpoint = 'http://services.sentinel-hub.com/ogc/fis'

    if user == 'niab':
        instance_id = '805ba5d1-9ae7-4fdf-9eaf-238b12ca6a26' # cropsar-niab
    elif user == 'geosys':
        instance_id = 'b4e5fb9a-5c9d-48f1-900d-bffb65a5ee84' # cropsar-geosys
    elif user == 'swissre':
        instance_id = 'f1da7e5f-3dc3-4f63-b4e1-257c475e8cce' # cropsar-swissre
    elif user == 'radicle':
        instance_id = 'c9c61677-da1c-4900-8191-1533f248340f' # cropsar-radicle
    else:
        instance_id = '69db9d34-5fc2-4583-adb9-7b1079e0fe67' # cropsar

    # Swap x,y to y,x (lat/long) coordinates

    geometry = shapely.ops.transform(lambda y, x: (x, y), geometry)

    # Determine the actual product name

    product = _map_fis_product(product)

    # Perform the HTTP get request

    url = endpoint + '/' + instance_id

    params = {'SERVICE': 'fis',
              'MAXCC': 100.0, # As set by sentinelhub.FisRequest
              'CRS': 'EPSG:4326',
              'LAYER': product,
              'RESOLUTION': '10m',
              'TIME': '{}/{}'.format(start, end),
              'GEOMETRY': geometry.wkt}

    if product == 'S2_SCENECLASSIFICATION':
        params['RESOLUTION'] = '20m'
        params['BINS'] = 12

    response = requests.get(url=url, params=params)

    # Raise an exception if an error occurred

    response.raise_for_status()

    # Otherwise, parse the response body as JSON

    results = response.json()

    # Check empty

    if not results:
        return {}
    
    # Convert to a simple dict containing only the averages

    if product.startswith('S1_GRD_'):
        return {r0['date']: {'vv': r0['basicStats']['mean'],
                             'vh': r1['basicStats']['mean'],
                             'angle': 0.0}
                for r0, r1 in zip(results['C0'], results['C1'])
                    if r0['date'] == r1['date']}
    elif product == 'S2_SCENECLASSIFICATION':
        return {r['date']: {b['value']: b['count']
                            for b in r['histogram']['bins']}
                for r in results['C0']}
    else:
        return {r['date']: {'data': r['basicStats']['mean']}
                for r in results['C0']}


def _get_timeseries_columns(product):
    if product.startswith('S1_GRD_'):
        return ['vv', 'vh', 'angle']
    elif product.startswith('S2_SCENECLASSIFICATION'):
        return [float(bin) for bin in range(12)]
    else:
        return ['data']


def _to_pandas_timeseries(product, ts):

    # If we got an empty dict, return an empty DataFrame but with
    # the correct columns and dtype set

    if not ts:
        return pd.DataFrame(index=pd.to_datetime([]),
                            columns=_get_timeseries_columns(product),
                            dtype=float)

    # Create a dataframe with a sorted DateTimeIndex

    df = pd.DataFrame.from_dict(ts, orient='index')
    df.index = pd.to_datetime(df.index)
    df.sort_index(inplace=True)

    # Explicitly convert columns to numbers (eg. 'NaN')

    for col in df.columns:
        df[col] = pd.to_numeric(df[col], errors='coerce')

    # For histograms, add missing columns and convert NaN to 0

    if product.startswith('S2_SCENECLASSIFICATION'):
        df = df.reindex(columns=_get_timeseries_columns(product))
        df[df.isna()] = 0

    return df


def retrieve_timeseries(product, start, end, geometry,
                        crs='epsg:4326', source='probav-mep',
                        user=None, referrer=None):

    """Gets a timeseries from the specified source.

    :param product: a Sentinel-2 product name (e.g. 'S2_FAPAR')
    :param start: start date of the generated timeseries (as 'YYYY-mm-dd')
    :param end:   end date of the generated timeseries (as 'YYYY-mm-dd')
    :param geometry: a Shapely geometry object
    :param crs: a pyproj 'projparams' object (e.g. 'epsg:4326')
    :param source: can be any of ['probav-mep', 'sentinel-hub']
    :param user: optional username (e.g. 'cropsar')
    :return: A pandas.DataFrame containing the timeseries
    """

    if source == 'sentinel-hub':
        ts = _get_fis_timeseries(product, start, end, geometry, crs,
                                 user, referrer)
    else:
        ts = _get_ts_timeseries(product, start, end, geometry, crs,
                                user, referrer)

    return _to_pandas_timeseries(product, ts)


def _update_recursive(d, u):
    for k, v in u.items():
        if isinstance(v, collections.abc.Mapping):
            d[k] = _update_recursive(d.get(k, {}), v)
        else:
            d[k] = v

    return d


def retrieve_cropsar_analysis(product, start, end, geometry,
                              crs='epsg:4326', source='probav-mep',
                              params=None, user=None, referrer=None):

    """Collects the required input data and generates a CropSAR timeseries.

    :param product: a Sentinel-2 product name (e.g. 'S2_FAPAR')
    :param start: start date of the generated timeseries (as 'YYYY-mm-dd')
    :param end:   end date of the generated timeseries (as 'YYYY-mm-dd')
    :param geometry: a Shapely geometry object
    :param crs: a pyproj 'projparams' object (e.g. 'epsg:4326')
    :param source: can be any of ['probav-mep', 'sentinel-hub']
    :param params: CropSAR algorithm parameters (see: cropsar._generate.py)
    :param user: optional username (e.g. 'cropsar')
    :return: A dict containing all collected and generated timeseries

    The result is a `dict` of the form::

        {
            'cropsar': pd.DataFrame(..., columns=['q10', 'q50', 'q90']),
            'whittaker': pd.DataFrame(..., columns=['smooth']),
            'clean': {
                's1-data': pd.DataFrame(..., columns=['vv', 'vh', 'angle']),
                's2-data': pd.DataFrame(..., columns=['data', 'clean', 'flag']),
                '.metadata': {
                    'flags': pd.DataFrame(..., columns=['flag', 'name', 'description'])
                }
            },
            'sources': {
                's1-data': pd.DataFrame(..., columns=['vv', 'vh', 'angle']),
                's2-data': pd.DataFrame(..., columns=['data']),
                's2-scenes':  {
                    -10: pd.DataFrame(..., columns=[0.0, 1.0, ..., 11.0]),
                    300: pd.DataFrame(..., columns=[0.0, 1.0, ..., 11.0]),
                    1000: pd.DataFrame(..., columns=[0.0, 1.0, ..., 11.0]),
                },
                '.metadata': {
                    'scene-classes': pd.DataFrame(..., columns=['value', 'description']),
                    's1-product': 'S1_GRD_GAMMA0',
                    's2-product': 'S2_FAPAR'
                }
            },
            '.metdata': {
                'cropsar-version': '1.2.6'
            }
        }
    """

    log.info('Retrieving CropSAR analysis')

    geometry, crs = _to_crs(geometry, crs, 'epsg:4326')

    ext_start, ext_end = prepare_date_range(start, end, params)

    s1_data_geometry = prepare_s1_data_geometry(geometry, params)
    s2_data_geometry = prepare_s2_data_geometry(geometry, params)
    s2_scene_geometries = prepare_s2_scene_geometries(geometry, params)

    log.info('Collecting input timeseries')

    with ThreadPoolExecutor(max_workers=2+len(s2_scene_geometries)) as pool:

        s1_data = pool.submit(retrieve_timeseries,
                              'S1_GRD_GAMMA0',
                              ext_start, ext_end,
                              s1_data_geometry, crs,
                              source, user, referrer)

        s2_data = pool.submit(retrieve_timeseries,
                              product,
                              ext_start, ext_end,
                              s2_data_geometry, crs,
                              source, user, referrer)

        s2_scenes = {b: pool.submit(retrieve_timeseries,
                                    'S2_SCENECLASSIFICATION',
                                    ext_start, ext_end,
                                    s2_scene_geometries[b], crs,
                                    source, user, referrer)
                     for b in s2_scene_geometries}

        s1_data = s1_data.result()
        s2_data = s2_data.result()
        s2_scenes = {b: f.result() for b, f in s2_scenes.items()}

    log.info('Generating CropSAR timeseries')

    result = generate_timeseries(s1_data,
                                 s2_data,
                                 s2_scenes,
                                 product,
                                 start,
                                 end,
                                 params)

    log.info('Constructing analysis')

    cropsar_result = result[0]
    clean_s1_data = result[1]
    clean_s2_data = result[2]
    whittaker_result = result[3]

    flags = describe_clean_s2_flags()
    classes = describe_clean_s2_scene_classes()

    result = {
        'cropsar': cropsar_result,
        'whittaker': whittaker_result,
        'clean': {
            's1-data': clean_s1_data,
            's2-data': clean_s2_data,
            '.metadata': {
                'flags': flags
            },
        },
        'sources': {
            's1-data': s1_data,
            's2-data': s2_data,
            's2-scenes':  {
                b: df for b, df in s2_scenes.items()
            },
            '.metadata': {
                'scene-classes': classes,
                's1-product': 'S1_GRD_GAMMA0',
                's2-product': product
            }
        },
        '.metadata': {
            'cropsar-version': __version__
        }
    }

    log.info('Done')

    return result


def retrieve_cropsar(product, start, end, geometry,
                     crs='epsg:4326', source='probav-mep',
                     params=None, user=None, referrer=None):

    """Collects the required input data and generates a CropSAR timeseries.

    :param product: a Sentinel-2 product name (e.g. 'S2_FAPAR')
    :param start: start date of the generated timeseries (as 'YYYY-mm-dd')
    :param end:   end date of the generated timeseries (as 'YYYY-mm-dd')
    :param geometry: a Shapely geometry object
    :param crs: a pyproj 'projparams' object (e.g. 'epsg:4326')
    :param source: can be any of ['probav-mep', 'sentinel-hub']
    :param params: CropSAR algorithm parameters (see: cropsar._generate.py)
    :param user: optional username (e.g. 'cropsar')
    :return: A pandas.DataFrame containing the generated CropSAR timeseries
    """

    data = retrieve_cropsar_analysis(product,
                                     start,
                                     end,
                                     geometry,
                                     crs,
                                     source,
                                     params,
                                     user,
                                     referrer)

    if data['cropsar'].empty:
        check_model_inputs(data['clean']['s1-data'], 
                           data['clean']['s2-data'],
                           start, end, params,
                           raise_if_fail=True)

    return data['cropsar']
