#
#
#
import pandas
import numpy
import datetime
import time
import threading
import logging
import traceback

import shapely.geometry

from . import smooth
from . import tsservice


#
#    default dataclient for accessing a timeseries service
#
_DATACLIENT = tsservice.dataclient

#
#    field classification values just for future reference, debug, logging,...
#
no_data_registered       = NREG = 0
data_value_registerd     = RVAL = 10
nan_value_registered     = RNAN = 11
dirty_scene_field_center = DSCC = 21
dirty_scene_field_region = DSCE = 22
local_minimum            = LMIN = 51

#
#    number of retries and interval between retries
#
_MAX_RETRIES   = 0 #10
_SLEEP_SECONDS = 600

#
#    new combined call (/geometries/histogram) which should be faster (to be seen)
#
_S2_SC_MULTI_GEOMETRIES_QUERY = True
_SC_USE_BOUNDINGBOX           = False

#
#
#
class S2TimeSeriesBase:
    """
    """

    def __init__(self, szlayername, fieldId, szyyyymmddfirst, szyyyymmddlast, verbose = True):
        """
        """
        self._szlayername                = szlayername
        self._fieldId                    = fieldId
        self._szyyyymmddfirst            = szyyyymmddfirst
        self._szyyyymmddlast             = szyyyymmddlast
        self._verbose                    = verbose

    #
    #
    #    assumes all sub-classes instantiate:
    #    : self._narrowedfielddatatimeseries   : pandas.Series( index=pandas.DatetimeIndex, ...)            - data (fApar)  for the 'narrowed' field geometry
    #    : self._narrowedfieldscenehistogram   : pandas.DataFrame( index=pandas.DatetimeIndex, ...)         - scene classification for the 'narrowed' field geometry
    #    : self._extendedfieldscenehistograms  : list of pandas.DataFrame( index=pandas.DatetimeIndex, ...) - scene classification for the 'extended' field geometries
    #    : self._inarrowfieldbordersinmeter    : integer specifying the 'inward' buffer. 
    #                                            only used for logging in getTimeSeriesData - so could be dummy
    #    : self._lstiextendfieldbordersinmeter : list of integers specifying the 'outward' buffers. 
    #                                            used for logging AND its size in getTimeSeriesData - so could be dummy, but size must match
    #
    def getTimeSeriesData(self, 
                          lstnarrowedfieldvalidscenevalues, inarrowedfieldminpctareavalid, 
                          lstlstextendedfieldvalidscenevalues = None, lstiextendedfieldminpctareavalid = None,
                          localminimamaxdip = None, localminimamaxdif = None, localminimamaxgap = None, localminimamaxpas = 999):
        """
        create cleaned data series according to parameters passed in. data is filtered based on scene classification values on the 'narrowed' and
        'extended' areas of the field.

        :param lstnarrowedfieldvalidscenevalues: list of scene classification values, considered 'valid' in the 'narrowed' field. e.g. [ 4, 5, 6, 7 ]
        :param inarrowedfieldminpctareavalid: minimum percentage of 'valid' scene classification values in the area (pixelcount) of the 'narrowed' field. e.g. 80 
        :param lstlstextendedfieldvalidscenevalues: list of lists of scene classification values, considered 'valid' in the 'extended' field. e.g. [[0, 1, 2, 3, 4, 5, 6, 7, 10, 11], [[0, 1, 2, 3, 4, 5, 6, 8, 9, 11]]
        :param lstiextendedfieldminpctareavalid: minimum percentage of 'valid' scene classification values in the area (pixelcount) of the 'extended' field. e.g. [90, 50] 
        :param localminimamaxdip, localminimamaxdif, localminimamaxgap, localminimamaxpas: parameters used by smooth.flaglocalminima

        lstnarrowedfieldvalidscenevalues and inarrowedfieldminpctareavalid specify the first condition: the field is considered valid, in case
        inarrowedfieldminpctareavalid% of the pixels in the narrowed field have scene class in the list lstnarrowedfieldvalidscenevalues

        the lenght of lstlstextendedfieldvalidscenevalues and lstiextendedfieldminpctareavalid must match. 
        each couple of scenevalues and minpctareavalid form one condition on an extended field

        in case lstiextendfieldbordersinmeter (in constructor) is a singleton (one 'extended' border specified), all conditions are applied to this extension
        in case multiple extended borders are specified, the number of conditions must match the number of extensions, and each condition is applied on its 'own' extension

        in case a minpctareavalid value is negative, the logic is reversed, e.g.:
            minpctareavalid = 85, fieldvalidscenevalues = [4, 5, 6] => field is valid if more than 85% of its pixels have scene class 4, 5 or 6
            minpctareavalid = -5, fieldvalidscenevalues = [3, 8, 9] => field is valid if less than  5% of its pixels have scene class 3, 8 or 9

        """
        #
        #
        #
        if not self._valid:
            logging.error("(%s) - S2TimeSeries.getTimeSeriesData refused: setup failed"%(self._fieldId,))              
            raise Exception("S2TimeSeries.getTimeSeriesData(fieldId: %s) refused: setup failed"%(self._fieldId,))
        #
        #
        #
        if self._verbose: logging.info("(%s) S2TimeSeries.getTimeSeriesData - clean-up - starts"%(self._fieldId,))
        #
        #
        #
        datetime_tick_clean_data  = datetime.datetime.now()
        #
        #    mandatories
        #
        if lstnarrowedfieldvalidscenevalues is None            : lstnarrowedfieldvalidscenevalues = list(range(256)) # better throw ?
        elif isinstance(lstnarrowedfieldvalidscenevalues, int) : lstnarrowedfieldvalidscenevalues = [lstnarrowedfieldvalidscenevalues]
        for narrowedfieldvalidscenevalues in lstnarrowedfieldvalidscenevalues:
            if not isinstance(narrowedfieldvalidscenevalues, int)       : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid entry in list of valid scene values in narrowed field"%(self._fieldId,))

        if inarrowedfieldminpctareavalid is None: inarrowedfieldminpctareavalid = 0  # better throw ?
        if not isinstance(inarrowedfieldminpctareavalid, int)           : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentage parameter for valid scene in narrowed field"%(self._fieldId,))
        if not (0 <= abs(inarrowedfieldminpctareavalid) <= 100)         : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentage value for valid scene in narrowed field"%(self._fieldId,))
        #
        #    valid combinations:
        #        lstiextendedfieldminpctareavalid and lstlstextendedfieldvalidscenevalues both None
        #        lstiextendedfieldminpctareavalid scalar and lstlstextendedfieldvalidscenevalues list of scalar
        #        lstiextendedfieldminpctareavalid list of scalar and lstlstextendedfieldvalidscenevalues list of list of scalar
        #
        #    I hate this
        #
        lstiminpct = None
        lstlstvald = None 
        if lstiextendedfieldminpctareavalid is None:
            if lstlstextendedfieldvalidscenevalues is not None             : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) no minimum percentage of valid scene in extended field specified"%(self._fieldId,))
            lstiminpct = []
            lstlstvald = [[]]

        elif isinstance(lstiextendedfieldminpctareavalid, int):
            if not (0 <= abs(lstiextendedfieldminpctareavalid) <= 100)     : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentage value for valid scenes in extended field"%(self._fieldId,))
            lstiminpct = [lstiextendedfieldminpctareavalid]
            if not isinstance(lstlstextendedfieldvalidscenevalues, list)   : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid list of valid scene values in extended field"%(self._fieldId,))
            lstlstvald = []
            for lstextendedfieldvalidscenevalues in lstlstextendedfieldvalidscenevalues:
                if not isinstance(lstextendedfieldvalidscenevalues, int)   : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid entry in list of valid scene values in extended field"%(self._fieldId,))
                lstlstvald.append(lstextendedfieldvalidscenevalues)
            lstlstvald = [lstlstvald]

        elif isinstance(lstiextendedfieldminpctareavalid, list):
            lstiminpct = []
            for iextendedfieldminpctareavalid in lstiextendedfieldminpctareavalid:
                if not isinstance(iextendedfieldminpctareavalid, int)      : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentage parameter for valid scene in extended field"%(self._fieldId,))
                if not (0 <= abs(iextendedfieldminpctareavalid) <= 100)    : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentage value for valid scene in extended field(s)"%(self._fieldId,))
                lstiminpct.append(iextendedfieldminpctareavalid)
            if not isinstance(lstlstextendedfieldvalidscenevalues, list)   : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid list of lists of valid scene values in extended field(s)"%(self._fieldId,))
            lstlstvald = []
            for lstextendedfieldvalidscenevalues in lstlstextendedfieldvalidscenevalues:          
                if not isinstance(lstextendedfieldvalidscenevalues, list)  : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid list of valid scene values in extended field"%(self._fieldId,))
                lstvald = []
                for extendedfieldvalidscenevalue in lstextendedfieldvalidscenevalues:          
                    if not isinstance(extendedfieldvalidscenevalue, int)   : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid entry in list of valid scene values in extended field"%(self._fieldId,))
                    lstvald.append(extendedfieldvalidscenevalue)
                lstlstvald.append(lstvald)
            if len(lstiminpct) != len(lstlstvald)                          : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) mismatch between percentages list (#values=%s) and lists of valid scene values list (#lists=%s)"%(self._fieldId,len(lstiminpct), len(lstlstvald)))

        else:
            raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) invalid percentages or lists of valid scene values list parameters"%(self._fieldId,))
        #
        #    valid combinations:
        #        no extended field conditions                            => only narrowed field condition checked
        #        single extended field border - any number of conditions => each condition set checked on the single extended border 
        #        n extended field borders     - n conditions             => each extended border has its own condition set
        #
        if 1 != len(self._extendedfieldscenehistograms):
            if len(self._extendedfieldscenehistograms) != len(lstiminpct)  : raise ValueError("S2TimeSeries.getTimeSeriesData(fieldId: %s) mismatch between extended fields (#%s) and conditions specifications (#%s)"%(self._fieldId, len(self._extendedfieldscenehistograms),  len(lstiminpct)))

        #
        #    determine common registrations in each series. actually, they should be identical, however, ...
        #
        commondatetimeindices = self._narrowedfielddatatimeseries.index.intersection(self._narrowedfieldscenehistogram.index)
        for extendedfieldscenehistogram in self._extendedfieldscenehistograms:
            commondatetimeindices = commondatetimeindices.intersection(extendedfieldscenehistogram.index)
        #
        #
        #
        if self._verbose:
            logging.info("(%s) S2TimeSeries.getTimeSeriesData - registered timeslots data narrowed field                          : %s" % (self._fieldId, len(self._narrowedfielddatatimeseries.index),))
            logging.info("(%s) S2TimeSeries.getTimeSeriesData - registered timeslots scene classification narrowed field %-9s: %s" % (self._fieldId, "(%sm)" % self._inarrowfieldbordersinmeter, len(self._narrowedfieldscenehistogram.index)))
            for iIdx in range(len(self._lstiextendfieldbordersinmeter)): 
                logging.info("(%s) S2TimeSeries.getTimeSeriesData - registered timeslots scene classification extended field %-9s: %s" % (self._fieldId, "(%sm)" % self._lstiextendfieldbordersinmeter[iIdx], len(self._extendedfieldscenehistograms[iIdx].index)))
            logging.info("(%s) S2TimeSeries.getTimeSeriesData - common registered timeslots                                       : %s" % (self._fieldId, len(commondatetimeindices),))
        #
        #    reduce data and scene classifications to common timeslots
        #
        narrowedfielddatatimeseries  = self._narrowedfielddatatimeseries.loc[commondatetimeindices]
        narrowedfieldscenehistogram  = self._narrowedfieldscenehistogram.loc[commondatetimeindices, :]
        extendedfieldscenehistograms = []
        for iIdx in range(len(self._lstiextendfieldbordersinmeter)):
            extendedfieldscenehistograms.append(self._extendedfieldscenehistograms[iIdx].loc[commondatetimeindices, :])
        #
        #    keep these original data values for future reference, debug, logging,...
        #    only purpose is actually to find the registered NaN's for the values classification
        #    this can only be done via the 'original' series; in 'narrowed' the original NaN's are removed
        #    hence original NaN's present cannot be separated from missing data
        #
        originalfielddatatimeseries = narrowedfielddatatimeseries.copy()
        #
        #    reduce series further to those registrations containing actual values (not NaN)
        #
        notnulldatadatetimeindices  = narrowedfielddatatimeseries.notnull()
        narrowedfielddatatimeseries = narrowedfielddatatimeseries.loc[notnulldatadatetimeindices]
        narrowedfieldscenehistogram = narrowedfieldscenehistogram.loc[notnulldatadatetimeindices, :]
        for iIdx in range(len(self._lstiextendfieldbordersinmeter)):
            extendedfieldscenehistograms[iIdx] = extendedfieldscenehistograms[iIdx].loc[notnulldatadatetimeindices, :]
        #
        #
        #
        if self._verbose: logging.info("(%s) S2TimeSeries.getTimeSeriesData - usable timeslots                                                  : %s ( common timeslots, data value not NaN )" % (self._fieldId, len(narrowedfielddatatimeseries.index)))

        #
        #    evaluate fraction of the area with valid scene classification
        #
        narrowedfieldscenehistogram.columns = narrowedfieldscenehistogram.columns.map(lambda x: float(x)) # pathetic. sometimes we get int, sometimes float, sometimes strings ...
        lstvalidscenevalues = [float(x) for x in lstnarrowedfieldvalidscenevalues]                        # these are int => make float for intersection
        narrowedfieldscenetotalpixels = narrowedfieldscenehistogram.sum(axis=1)
        narrowedfieldscenevalidpixels = narrowedfieldscenehistogram.loc[:, narrowedfieldscenehistogram.columns.intersection(lstvalidscenevalues)].sum(axis=1)
        if ( inarrowedfieldminpctareavalid >= 0 ) :
            narrowedfieldsceneisvalid = (narrowedfieldscenevalidpixels.div(narrowedfieldscenetotalpixels, axis=0) * 100) >= inarrowedfieldminpctareavalid
        else:
            narrowedfieldsceneisvalid = (narrowedfieldscenevalidpixels.div(narrowedfieldscenetotalpixels, axis=0) * 100) <= -inarrowedfieldminpctareavalid
        #
        #
        #
        lstextendedfieldsceneisvalid = []
        for iIdx in range(len(lstiminpct)):
            if   len(extendedfieldscenehistograms) == 0: extendedfieldscenehistogram = narrowedfieldscenehistogram
            elif len(extendedfieldscenehistograms) == 1: extendedfieldscenehistogram = extendedfieldscenehistograms[0]
            else                                       : extendedfieldscenehistogram = extendedfieldscenehistograms[iIdx]

            extendedfieldscenehistogram.columns = extendedfieldscenehistogram.columns.map(lambda x: float(x)) # again
            lstvalidscenevalues = [float(x) for x in lstlstvald[iIdx]]  
            extendedfieldscenetotalpixels = extendedfieldscenehistogram.sum(axis=1)
            extendedfieldscenevalidpixels = extendedfieldscenehistogram.loc[:, extendedfieldscenehistogram.columns.intersection(lstvalidscenevalues)].sum(axis=1)
            if ( lstiminpct[iIdx] >= 0 ) :
                lstextendedfieldsceneisvalid.append( (extendedfieldscenevalidpixels.div(extendedfieldscenetotalpixels, axis=0) * 100) >= lstiminpct[iIdx] )
            else:
                lstextendedfieldsceneisvalid.append( (extendedfieldscenevalidpixels.div(extendedfieldscenetotalpixels, axis=0) * 100) <= -lstiminpct[iIdx] )
        extendedfieldsceneisvalid = pandas.Series(data=numpy.logical_and.reduce(lstextendedfieldsceneisvalid), index=narrowedfielddatatimeseries.index, dtype=bool)
        #
        #
        #
        if self._verbose:
            logging.info("(%s) S2TimeSeries.getTimeSeriesData - available timeslots in series    : %s" % ( self._fieldId, len(narrowedfielddatatimeseries.index)) )
            if inarrowedfieldminpctareavalid >= 0:
                logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in narrow area   : %s ( more than %3s%% narrowed area has scene values in %s )" % ( self._fieldId, numpy.sum(narrowedfieldsceneisvalid.values), inarrowedfieldminpctareavalid, lstnarrowedfieldvalidscenevalues))
            else:
                logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in narrow area   : %s ( less than %3s%% narrowed area has scene values in %s )" % ( self._fieldId, numpy.sum(narrowedfieldsceneisvalid.values), -inarrowedfieldminpctareavalid, lstnarrowedfieldvalidscenevalues))

            if len(extendedfieldscenehistograms) == 1:
                for iIdx in range(len(lstiminpct)):
                    if lstiminpct[iIdx] >= 0:
                        logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in exteded area  : %s ( more than %3s%% of extended area has scene values in %s )" % ( self._fieldId, numpy.sum(lstextendedfieldsceneisvalid[iIdx].values), lstiminpct[iIdx], lstlstvald[iIdx]))
                    else:
                        logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in exteded area  : %s ( less than %3s%% of extended area has scene values in %s )" % ( self._fieldId, numpy.sum(lstextendedfieldsceneisvalid[iIdx].values), -lstiminpct[iIdx], lstlstvald[iIdx]))
            else:
                for iIdx in range(len(lstiminpct)):
                    if lstiminpct[iIdx] >= 0:
                        logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in exteded area  : %s ( more than %3s%% of %sm extended area has scene values in %s )" % ( self._fieldId, numpy.sum(lstextendedfieldsceneisvalid[iIdx].values), lstiminpct[iIdx], self._lstiextendfieldbordersinmeter[iIdx], lstlstvald[iIdx]))
                    else:
                        logging.info("(%s) S2TimeSeries.getTimeSeriesData - valid timeslots in exteded area  : %s ( less than %3s%% of %sm extended area has scene values in %s )" % ( self._fieldId, numpy.sum(lstextendedfieldsceneisvalid[iIdx].values), -lstiminpct[iIdx], self._lstiextendfieldbordersinmeter[iIdx], lstlstvald[iIdx]))
            logging.info("(%s) S2TimeSeries.getTimeSeriesData - clear sky data values            : %s" % (self._fieldId, numpy.sum((narrowedfieldsceneisvalid & extendedfieldsceneisvalid).values),))

        #
        #    drop timeslots with dirty scene
        #
        cleanfielddatatimeseries = narrowedfielddatatimeseries.loc[narrowedfieldsceneisvalid & extendedfieldsceneisvalid]
        #
        #    field classification series for debug
        #
        fielddataclassification = pandas.Series(data=[RNAN if pandas.isnull(originalfielddatatimeseries.loc[i]) else RVAL for i in originalfielddatatimeseries.index], index=originalfielddatatimeseries.index, dtype=int)
        fielddataclassification.loc[fielddataclassification.index.intersection( extendedfieldsceneisvalid.loc[~extendedfieldsceneisvalid].index)] = DSCE
        fielddataclassification.loc[fielddataclassification.index.intersection( narrowedfieldsceneisvalid.loc[~narrowedfieldsceneisvalid].index)] = DSCC
        #
        #
        #
        if ((localminimamaxdip is None) and (localminimamaxdif is None)) or localminimamaxpas is None:
            if self._verbose: logging.info("(%s) S2TimeSeries.getTimeSeriesData - Smooth: flaglocalminima (maxdip: %s, maxdif: %s, maxgap: %s, maxpasses: %s) - no smoothing executed" % (self._fieldId, localminimamaxdip, localminimamaxdif, localminimamaxgap, localminimamaxpas))
        else:
            if self._verbose: logging.info("(%s) S2TimeSeries.getTimeSeriesData - Smooth: flaglocalminima (maxdip: %s, maxdif: %s, maxgap: %s, maxpasses: %s)" % (self._fieldId, localminimamaxdip, localminimamaxdif, localminimamaxgap, localminimamaxpas))
            cleanfieldfulldatatimeseries = pandas.Series(index=pandas.date_range(start=self._szyyyymmddfirst, end=self._szyyyymmddlast), dtype=float)
            cleanfieldfulldatatimeseries.loc[cleanfielddatatimeseries.index] = cleanfielddatatimeseries
            smooth.flaglocalminima(cleanfieldfulldatatimeseries.values, localminimamaxdip, localminimamaxdif, localminimamaxgap, localminimamaxpas)
            dippedindices = cleanfieldfulldatatimeseries.loc[cleanfieldfulldatatimeseries.isnull()].index.intersection(cleanfielddatatimeseries.loc[cleanfielddatatimeseries.notnull()].index)
            cleanfielddatatimeseries.loc[dippedindices] = numpy.nan
            cleanfielddatatimeseries.dropna(inplace=True)
            fielddataclassification.loc[dippedindices]  = LMIN
        #
        #
        #
        self._originalfielddatatimeseries = originalfielddatatimeseries
        self._cleanfielddatatimeseries    = cleanfielddatatimeseries
        self._fielddataclassification     = fielddataclassification
        #
        #
        #
        resultDict = {
            self._szlayername: {
                'ORIGN': pandas.DataFrame(index=self._originalfielddatatimeseries.index, columns=[self._fieldId], data=self._originalfielddatatimeseries.values),
                'CLEAN': pandas.DataFrame(index=self._cleanfielddatatimeseries.index,    columns=[self._fieldId], data=self._cleanfielddatatimeseries.values),
                'FLAGS': pandas.DataFrame(index=self._fielddataclassification.index,     columns=[self._fieldId], data=self._fielddataclassification.values)
                }
            }
        #
        #
        #
        datetime_tock_clean_data  = datetime.datetime.now()
        self._secondscleandata    = int((datetime_tock_clean_data-datetime_tick_clean_data).total_seconds())
        if self._verbose: logging.info("(%s) S2TimeSeries.getTimeSeriesData - clean-up - done: time (seconds) : %s"%(self._fieldId, self._secondscleandata))
        #
        #
        #
        return resultDict

#
#
#
class S2TimeSeriesFromDataFrames(S2TimeSeriesBase):
    """
    class retrieves the data and scene classifications as a dict containing pandas.Series and pandas.DataFrames.
    method getTimeSeriesData(...) masks the data according to parameters passed in.
    can be used for evaluation and tests, without bothering the webservices
    """

    #
    #
    #
    def __init__(self, szlayername, fieldId, framesdict, szyyyymmddfirst, szyyyymmddlast, inarrowfieldbordersinmeter=None, lstiextendfieldbordersinmeter=None, verbose=True):
        """
        :param framesdict : dictionary containing S2 layers time series data
               dict { 'DATA'     : pandas.Series( index=pandas.DatetimeIndex, ...)
                      'SCENE'    : pandas.DataFrame( index=pandas.DatetimeIndex, ...)
                      'EXSCENES' : [pandas.DataFrame( index=pandas.DatetimeIndex, ...), ...]
                      }
        """
        super().__init__(szlayername, fieldId, szyyyymmddfirst, szyyyymmddlast, verbose)

        self._narrowedfielddatatimeseries   = framesdict['DATA']
        self._narrowedfieldscenehistogram   = framesdict['SCENE']
        self._extendedfieldscenehistograms  = framesdict['EXSCENES']
        #
        #    in this case, _inarrowfieldbordersinmeter and _lstiextendfieldbordersinmeter are used only in logging
        #
        if inarrowfieldbordersinmeter: 
            self._inarrowfieldbordersinmeter = -abs(self._inarrowfieldbordersinmeter) # always inward
        else:
            self._inarrowfieldbordersinmeter    = "In " # dummy

        if lstiextendfieldbordersinmeter:
            self._lstiextendfieldbordersinmeter = lstiextendfieldbordersinmeter # should be [] for NO extended geometries
        else:    
            self._lstiextendfieldbordersinmeter = [ ("Ex(" + str(iIdx+1) + ") ") for iIdx in range(len(framesdict['EXSCENES'])) ] # dummy - count matters

        self._valid = True

#
#
#
class S2TimeSeries(S2TimeSeriesBase):
    """
    class retrieves the data and scene classifications from the time series service.
    method getTimeSeriesData(...) masks the data according to parameters passed in.
    this way multiple parameter combinations can be tested on the same data, without waiting for the web each time.
    :param szlayername : data layer name as specified by the dataclient ('S2_FAPAR', 'S2_NDVI' ...)
    :param fieldId : single fieldId
    :param fieldsgeodataframe : geopandas geodataframe - expected to contain the specified fieldId
    :param szyyyymmddfirst : timeseries start date
    :param szyyyymmddlast : timeseries end date
    :param inarrowfieldbordersinmeter
    :param lstiextendfieldbordersinmeter
    :param imaxretries   (nrt - suggested imaxretries   = 0, training data creation - suggested: imaxretries = 5)
    :param isleepseconds (nrt - suggested isleepseconds = 0, training data creation - suggested: isleepseconds = 120)
    """

    #
    #    thread querying S2_FAPAR time series
    #
    class Thread_S2_LAYER(threading.Thread):
        """
        """

        #
        #
        #
        def __init__(self, layername, fieldId, geodataframe, szyyyymmddfirst, szyyyymmddlast, imaxretries, isleepseconds, dataclient = _DATACLIENT, verbose = True):
            threading.Thread.__init__(self)
            #
            #
            #
            self.layername        = layername
            self.fieldId          = fieldId
            self.geodataframe     = geodataframe
            self.szyyyymmddfirst  = szyyyymmddfirst
            self.szyyyymmddlast   = szyyyymmddlast
            self.imaxretries      = imaxretries
            self.isleepseconds    = isleepseconds
            self.dataclient       = dataclient
            self.verbose          = verbose
            #
            #
            #
            if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_LAYER(%30s) - init" % (self.fieldId, self.layername))

        #
        #
        #
        def run(self):
            #
            #
            #
            attempt = 0
            while True:
                #
                #
                #
                datetime_tick_attempt = datetime.datetime.now()
                #
                #
                #
                attempt += 1
                #
                #
                #
                try:
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_LAYER(%30s) - get_timeseries() attempt(%s) starts" % (self.fieldId, self.layername, attempt))
                    #
                    #
                    #
                    self.data = self.dataclient.get_timeseries(self.geodataframe.loc[self.fieldId], self.layername, self.szyyyymmddfirst, self.szyyyymmddlast)

                except Exception:
                    self.data = None
                    if self.verbose: logging.info (traceback.format_exc())

                #
                #
                #
                datetime_tock_attempt = datetime.datetime.now()
                #
                #
                #
                if (self.data is not None):
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_LAYER(%30s) - get_timeseries() attempt(%s) success (%s seconds)" % (self.fieldId, self.layername, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return 

                #
                #    if data series was not (yet) obtained: consider retry
                #
                if (self.imaxretries <= attempt):
                    logging.error("(%s) S2TimeSeries - Thread_S2_LAYER(%30s) - get_timeseries() attempt(%s) failed (%s seconds) - abend" % (self.fieldId, self.layername, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return

                #
                #    wait a while
                #
                logging.warning("(%s) S2TimeSeries - Thread_S2_LAYER(%30s) - get_timeseries() attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.fieldId, self.layername, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                time.sleep(self.isleepseconds * attempt)

    #
    #    thread querying S2_SCENECLASSIFICATION histogram
    #
    class Thread_S2_SCENECLASSIFICATION(threading.Thread):
        """
        """

        #
        #
        #
        def __init__(self, fieldId, geodataframe, szyyyymmddfirst, szyyyymmddlast, imaxretries, isleepseconds, dataclient = _DATACLIENT, verbose = True):
            threading.Thread.__init__(self)
            #
            #
            #
            self.fieldId          = fieldId
            #
            if _SC_USE_BOUNDINGBOX:
                boundsdataframe   = geodataframe.bounds # has columns minx miny maxx  maxy
                boundsdataseries  = boundsdataframe.apply(lambda row: shapely.geometry.box(row['minx'], row['miny'], row['maxx'], row['maxy']), axis=1)
                self.geodataframe = geodataframe.copy()
                self.geodataframe['geometry'] = boundsdataseries
            else:
                self.geodataframe = geodataframe
            #
            self.szyyyymmddfirst  = szyyyymmddfirst
            self.szyyyymmddlast   = szyyyymmddlast
            self.imaxretries      = imaxretries
            self.isleepseconds    = isleepseconds
            self.dataclient       = dataclient
            self.verbose          = verbose
            #
            #
            #
            if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SCENECLASSIFICATION - init %s" % (self.fieldId, ('' if not _SC_USE_BOUNDINGBOX else '(using bounding box geometry)') ))

        #
        #
        #
        def run(self):
            #
            #
            #
            attempt = 0
            while True:
                #
                #
                #
                datetime_tick_attempt = datetime.datetime.now()
                #
                #
                #
                attempt += 1
                #
                #
                #
                try:
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SCENECLASSIFICATION - get_histogram() attempt(%s) starts" % (self.fieldId, attempt))
                    #
                    #    self.data: 
                    #    type:    <class 'pandas.core.frame.DataFrame'>
                    #    index:   <class 'pandas.core.indexes.datetimes.DatetimeIndex'> e.g. DatetimeIndex(['2016-06-07', '2016-06-10', '2016-06-13', '2016-06-20'], dtype='datetime64[ns]', length=4, freq=None)
                    #    columns: <class 'pandas.core.indexes.numeric.Float64Index'> e.g. Float64Index([0.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0], dtype='float64')
                    #
                    self.data = self.dataclient.get_histogram(self.geodataframe.loc[self.fieldId],  "S2_SCENECLASSIFICATION", self.szyyyymmddfirst, self.szyyyymmddlast)
                except Exception:
                    self.data = None
                    if self.verbose: logging.info (traceback.format_exc())

                #
                #
                #
                datetime_tock_attempt = datetime.datetime.now()
                #
                #
                #
                if (self.data is not None):
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SCENECLASSIFICATION - get_histogram() attempt(%s) success (%s seconds)" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return 

                #
                #    if data series was not (yet) obtained: consider retry
                #
                if (self.imaxretries <= attempt):
                    logging.error("(%s) S2TimeSeries - Thread_S2_SCENECLASSIFICATION - get_histogram() attempt(%s) failed (%s seconds) - abend" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return

                #
                #    wait a while
                #
                logging.warning("(%s) S2TimeSeries - Thread_S2_SCENECLASSIFICATION - get_histogram() attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                time.sleep(self.isleepseconds * attempt)

    #
    #    thread querying S2_SCENECLASSIFICATION histogram
    #
    class Thread_S2_SCENECLASSIFICATION_MULTI_GEOMETRIES(threading.Thread):
        """
        """

        #
        #
        #
        def __init__(self, fieldId, multigeometriesgeodataframe, szyyyymmddfirst, szyyyymmddlast, imaxretries, isleepseconds, dataclient = _DATACLIENT, verbose = True):
            threading.Thread.__init__(self)
            #
            #
            #
            self.fieldId         = fieldId
            #
            if _SC_USE_BOUNDINGBOX:
                boundsdataframe   = multigeometriesgeodataframe.bounds # has columns minx miny maxx  maxy
                boundsdataseries  = boundsdataframe.apply(lambda row: shapely.geometry.box(row['minx'], row['miny'], row['maxx'], row['maxy']), axis=1)
                self.geodataframe = multigeometriesgeodataframe.copy()
                self.geodataframe['geometry'] = boundsdataseries
            else:
                self.geodataframe = multigeometriesgeodataframe
            #
            self.szyyyymmddfirst  = szyyyymmddfirst
            self.szyyyymmddlast   = szyyyymmddlast
            self.imaxretries      = imaxretries
            self.isleepseconds    = isleepseconds
            self.dataclient       = dataclient
            self.verbose          = verbose
            #
            #
            #
            if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SC_MULTI_GEOMETRIES init %s" % (self.fieldId, ('' if not _SC_USE_BOUNDINGBOX else '(using bounding box geometry)') ))

        #
        #
        #
        def run(self):
            #
            #
            #
            attempt = 0
            while True:
                #
                #
                #
                datetime_tick_attempt = datetime.datetime.now()
                #
                #
                #
                attempt += 1
                #
                #
                #
                try:
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SC_MULTI_GEOMETRIES - get_histogram_n_features() attempt(%s) starts" % (self.fieldId, attempt))
                    #
                    #    self.data: dict{ fieldborder : <class 'pandas.core.frame.DataFrame'> }
                    #    index: <class 'pandas.core.indexes.datetimes.DatetimeIndex'> e.g. DatetimeIndex(['2016-06-07', '2016-06-10', '2016-06-13', '2016-06-20'], dtype='datetime64[ns]', length=4, freq=None)
                    #    columns: <class 'pandas.core.indexes.numeric.Float64Index'> e.g. Float64Index([0.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0], dtype='float64')
                    #
                    self.data = self.dataclient.tsservice.get_histogram_n_features(self.geodataframe,  "S2_SCENECLASSIFICATION", self.szyyyymmddfirst, self.szyyyymmddlast)
                except Exception:
                    self.data = None
                    if self.verbose: logging.info (traceback.format_exc())

                #
                #
                #
                datetime_tock_attempt = datetime.datetime.now()
                #
                #
                #
                if (self.data is not None):
                    if self.verbose: logging.info("(%s) S2TimeSeries - Thread_S2_SC_MULTI_GEOMETRIES - get_histogram_n_features() attempt(%s) success (%s seconds)" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return 

                #
                #    if data series was not (yet) obtained: consider retry
                #
                if (self.imaxretries <= attempt):
                    logging.error("(%s) S2TimeSeries - Thread_S2_SC_MULTI_GEOMETRIES - get_histogram_n_features() attempt(%s) failed (%s seconds) - abend" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    return

                #
                #    wait a while
                #
                logging.warning("(%s) S2TimeSeries - Thread_S2_SC_MULTI_GEOMETRIES - get_histogram_n_features() attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                time.sleep(self.isleepseconds * attempt)
    #
    #
    #
    def __init__(self, szlayername, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, inarrowfieldbordersinmeter = 0, lstiextendfieldbordersinmeter = None, imaxretries = _MAX_RETRIES, isleepseconds = _SLEEP_SECONDS, sceneclassificationsbysinglequery = _S2_SC_MULTI_GEOMETRIES_QUERY, dataclient = _DATACLIENT, verbose=True):
        """
        """
        super().__init__(szlayername, fieldId, szyyyymmddfirst, szyyyymmddlast, verbose)
        #
        #
        #
        datetime_tick_setup_data  = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("(%s) S2TimeSeries - setup"%(fieldId,))
        #
        #
        #
        self._imaxretries                = imaxretries
        self._isleepseconds              = isleepseconds
        self._dataclient                 = dataclient
        self._valid                      = False
        #
        #
        #
        try:
            #
            if   inarrowfieldbordersinmeter is None          : self._inarrowfieldbordersinmeter = 0
            elif isinstance(inarrowfieldbordersinmeter, int) : self._inarrowfieldbordersinmeter = inarrowfieldbordersinmeter
            else : raise ValueError("S2TimeSeries(fieldId: %s) invalid inward border value"%(self._fieldId,))
            self._inarrowfieldbordersinmeter = -abs(self._inarrowfieldbordersinmeter) # always inward
            #
            #
            #
            if   lstiextendfieldbordersinmeter is None         : self._lstiextendfieldbordersinmeter = []
            elif isinstance(lstiextendfieldbordersinmeter, int): self._lstiextendfieldbordersinmeter = [lstiextendfieldbordersinmeter]
            else:
                self._lstiextendfieldbordersinmeter = []
                for iextendfieldbordersinmeter in lstiextendfieldbordersinmeter:
                    if not isinstance(iextendfieldbordersinmeter, int)  : raise ValueError("S2TimeSeries(fieldId: %s) invalid entry in list of extended borders"%(self._fieldId,))
                    self._lstiextendfieldbordersinmeter.append(iextendfieldbordersinmeter)
            #
            #    get the specified (single) field as GeoDataFrame from the GeoDataFrame passed in (which could contain multiple fields)
            #
            self._origfieldgeodataframe = fieldsgeodataframe.loc[self._fieldId:self._fieldId,:]
            #
            #    geometry to "UTM 31 N" - actually any meter-based system would do - on crashes: check GDAL environment variables
            #
            geodataframe = self._origfieldgeodataframe.to_crs('epsg:32631')
            #
            #    setup narrowed GeoDataFrame - this one will retrieve the average layer value
            #
            narrowedfieldgeodataframe = geodataframe.copy()                                                                   # copy original field
            narrowedfieldgeodataframe['geometry'] = narrowedfieldgeodataframe.buffer(self._inarrowfieldbordersinmeter)        # calculate buffer (always inward) 
            narrowedfieldgeodataframe = narrowedfieldgeodataframe.to_crs('epsg:4326')                               # back to EPSG:4326 (WGS 84) to access data-client
            narrowedfieldgeodataframe['buffer'] = self._inarrowfieldbordersinmeter                                            # add column with buffers in meter for the everlasting joy of the dataclient
            #
            #    setup extended GeoDataFrames - will be used for the scene classification criteria.
            #
            #    remark: self._lstiextendfieldbordersinmeter can contain duplicates. extendedfieldgeodataframes will only contain unique values
            #
            extendedfieldgeodataframes = dict()
            for fieldborder in self._lstiextendfieldbordersinmeter:
                if fieldborder in extendedfieldgeodataframes : continue
                extendedfieldgeodataframe = geodataframe.copy()                                       # copy original field
                extendedfieldgeodataframe['geometry'] = extendedfieldgeodataframe.buffer(fieldborder) # calculate buffer (negative border is inward, positive border extends, normally positive) 
                extendedfieldgeodataframe = extendedfieldgeodataframe.to_crs('epsg:4326')   # back to EPSG:4326 (WGS 84) to access data-client
                extendedfieldgeodataframe['buffer'] = fieldborder                                     # add column with buffers in meter for the everlasting joy of the dataclient  
                extendedfieldgeodataframes[fieldborder] = extendedfieldgeodataframe
            #
            #    obtain the data for the specified GeoDataFrames
            #    (converted back from GeoDataFrames to GeoDataSeries since GeoDataFrames cannot be handled by data-client)
            #
            datetime_tick_retrieve_data  = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S2TimeSeries - retrieve data (inward buff: %s m, outward buffers: %s m)" % (self._fieldId, self._inarrowfieldbordersinmeter, self._lstiextendfieldbordersinmeter))
            #
            #
            #
            if sceneclassificationsbysinglequery:
                #
                #    using dataclient.get_histogram_n_features - single Thread_S2_SCENECLASSIFICATION_MULTI_GEOMETRIES thread
                #
                multigeometriesdataframeslist = [extendedfieldgeodataframes[fieldborder] for fieldborder in extendedfieldgeodataframes]
                if not self._inarrowfieldbordersinmeter in extendedfieldgeodataframes: 
                    multigeometriesdataframeslist = [narrowedfieldgeodataframe] + multigeometriesdataframeslist
                multigeometriesdataframe = pandas.concat(multigeometriesdataframeslist, ignore_index=True)
                multigeometriesdataframe.set_index('buffer', inplace=True, verify_integrity=True)
                multigeometriesscenehistogramthread = self.Thread_S2_SCENECLASSIFICATION_MULTI_GEOMETRIES(self._fieldId, multigeometriesdataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, dataclient=self._dataclient)
                multigeometriesscenehistogramthread.start()

            else:
                #
                #    using dataclient.get_histogram - multiple Thread_S2_SCENECLASSIFICATION threads
                #
                narrowedfieldscenehistogramthread = self.Thread_S2_SCENECLASSIFICATION(self._fieldId, narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, dataclient=self._dataclient)
                narrowedfieldscenehistogramthread.start()

                extendedfieldscenehistogramthreads = dict()
                for fieldborder in extendedfieldgeodataframes:
                    extendedfieldscenehistogramthread = self.Thread_S2_SCENECLASSIFICATION(self._fieldId, extendedfieldgeodataframes[fieldborder], self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, dataclient=self._dataclient)
                    extendedfieldscenehistogramthread.start()
                    extendedfieldscenehistogramthreads[fieldborder] = extendedfieldscenehistogramthread

            #
            #    S2_FAPAR
            #
            narrowedfielddatatimeseriesthread = self.Thread_S2_LAYER(self._szlayername, self._fieldId, narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, dataclient=self._dataclient)
            narrowedfielddatatimeseriesthread.run() # executed in 'this' thread
            narrowedfielddatatimeseries = narrowedfielddatatimeseriesthread.data
            if narrowedfielddatatimeseries is None:
                raise Exception("S2TimeSeries(fieldId: %s) %s timeseries data for narrowed field could not be retrieved "%(self._fieldId, self._szlayername))
            narrowedfielddatatimeseries.sort_index(inplace=True)

            #
            #    collect and organize results
            #
            extendedfieldscenehistograms = dict()

            if sceneclassificationsbysinglequery:
                multigeometriesscenehistogramthread.join()
                if multigeometriesscenehistogramthread.data is None:
                    raise Exception("S2TimeSeries(fieldId: %s) scene classification histograms for field could not be retrieved"%(self._fieldId,))
                if (self._inarrowfieldbordersinmeter not in multigeometriesscenehistogramthread.data) :
                    raise Exception("S2TimeSeries(fieldId: %s) scene classification histograms for narrowed field not available"%(self._fieldId,))
                narrowedfieldscenehistogram = multigeometriesscenehistogramthread.data[self._inarrowfieldbordersinmeter]
                if narrowedfieldscenehistogram is None :
                    raise Exception("S2TimeSeries(fieldId: %s) scene classification histogram for narrowed field could not be retrieved"%(self._fieldId,))
                narrowedfieldscenehistogram.sort_index(inplace=True)
#                 #
#                 #    debug - checking if the results from two types of queries are consistent
#                 #
#                 print ("Thread_S2_SCENECLASSIFICATION_MULTI_GEOMETRIES fieldborder %s" % (self._inarrowfieldbordersinmeter, ))
#                 print (type(narrowedfieldscenehistogram))
#                 print (type(narrowedfieldscenehistogram.index))
#                 print (narrowedfieldscenehistogram.index)
#                 print (type(narrowedfieldscenehistogram.columns))
#                 print (narrowedfieldscenehistogram.columns)
#                 narrowedfieldscenehistogram.to_csv(r"D:\Tmp\SC\MULTI_GEOM\SC_" + self._fieldId + "_" + str(self._szyyyymmddfirst) + "_" + str(self._szyyyymmddlast) + "_INW_" + str(abs(self._inarrowfieldbordersinmeter)) + ".csv")

                for fieldborder in extendedfieldgeodataframes:
                    if (fieldborder not in multigeometriesscenehistogramthread.data) :
                        raise Exception("S2TimeSeries(fieldId: %s) scene classification histogram for extended field %-9s not available"%(self._fieldId, fieldborder))
                    extendedfieldscenehistograms[fieldborder] = multigeometriesscenehistogramthread.data[fieldborder]
                    if extendedfieldscenehistograms[fieldborder] is None: 
                        raise Exception("S2TimeSeries(fieldId: %s) scene classification histogram for extended field %-9s could not be retrieved"%(self._fieldId, fieldborder))
                    extendedfieldscenehistograms[fieldborder].sort_index(inplace=True)
#                     #
#                     #    debug - checking if the results from two types of queries are consistent
#                     #
#                     print ("Thread_S2_SCENECLASSIFICATION_MULTI_GEOMETRIES fieldborder %s" % (fieldborder, ))
#                     print (type(extendedfieldscenehistograms[fieldborder]))
#                     print (type(extendedfieldscenehistograms[fieldborder].index))
#                     print (extendedfieldscenehistograms[fieldborder].index)
#                     print (type(extendedfieldscenehistograms[fieldborder].columns))
#                     print (extendedfieldscenehistograms[fieldborder].columns)
#                     extendedfieldscenehistograms[fieldborder].to_csv(r"D:\Tmp\SC\MULTI_GEOM\SC_" + self._fieldId + "_" + str(self._szyyyymmddfirst) + "_" + str(self._szyyyymmddlast) + "_OUT_" + str(abs(fieldborder)) + ".csv")

            else:
                narrowedfieldscenehistogramthread.join()
                narrowedfieldscenehistogram = narrowedfieldscenehistogramthread.data
                if narrowedfieldscenehistogram is None: 
                    raise Exception("S2TimeSeries(fieldId: %s) scene classification histogram for narrowed field could not be retrieved "%(self._fieldId,))
                narrowedfieldscenehistogram.sort_index(inplace=True)
#                 #
#                 #    debug - checking if the results from two types of queries are consistent
#                 #
#                 print ("Thread_S2_SCENECLASSIFICATION fieldborder %s" % (self._inarrowfieldbordersinmeter, ))
#                 print (type(narrowedfieldscenehistogram))
#                 print (type(narrowedfieldscenehistogram.index))
#                 print (narrowedfieldscenehistogram.index)
#                 print (type(narrowedfieldscenehistogram.columns))
#                 print (narrowedfieldscenehistogram.columns)
#                 narrowedfieldscenehistogram.to_csv(r"D:\Tmp\SC\SINGLE_GEOM\SC_" + self._fieldId + "_" + str(self._szyyyymmddfirst) + "_" + str(self._szyyyymmddlast) + "_INW_" + str(abs(self._inarrowfieldbordersinmeter)) + ".csv")

                for fieldborder in extendedfieldgeodataframes:
                    extendedfieldscenehistogramthreads[fieldborder].join()
                    extendedfieldscenehistograms[fieldborder] = extendedfieldscenehistogramthreads[fieldborder].data
                    if extendedfieldscenehistograms[fieldborder] is None: 
                        raise Exception("S2TimeSeries(fieldId: %s) scene classification histogram for extended field %-9s could not be retrieved "%(self._fieldId, fieldborder))
                    extendedfieldscenehistograms[fieldborder].sort_index(inplace=True)
#                     #
#                     #    debug - checking if the results from two types of queries are consistent
#                     #
#                     print ("Thread_S2_SCENECLASSIFICATION fieldborder %s" % (fieldborder, ))
#                     print (type(extendedfieldscenehistograms[fieldborder]))
#                     print (type(extendedfieldscenehistograms[fieldborder].index))
#                     print (extendedfieldscenehistograms[fieldborder].index)
#                     print (type(extendedfieldscenehistograms[fieldborder].columns))
#                     print (extendedfieldscenehistograms[fieldborder].columns)
#                     extendedfieldscenehistograms[fieldborder].to_csv(r"D:\Tmp\SC\SINGLE_GEOM\SC_" + self._fieldId + "_" + str(self._szyyyymmddfirst) + "_" + str(self._szyyyymmddlast) + "_OUT_" + str(abs(fieldborder)) + ".csv")

            #
            #
            #
            datetime_tock_retrieve_data  = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S2TimeSeries - field data retrieval time (seconds)                               : %s" % (self._fieldId, int((datetime_tock_retrieve_data-datetime_tick_retrieve_data).total_seconds())))
            #
            #
            #
            self._narrowedfielddatatimeseries  = narrowedfielddatatimeseries
            self._narrowedfieldscenehistogram  = narrowedfieldscenehistogram
            self._extendedfieldscenehistograms = []
            for fieldborder in self._lstiextendfieldbordersinmeter:
                self._extendedfieldscenehistograms.append(extendedfieldscenehistograms[fieldborder])
            #
            #
            #
            self._valid = True
            #
            #
            #
            datetime_tock_setup_data = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S2TimeSeries - field total setup time (seconds)                                  : %s" % (self._fieldId, int((datetime_tock_setup_data-datetime_tick_setup_data).total_seconds())))

        except Exception:
            logging.error (traceback.format_exc())
            logging.error("(%s) S2TimeSeries - setup failed"%(self._fieldId,))



#
#
#
class CleanS2TimeSeriesService:
    """
    class is setup with all 'cleaning parameters'.
    method getTimeSeriesData(...) return the cleaned data for the specified field over the specified interval.
    this way multiple field/timeseries can be retrieved for a given set of 'cleaning parameters'
    """
    #
    #
    #
    def __init__(self,
                 szlayername,
                 inarrowfieldbordersinmeter, lstnarrowedfieldvalidscenevalues, inarrowedfieldminpctareavalid,
                 lstiextendfieldbordersinmeter = None, lstlstextendedfieldvalidscenevalues = None, lstiextendedfieldminpctareavalid = None,
                 localminimamaxdip = None, localminimamaxdif = None, localminimamaxgap = None, localminimamaxpas = 999,
                 imaxretries = 0, isleepseconds = 120, dataclient = _DATACLIENT):
        """
        """
        #
        #
        #
        self._szlayername                         = szlayername

        self._inarrowfieldbordersinmeter          = inarrowfieldbordersinmeter
        self._lstnarrowedfieldvalidscenevalues    = lstnarrowedfieldvalidscenevalues
        self._inarrowedfieldminpctareavalid       = inarrowedfieldminpctareavalid

        self._lstiextendfieldbordersinmeter       = lstiextendfieldbordersinmeter
        self._lstlstextendedfieldvalidscenevalues = lstlstextendedfieldvalidscenevalues
        self._lstiextendedfieldminpctareavalid    = lstiextendedfieldminpctareavalid

        self._localminimamaxdip                   = localminimamaxdip
        self._localminimamaxdif                   = localminimamaxdif
        self._localminimamaxgap                   = localminimamaxgap
        self._localminimamaxpas                   = localminimamaxpas

        self._imaxretries                         = imaxretries
        self._isleepseconds                       = isleepseconds
        self._dataclient                          = dataclient

    #
    #
    #
    def getTimeSeries(self, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, verbose=True):
        """
        debug method returning the actual TimeSeriesData object, so client can has access to 
        detail data (._originalfielddatatimeseries ._cleanfielddatatimeseries and ._fielddataclassification)
        """
        datetime_tick_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("CleanS2TimeSeriesService.getTimeSeries(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s)" % (fieldId, szyyyymmddfirst, szyyyymmddlast))
        #
        #    setup - it will retrieve raw data
        #
        timeseries = S2TimeSeries(self._szlayername, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, self._inarrowfieldbordersinmeter, self._lstiextendfieldbordersinmeter, self._imaxretries, self._isleepseconds, dataclient=self._dataclient, verbose=verbose)
        #
        #    have it calculate clean data
        #
        timeseries.getTimeSeriesData(self._lstnarrowedfieldvalidscenevalues, self._inarrowedfieldminpctareavalid, self._lstlstextendedfieldvalidscenevalues, self._lstiextendedfieldminpctareavalid, self._localminimamaxdip, self._localminimamaxdif, self._localminimamaxgap, self._localminimamaxpas)
        #
        #
        #
        datetime_tock_gettimeseriesdata = datetime.datetime.now()
        if verbose: logging.info("CleanS2TimeSeriesService.getTimeSeries(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s) done  - (%s seconds)" % (fieldId, szyyyymmddfirst, szyyyymmddlast, int((datetime_tock_gettimeseriesdata-datetime_tick_gettimeseriesdata).total_seconds())) )
        #
        #    return object; client can access ._originalfielddatatimeseries ._cleanfielddatatimeseries and ._fielddataclassification
        #
        return timeseries

    #
    #
    #
    def getTimeSeriesData(self, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, verbose=True):
        """
        """
        datetime_tick_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("CleanS2TimeSeriesService.getTimeSeriesData(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s)" % (fieldId, szyyyymmddfirst, szyyyymmddlast))
        #
        #    setup - it will retrieve raw data
        #
        timeseries = S2TimeSeries(self._szlayername, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, self._inarrowfieldbordersinmeter, self._lstiextendfieldbordersinmeter, self._imaxretries, self._isleepseconds, dataclient=self._dataclient, verbose=verbose)
        #
        #    have it calculate clean data
        #
        resultDict = timeseries.getTimeSeriesData(self._lstnarrowedfieldvalidscenevalues, self._inarrowedfieldminpctareavalid, self._lstlstextendedfieldvalidscenevalues, self._lstiextendedfieldminpctareavalid, self._localminimamaxdip, self._localminimamaxdif, self._localminimamaxgap, self._localminimamaxpas)
        #
        #
        #
        datetime_tock_gettimeseriesdata = datetime.datetime.now()
        if verbose: logging.info("CleanS2TimeSeriesService.getTimeSeriesData(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s) done  - (%s seconds)" % (fieldId, szyyyymmddfirst, szyyyymmddlast, int((datetime_tock_gettimeseriesdata-datetime_tick_gettimeseriesdata).total_seconds())) )
        #
        #
        #
        return resultDict

