from pathlib import Path

from openeo.api.process import Parameter

BASEPATH = Path(__file__).parent
RESOURCES = Path(BASEPATH) / 'resources' / 'process_graphs'
MARKDOWN = Path(BASEPATH) / 'resources'

GEOJSON_TYPE = {"type": "object", "subtype": "geojson"}


def polygon_param(
        description="This polygon defines the bounds of the resulting image. Pixels outside the polygon are set to nodata."):
    return Parameter(name="polygon", description=description, schema=GEOJSON_TYPE)


DATE_DESCRIPTION = """Left-closed temporal interval, i.e. an array with exactly two elements:\n\n1. The first element is the start of the temporal interval. The specified instance in time is **included** in the interval.\n2. The second element is the end of the temporal interval. The specified instance in time is **excluded** from the interval.\n\nThe specified temporal strings follow [RFC 3339](https://www.rfc-editor.org/rfc/rfc3339.html). Also supports open intervals by setting one of the boundaries to `null`, but never both."""
DATE_SCHEMA = {
    "type": "array",
    "subtype": "temporal-interval",
    "minItems": 2,
    "maxItems": 2,
    "items": {
        "anyOf": [
            {
                "type": "string",
                "format": "date-time",
                "subtype": "date-time"
            },
            {
                "type": "string",
                "format": "date",
                "subtype": "date"
            },
            {
                "type": "string",
                "subtype": "year",
                "minLength": 4,
                "maxLength": 4,
                "pattern": "^\\d{4}$"
            },
            {
                "type": "null"
            }
        ]
    },
    "examples": [
        [
            "2015-01-01T00:00:00Z",
            "2016-01-01T00:00:00Z"
        ],
        [
            "2015-01-01",
            "2016-01-01"
        ]
    ]
}


def date_parameter(description=DATE_DESCRIPTION):
    return Parameter(
        name="date", description=description,
        schema=DATE_SCHEMA
    )

def biopar_parameter(description = "BIOPAR type [FAPAR,FCOVER] used to calculate the CropSAR curve on. As default the FAPAR is used"):
    return Parameter.string(
    name="biopar_type", description=description, default="FAPAR")

s2_collection_schema = {"type": "string", "subtype": "collection-id"}
s2_collection_schema["enum"] = ["SENTINEL2_L2A_SENTINELHUB", "TERRASCOPE_S2_TOC_V2"]

SENTINEL2_COLLECTION_PARAM = Parameter(
    name="collection_id", description="The openEO collecion_id for (Sentinel-2) input. ",
    default="SENTINEL2_L2A_SENTINELHUB", schema=s2_collection_schema, optional=True
)


def year_parameter(description, default):
    return Parameter.integer(
        name='year',
        description=description,
        default=default
    )


def feature_paranter(description):
    return Parameter(name='features', description=description, schema=GEOJSON_TYPE)
