#!/usr/bin/env python3

import re
import datetime
import logging

import pandas

from ._model import get_model
from ._crs import buffer_geometry

from s2_clean.cleans2timeseries import NREG, RVAL, RNAN, DSCC, DSCE, LMIN
from s2_clean.cleans1timeseries import S1TimeSeriesFromDataSeries
from s2_clean.cleans2timeseries import S2TimeSeriesFromDataFrames

from s2_clean.smooth import WeightValues, \
                            makeweighttypescube, \
                            makesimpleweightscube, \
                            whittaker_second_differences


log = logging.getLogger(__name__)


DEFAULT_PARAMS = {
    's1_clean': {
        'inarrowfieldbordersinmeter': 10
    },
    's2_clean': {
        'inarrowfieldbordersinmeter': 10,
        'lstnarrowedfieldvalidscenevalues': [2, 4, 5, 6, 7],
        'inarrowedfieldminpctareavalid': 85,
        'lstiextendfieldbordersinmeter': [300, 1000, 1000],
        'lstlstextendedfieldvalidscenevalues': [[3, 8, 9, 10], [8, 9], [4, 5, 6]],
        'lstiextendedfieldminpctareavalid': [-4, -3, 40],
        'localminimamaxdip': 0.007,
        'localminimamaxdif': 0.1,
        'localminimamaxgap': 60,
        'localminimamaxpas': 999
    },
    'cropsar': {
        'model': 'RNNfull',
        'simulate_nrt': False,
        'max_s1_gap': '30D'
    },
    'whittaker': {
    }
}


class MaxGapError(Exception):
    pass


def _get_param(params, section, name):

    default = DEFAULT_PARAMS[section][name]

    if params is not None:
        if section in params:
            if name in params[section]:
                return params[section][name]

    return default


def _buffer_geometry_if_non_empty(geometry, crs, dist, **kwargs):

    # If after inward buffering the feature is empty, we ignore the
    # inward buffer and fall back to the original geometry.
    #
    # So this effectively disables inward buffering for small parcels...

    buffered = buffer_geometry(geometry, crs, dist, **kwargs)

    if buffered.is_empty:
        return geometry
    
    return buffered


def _whittaker(series, lmbda=1, passes=3, dokeepmaxima=True,
               aboutequalepsilon=0.02, minimumdatavalue=0.0,
               maximumdatavalue=1.0, weightvalues=None):

    daily_index = pandas.date_range(series.index.min(), series.index.max())
    daily_series = series.reindex(daily_index)

    if weightvalues is None:
        weightvalues = WeightValues(
            maximum    =  1.5,
            minimum    =  0.005,
            posslope   =  0.5,
            negslope   =  0.02,
            aboutequal =  1.0,
            default    =  1.0)

    weighttypes = makeweighttypescube(numpydatacube=daily_series.values,
                                      aboutequalepsilon=aboutequalepsilon)

    weights = makesimpleweightscube(weighttypescube=weighttypes,
                                    weightvalues=weightvalues)

    sm = whittaker_second_differences(lmbda=lmbda,
                                      numpydatacube=daily_series.values,
                                      numpyweightscube=weights,
                                      minimumdatavalue=minimumdatavalue,
                                      maximumdatavalue=maximumdatavalue,
                                      passes=passes,
                                      dokeepmaxima=dokeepmaxima)

    return pandas.Series(data=sm, index=daily_index)


def prepare_s1_data_geometry(geometry, params=None, crs='epsg:4326'):

    border = _get_param(params, 's1_clean', 'inarrowfieldbordersinmeter')

    kwargs = {'cap_style': 1, 'join_style': 2, 'resolution': 4}

    return _buffer_geometry_if_non_empty(geometry, crs, -border, **kwargs)


def prepare_s2_data_geometry(geometry, params=None, crs='epsg:4326'):

    border = _get_param(params, 's2_clean', 'inarrowfieldbordersinmeter')

    kwargs = {'cap_style': 1, 'join_style': 2, 'resolution': 4}

    return _buffer_geometry_if_non_empty(geometry, crs, -border, **kwargs)


def prepare_s2_scene_geometries(geometry, params=None, crs='epsg:4326'):

    border = _get_param(params, 's2_clean', 'inarrowfieldbordersinmeter')
    extend = _get_param(params, 's2_clean', 'lstiextendfieldbordersinmeter')

    kwargs = {'cap_style': 1, 'join_style': 2, 'resolution': 4}

    result = {-border: _buffer_geometry_if_non_empty(geometry,
                                                     crs,
                                                     -border,
                                                     **kwargs)}

    kwargs = {'cap_style': 1, 'join_style': 1, 'resolution': 4}

    geometry = geometry.convex_hull

    for border in set(extend):
        result[border] =  _buffer_geometry_if_non_empty(geometry,
                                                        crs,
                                                        border,
                                                        **kwargs)

    return result


def prepare_date_range(start_date, end_date, params=None):

    modelname = _get_param(params, 'cropsar', 'model')

    model = get_model(modelname)

    margin = datetime.timedelta(days=model.get_margin_in_days())

    start_date = datetime.datetime.strptime(start_date, '%Y-%m-%d')
    start_date = start_date - margin
    start_date = start_date.strftime('%Y-%m-%d')
    
    simulate_nrt = _get_param(params, 'cropsar', 'simulate_nrt')

    if not simulate_nrt:
        end_date = datetime.datetime.strptime(end_date, '%Y-%m-%d')
        end_date = end_date + margin
        end_date = end_date.strftime('%Y-%m-%d')

    return start_date, end_date


def clean_s1(data_df, start_date, end_date,
             params=None, field_id='unidentified-field'):

    data = {
        field_id: {
            'VV': data_df.vv,
            'VH': data_df.vh,
            'IA': data_df.angle
        }
    }

    cleaner = S1TimeSeriesFromDataSeries(field_id, data, start_date, end_date)

    # XXX: this also converts to dB!

    result = cleaner.getTimeSeriesData(imovingaveragewindow=0)

    return pandas.DataFrame({
        'vv': result[field_id]['VV'],
        'vh': result[field_id]['VH'],
        'angle': result[field_id]['IA']
    })


def clean_s2(data_df, scene_dfs_per_border, start_date, end_date,
             params=None, field_id='unidentified-field'):

    in_border = _get_param(params, 's2_clean', 'inarrowfieldbordersinmeter')
    out_borders = _get_param(params, 's2_clean', 'lstiextendfieldbordersinmeter')

    data = {
        'DATA': data_df.iloc[:, 0],
        'SCENE': scene_dfs_per_border[-in_border],
        'EXSCENES': [scene_dfs_per_border[b] for b in out_borders]
    }

    cleaner = S2TimeSeriesFromDataFrames('LAYER', field_id, data, start_date, end_date)

    result = cleaner.getTimeSeriesData(
        _get_param(params, 's2_clean', 'lstnarrowedfieldvalidscenevalues'),
        _get_param(params, 's2_clean', 'inarrowedfieldminpctareavalid'),
        _get_param(params, 's2_clean', 'lstlstextendedfieldvalidscenevalues'),
        _get_param(params, 's2_clean', 'lstiextendedfieldminpctareavalid'),
        _get_param(params, 's2_clean', 'localminimamaxdip'),
        _get_param(params, 's2_clean', 'localminimamaxdif'),
        _get_param(params, 's2_clean', 'localminimamaxgap'),
        _get_param(params, 's2_clean', 'localminimamaxpas')
    )

    return pandas.DataFrame({
        'data': result['LAYER']['ORIGN'][field_id],
        'clean': result['LAYER']['CLEAN'][field_id],
        'flag': result['LAYER']['FLAGS'][field_id]
    })


def evaluate_model(cleaned_s1_data_df, cleaned_s2_data_df, s2_data_layer,
                   start_date, end_date, params=None,
                   field_id='unidentified-field', s1_var='gamma'):

    for df in [cleaned_s1_data_df, cleaned_s2_data_df]:
        if df.dropna().empty:
            return pandas.DataFrame(columns=['q10', 'q50', 'q90'],
                                    index=pandas.to_datetime([]),
                                    dtype=float)

    if s1_var == 'sigma':
        log.warning('Use of Sentinel-1 sigma timeseries is not recommended '
                    'and will trigger an approximate conversion to gamma')

    modelname = _get_param(params, 'cropsar', 'model')

    model = get_model(modelname)

    s2_data_layer = re.sub('^S2_', '', s2_data_layer)

    use_after = not _get_param(params, 'cropsar', 'simulate_nrt')

    q10, q50, q90 = model.get_timeseries(
        cleaned_s1_data_df.vv,
        cleaned_s1_data_df.vh,
        cleaned_s1_data_df.angle,
        cleaned_s2_data_df.clean,
        field_id,
        start_date,
        end_date,
        s2_data_layer,
        s1_var,
        use_after
    )

    return pandas.DataFrame({
        'q10': q10,
        'q50': q50,
        'q90': q90
    })


def smooth_whittaker(cleaned_s2_data_df, start_date, end_date, params):

    if cleaned_s2_data_df.dropna().empty:
        return pandas.DataFrame(columns=['smooth'],
                                index=pandas.to_datetime([]),
                                dtype=float)

    series = _whittaker(cleaned_s2_data_df.clean)
    series = series.loc[start_date : end_date]

    return pandas.DataFrame({'smooth': series})


def describe_clean_s2_flags():
    return pandas.DataFrame.from_records([
            {'flag': NREG, 'name': 'NREG', 'description': 'no_data_registered'},
            {'flag': RVAL, 'name': 'RVAL', 'description': 'data_value_registered'},
            {'flag': RNAN, 'name': 'RNAN', 'description': 'nan_value_registered'},
            {'flag': DSCC, 'name': 'DSCC', 'description': 'dirty_scene_field_center'},
            {'flag': DSCE, 'name': 'DSCE', 'description': 'dirty_scene_field_region'},
            {'flag': LMIN, 'name': 'LMIN', 'description': 'local_minimum'}
        ], index='flag')


def describe_clean_s2_scene_classes():
    return pandas.DataFrame.from_records([
            {'value':  0, 'description': 'no_data'},
            {'value':  1, 'description': 'sat_or_def'},
            {'value':  2, 'description': 'dark_feat/shad'},
            {'value':  3, 'description': 'shadows'},
            {'value':  4, 'description': 'vegetation'},
            {'value':  5, 'description': 'bare_soils'},
            {'value':  6, 'description': 'water'},
            {'value':  7, 'description': 'cloud_low'},
            {'value':  8, 'description': 'cloud_medium'},
            {'value':  9, 'description': 'cloud_high'},
            {'value': 10, 'description': 'cirrus'},
            {'value': 11, 'description': 'snow/ice'}
        ], index='value')


def generate_timeseries(s1_data_df, s2_data_df, s2_scene_dfs_per_border,
                        s2_data_layer, start_date, end_date, params=None,
                        field_id='unidentified-field', s1_var='gamma'):

    # s1_data_df:               pd.DataFrame(..., dtype=float, columns=['vv', 'vh', 'angle'], index=pd.DatetimeIndex(...))
    # s2_data_df:               pd.DataFrame(..., dtype=float, index=pd.DatetimeIndex(...))
    # s2_scene_dfs_per_border:  {
    #                               border: pd.DataFrame(..., dtype=int, columns=[0.0, 1.0, ..., 11.0], index=pd.DatetimeIndex(...)),
    #                               ...
    #                           }
    #                           If none, it is assumed that the S2 data is already cleaned
    # s2_data_layer:            S2_FAPAR
    # start_date:               'yyyy-mm-dd'
    # end_date:                 'yyyy-mm-dd'
    # params:                   None for defaults
    # field_id:                 only used for logging
    # s1_var:                   gamma

    # IMPORTANT:
    #
    # Input DataFrames should be for the interval [start_date-margin, end_date+margin],
    # and should use the appropriate borders
    #
    # See: prepare_date_range()
    #      prepare_s1_data_geometry()
    #      prepare_s2_data_geometry()
    #      prepare_s2_scene_geometries()

    extended_start_date, extended_end_date = prepare_date_range(start_date,
                                                                end_date,
                                                                params)

    cleaned_s1_data_df = clean_s1(s1_data_df,
                                  extended_start_date,
                                  extended_end_date,
                                  params,
                                  field_id)

    max_s1_gap = _get_param(params, 'cropsar', 'max_s1_gap')
    s1_gaps = cleaned_s1_data_df.index.to_series().diff()

    if (s1_gaps > pandas.to_timedelta(max_s1_gap)).any():
        raise MaxGapError('Found a gap of more than {} in the S1 timeseries'.format(max_s1_gap))

    if s2_scene_dfs_per_border is not None:
        cleaned_s2_data_df = clean_s2(s2_data_df,
                                      s2_scene_dfs_per_border,
                                      extended_start_date,
                                      extended_end_date,
                                      params,
                                      field_id)
    else:
        cleaned_s2_data_df = s2_data_df


    cropsar_df = evaluate_model(cleaned_s1_data_df,
                                cleaned_s2_data_df,
                                s2_data_layer,
                                start_date,
                                end_date,
                                params,
                                field_id,
                                s1_var)

    whittaker_df = smooth_whittaker(cleaned_s2_data_df,
                                    extended_start_date,
                                    extended_end_date,
                                    params)

    return cropsar_df, cleaned_s1_data_df, cleaned_s2_data_df, whittaker_df
