######################################################################################################
# biomassproduction.py
# ---------------------------
# Purpose
#       Build OpenEO biomass production service
#
######################################################################################################

#imports
from pathlib import Path

from nextland_services.helpers import load_markdown
from openeo.api.process import Parameter
import shapely
import openeo
from openeo.rest.datacube import DataCube
import numpy as np
import scipy.signal
import logging
import json
import os
from openeo.rest.udp import build_process_dict
from openeo.rest.datacube import PGNode

from nextland_services.constants import *
from openeo.processes import eq


_log = logging.getLogger("biopar")

#Generate results paths
OUTPUT_DIR = Path(__file__).with_suffix('.output')
OUTPUT_DIR.mkdir(parents=True, exist_ok=True)


#Write content to named file
def fwrite(fname, fcontent):
    f = open(fname, 'w')
    f.write(str(fcontent))
    f.close()

#Read Python code for UDF
def load_udf(udf):
    with open(udf, 'r+', encoding="utf8") as f:
        return f.read()

#---- Cropsar cloud masking ----

def get_sentinel_bands(eoconn):

    S2_bands:DataCube = eoconn.load_collection('SENTINEL2_L2A_SENTINELHUB',
                                            bands=["B03", "B04", "B08", "sunAzimuthAngles", "sunZenithAngles",
                                                   "viewAzimuthMean", "viewZenithMean","SCL"])

    S2_bands_mask = S2_bands.process("mask_scl_dilation", data=S2_bands, scl_band_name="SCL")
    return S2_bands_mask


def get_biopar(eoconn,biopar_type):
    S2_bands_mask = get_sentinel_bands(eoconn)
    udf = load_udf(Path(__file__).parent / 'biopar_udf.py')

    biopar_masked = S2_bands_mask.reduce_dimension(dimension="bands", reducer=PGNode(
        process_id="run_udf",
        data={"from_parameter": "data"},
        udf=udf,
        runtime="Python",
        context={"biopar": {"from_parameter": "biopar_type"}}
    ))

    #fapar_masked = S2_bands_mask.reduce_bands_udf(udf)
    biopar_masked = biopar_masked.add_dimension('bands', label=biopar_type, type='bands')

    return biopar_masked


def biopar_build_graph(eoconn, output_file=os.path.join(RESOURCES, 'biopar_graph.json')):
    """
    Creates the biopar UDP
    @param eoconn:
    @param output_file:
    @return:
    """

    #Define service parameters
    date = date_parameter()
    polygon = polygon_param()
    biopar_type = Parameter.string(
        name="biopar_type", description="BIOPAR type [FAPAR,LAI,FCOVER,CCC,CWC]", default="FAPAR"#, values=['FAPAR','LAI','FCOVER','CCC','CWC']
    )

    sb = get_biopar(eoconn,biopar_type)

    biopar_input = sb.filter_temporal(date).mask_polygon(polygon)

    bio_tmp = biopar_input.to_json()
    bio_json = json.loads(bio_tmp)
    bio_pg = bio_json["process_graph"]

    biop_descript = load_markdown('biopar.md')

    biop_dict = build_process_dict(
        process_id="BIOPAR",
        description=biop_descript,
        process_graph=bio_pg,
        #**bio_json,
        parameters=[date,polygon,biopar_type]
    )

    if output_file is not None:
        # Write JSON graph
        fwrite(output_file, json.dumps(biop_dict, indent=4))

    return biop_dict


def test_load_inputs():
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()

    date = "2020-05-06", "2020-05-30"

    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    datacube = get_sentinel_bands(eoconn).filter_temporal(date).filter_bbox(polygon)
    #datacube = get_sentinel_bands(eoconn).filter_temporal(date).mask_polygon(polygon)
    datacube.download(os.path.join(OUTPUT_DIR, 'biopar_result.nc'), format="netCDF")

def test_debug_biopar_udf():
    from openeo_udf.api.datacube import DataCube
    from .biopar_udf import apply_datacube
    import xarray as xr
    ds = xr.load_dataset(os.path.join(OUTPUT_DIR, 'biopar_result.nc')).to_array(dim="bands")
    #res = udf_biopar(DataCube(ds),{})
    res = apply_datacube(DataCube(ds),{'biopar':'FCOVER'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_FCover.nc'))
    res = apply_datacube(DataCube(ds),{'biopar':'CCC'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_CCC.nc'))
    res = apply_datacube(DataCube(ds),{'biopar':'FAPAR'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_FAPAR.nc'))
    res = apply_datacube(DataCube(ds),{'biopar':'LAI'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_LAI.nc'))
    res = apply_datacube(DataCube(ds),{'biopar':'CWC'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_CWC.nc'))
    #Failed test TTT not a biopar -> should default to FAPAR
    res = apply_datacube(DataCube(ds), {'biopar': 'TTT'})
    res.get_array().to_netcdf(os.path.join(OUTPUT_DIR, 'biopar_debug_result_TTT.nc'))

def test_build_biopar_graph():
    logging.basicConfig(level=logging.INFO)
    _log = logging.getLogger("biopar")
    _log.info("Create Graph Biopar Service")
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    biopar_build_graph(eoconn)


def test_biopar_call():
    _log.info("Biopar (UDP call)")

    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()

    _log.info("Downloading BIOPAR (call)")

    date = "2020-05-06", "2020-05-30"
    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='FAPAR')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_FAPAR_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_FAPAR_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_FAPAR_call_result_0001.nc"), format="NetCDF")

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='LAI')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_LAI_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_LAI_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_LAI_call_result_0001.nc"), format="NetCDF")

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='FCOVER')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_FCOVER_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_FCOVER_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_FCOVER_call_result_0001.nc"), format="NetCDF")

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='CCC')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_CCC_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_CCC_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_CCC_call_result_0001.nc"), format="NetCDF")

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='CWC')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_CWC_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_CWC_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_CWC_call_result_0001.nc"), format="NetCDF")

    biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly),biopar_type='TTT')
    fwrite(os.path.join(OUTPUT_DIR, 'biop_TTT_call_0001.json'), biop.to_json())
    biop.download(os.path.join(OUTPUT_DIR,"biop_TTT_call_result_0001.tiff"), format="GTIFF")
    biop.download(os.path.join(OUTPUT_DIR,"biop_TTT_call_result_0001.nc"), format="NetCDF")


def test_use_shape():

    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    date = "2020-05-06", "2020-05-30"

    import geopandas as gpd

    biopars = ['FAPAR','LAI','FCOVER','CCC','CWC']
    shapes = []

    shapes.append(Path(BASEPATH).joinpath('data','parcel-in-retie.shp'))
    shapes.append(Path(BASEPATH).joinpath('data','D0911317_sup_depurada_WGS84.shp'))

    for shp in shapes:
        gpd_shp = gpd.read_file(shp)
        js_shp = json.loads(gpd_shp.to_json())
        shp_nm = Path(shp).stem

        for poly in js_shp['features']:
            polyid = poly['id']
            polyg = poly['geometry']

            for biopar in biopars:
                biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                                    , polygon=poly, biopar_type=biopar)
                fwrite(os.path.join(OUTPUT_DIR, 'biop_CALL_%s_%s_ID-%s_0001.json' % (biopar,shp_nm,polyid)), biop.to_json())
                biop.download(os.path.join(OUTPUT_DIR, 'biop_%s_%s_ID-%s_0001.tiff' % (biopar,shp_nm,polyid)), format="GTIFF")
                biop.download(os.path.join(OUTPUT_DIR, 'biop_%s_%s_ID-%s_0001.nc' % (biopar,shp_nm,polyid)), format="NetCDF")
                biop.download(os.path.join(OUTPUT_DIR, 'biop_%s_%s_ID-%s_0001.json' % (biopar,shp_nm,polyid)), format="JSON")


def test_biopar_timeseries():
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    date = "2020-05-06", "2020-05-30"

    import geopandas as gpd

    biopars = ['FAPAR','LAI','FCOVER','CCC','CWC']
    shapes = []

    shapes.append(Path(BASEPATH).joinpath('data','parcel-in-retie.shp'))
    shapes.append(Path(BASEPATH).joinpath('data','D0911317_sup_depurada_WGS84.shp'))

    for shp in shapes:
        gpd_shp = gpd.read_file(shp)
        js_shp = json.loads(gpd_shp.to_json())
        shp_nm = Path(shp).stem

        for poly in js_shp['features']:
            polyid = poly['id']
            polyg = poly['geometry']

            for biopar in biopars:
                biop = eoconn.datacube_from_process("BIOPAR", namespace="nextland", date=date
                                                    , polygon=poly, biopar_type=biopar).polygonal_mean_timeseries(polyg)
                fwrite(os.path.join(OUTPUT_DIR, 'MTS-biop_CALL_%s_%s_ID-%s_0001.json' % (biopar,shp_nm,polyid)), biop.to_json())
                biop.download(os.path.join(OUTPUT_DIR, 'MTS-biop_%s_%s_ID-%s_0001.json' % (biopar,shp_nm,polyid)), format="JSON")

