#!/usr/bin/env python3

import os
import abc
import logging
import datetime

import requests
import numpy as np
import shapely.geometry
import shapely.geometry.base
import pandas as pd


log = logging.getLogger(__name__)


# Converts an object to GeoJSON

def _as_geojson(obj):

    # If the object has a __geo_interface__, simply return that
    # to get a GeoJSON-like object

    if hasattr(obj, '__geo_interface__'):
        return obj.__geo_interface__

    # A list is considered a special case, try to collect the list
    # items as Features and wrap them in a single FeatureCollection

    if isinstance(obj, list):
        items = []

        for item in obj:
            item = _as_geojson(item)

            if item['type'] == 'FeatureCollection':
                raise ValueError('A nested FeatureCollection is not allowed')

            if item['type'] != 'Feature':
                item = {
                    'type': 'Feature',
                    'properties': {},
                    'geometry': item
                }

            items.append(item)

        return {
            'type': 'FeatureCollection',
            'features': items
        }

    try:

        # Otherwise try to use it as a GeoJSON-like object, and make
        # sure any (nested) geometry is also in the proper format.
        #
        # Test for 'Feature' first, because shapely's asShape() converts
        # it to a 'Polygon' object, thereby loosing all metadata.

        if obj['type'] == 'FeatureCollection':
            obj['features'] = [_as_geojson(f) for f in obj['features']]
        elif obj['type'] == 'Feature':
            obj['geometry'] = _as_geojson(obj['geometry'])
        else:
            obj = shapely.geometry.asShape(obj).__geo_interface__

        return obj

    except KeyError:

        # If it doesn't look like a GeoJSON-like object, it might be a
        # record in the format of a pandas Series for example.  As a last
        # resort try to get a GeoJSON-like geometry from a 'geometry'
        # attribute or dict entry.

        if hasattr(obj, 'geometry'):
            return _as_geojson(obj.geometry)
        else:
            return _as_geojson(obj['geometry'])


# Converts an object to a GeoJSON FeatureCollection

def _as_geojson_features(obj):

    # First convert the object to GeoJSON

    obj = _as_geojson(obj)

    # Then wrap it in a FeatureCollection if necessary

    if obj['type'] != 'FeatureCollection':
        if obj['type'] != 'Feature':
            obj = {
                'type': 'Feature',
                'properties': {},
                'geometry': obj
            }

        obj = {
            'type': 'FeatureCollection',
            'features': [obj]
        }

    return obj


# Structures results depending on the features and layers in a request

def _pack_results(results, request_features, request_layers):

    # If we got just a layer name, remove the layer dict 'level'

    if isinstance(request_layers, str):
        results = [f[request_layers] for f in results]

    # If we got a list of features, return as is

    if isinstance(request_features, list):
        return results

    # If we got a FeatureCollection, return as is

    try:

        if hasattr(request_features, '__geo_interface__'):
            request_features = request_features.__geo_interface__

        if request_features['type'] == 'FeatureCollection':
            return results

    except KeyError:
        pass

    # Otherwise we got a single feature, so remove the feature list 'level'

    return results[0]


# Converts a date to a yyyy-mm-dd string, if necessary

def _as_date_str(date):

    # If it's already a string, make sure it uses '-' separators

    if isinstance(date, str) and '-' not in date:

        # Remove '/' and '.' separators

        date = date.replace('/', '').replace('.', '')

        # Parse date without separators as a datetime object

        date = datetime.datetime.strptime(date, '%Y%m%d')

    # Format a datetime object as yyyy-mm-dd

    if isinstance(date, datetime.datetime):
        date = date.strftime('%Y-%m-%d')

    return date


##############################################################################
# Abstract base class for Timeseries Services

class TSService(object, metaclass=abc.ABCMeta):

    # TODO: write some proper documentation on input/output
    #
    # features: can be a geometry, a list of geometries, a feature, a list of
    #           features, a featurecollection, or any object that has a
    #           __geo_interface__ (geopandas GeoSeries, GeoDataFrame)
    #
    # layers: can be a single layername, or a list of names
    #
    # start_date, end_date: YYYY-mm-dd
    #
    # for one geometry, one layer:  a pandas Series
    # for one geometry, multiple layers: a dict of pandas Series
    # for multiple geometries, one layer: a list of pandas Series
    # for multiple geometries, multiple layers: a list of dicts of pandas Series

    @abc.abstractmethod
    def get_timeseries(self, features, layers, start_date, end_date):
        raise NotImplementedError('Should have implemented this')

    # TODO: write some proper documentation on input/output
    #
    # features: can be a geometry, a list of geometries, a feature, a list of
    #           features, a featurecollection, or any object that has a
    #           __geo_interface__ (geopandas GeoSeries, GeoDataFrame)
    #
    # layers: can be a single layername, or a list of names
    #
    # start_date, end_date: YYYY-mm-dd
    #
    # for one geometry, one layer:  a pandas DataFrame
    # for one geometry, multiple layers: a dict of pandas DataFrames
    # for multiple geometries, one layer: a list of pandas DataFrames
    # for multiple geometries, multiple layers: a list of dicts of pandas DataFrames

    @abc.abstractmethod
    def get_histogram(self, features, layers, start_date, end_date):
        raise NotImplementedError('Should have implemented this')


##############################################################################
# Proba-V MEP Timeseries Service

class TSServiceProbaV(TSService):

    def __init__(self, endpoint=None, verify=True):
        if endpoint is None:
            endpoint = 'https://services.terrascope.be/timeseries/v1.0/ts'

        if endpoint.endswith('/'):
            endpoint = endpoint[:-1]

        self.endpoint = endpoint
        self.verify = verify

    def get_timeseries(self, features, layers, start_date, end_date):

        # TODO: remove/take care of empty features...

        features_arg = features
        layers_arg = layers

        # Make sure we have a list of layer names

        if isinstance(layers, str):
            layers = [layers]

        # Make sure we have a FeatureCollection

        features = _as_geojson_features(features)

        # Make sure we have string dates

        start_date = _as_date_str(start_date)
        end_date   = _as_date_str(end_date)

        # Pre-allocate a list with one dict per feature

        results = [{} for f in features['features']]

        # Loop over all features

        for i in range(len(features['features'])):

            # Extract the feature from the FeatureCollection

            feature = features['features'][i]

            # This dict will serve as a cache for multiband layers

            multiband = {}

            # Loop over all requested layers

            for layer in layers:

                # 'multiband' layers need special handling

                if layer.startswith('S1_GRD_GAMMA0_'):

                    # Cut off the extension (_VH, _VV or _ANGLE), and derive
                    # the base name for multiband layer

                    ext = layer.split('_')[-1]
                    base = layer[:-len(ext)-1]

                    # Check the multiband cache to see whether we have
                    # already fetched this layer before

                    if layer not in multiband:

                        # If not, query the service for the multiband layer
                        # and store all bands as separate layers in the
                        # multiband cache.

                        if base == 'S1_GRD_GAMMA0':
                            result = self._get_ts_s1_grd_merged(feature,
                                                                base,
                                                                start_date,
                                                                end_date)
                        else:
                            result = self._get_ts_multiband(feature,
                                                            base,
                                                            start_date,
                                                            end_date)

                        # Split and store the results individually

                        multiband[base + '_VH']    = result[0]
                        multiband[base + '_VV']    = result[1]
                        multiband[base + '_ANGLE'] = result[2]

                    # Retrieve the layer data from the multiband cache

                    results[i][layer] = multiband[layer]

                else:

                    # Query the TS service using the standard timeseries API

                    result = self._get_ts(feature,
                                          layer,
                                          start_date,
                                          end_date)

                    results[i][layer] = result

        # Finally restructure the list of results depending on the type of
        # arguments passed by the user

        return _pack_results(results, features_arg, layers_arg)

    def get_histogram(self, features, layers, start_date, end_date):

        # TODO: remove/take care of empty features...

        features_arg = features
        layers_arg = layers

        # Make sure we have a list of layer names

        if isinstance(layers, str):
            layers = [layers]

        # Make sure we have a FeatureCollection

        features = _as_geojson_features(features)

        # Make sure we have string dates

        start_date = _as_date_str(start_date)
        end_date   = _as_date_str(end_date)

        # Pre-allocate a list with one dict per feature

        results = [{} for f in features['features']]

        # Loop over all features

        for i in range(len(features['features'])):

            # Extract the feature from the FeatureCollection

            feature = features['features'][i]

            # Loop over all requested layers

            for layer in layers:

                # Query the TS service using the standard histogram API

                result = self._get_hist(feature,
                                        layer,
                                        start_date,
                                        end_date)

                results[i][layer] = result

        # Finally restructure the list of results depending on the type of
        # arguments passed by the user

        return _pack_results(results, features_arg, layers_arg)

    def _get_ts_s1_grd_merged(self, feature, layer, start_date, end_date):

        asc = self._get_ts_multiband(feature,
                                     layer + '_ASCENDING',
                                     start_date,
                                     end_date)

        dsc = self._get_ts_multiband(feature,
                                     layer + '_DESCENDING',
                                     start_date,
                                     end_date)

        vh = pd.concat([asc[0], dsc[0]]).groupby(level=0).mean()
        vv = pd.concat([asc[1], dsc[1]]).groupby(level=0).mean()
        angle = pd.concat([asc[2], dsc[2]]).groupby(level=0).mean()

        return vh, vv, angle

    def _get_ts(self, feature, layer, start_date, end_date):

        # Uses HTTP request:
        #
        #    POST: /v1.0/ts/<layer>/geometry
        #
        #    Params
        #    ------
        #      - startDate: <start_date>
        #      - endDate:   <end_date>
        #
        #    Body
        #    ----
        #      A GeoJSON feature or geometry
        #
        #    Result
        #    ------
        #       Returns a JSON object of the form:
        #
        #       {
        #         'results': [
        #           {
        #             'date': '2017-01-01',
        #             'result': {
        #               'average': 0.0,
        #               'totalCount': 0,
        #               'validCount': 0
        #             }
        #           }, {
        #             'date': '2017-01-02',
        #             'result': {
        #               'average': 0.0,
        #               'totalCount': 0,
        #               'validCount': 0
        #             }
        #           },
        #           ...
        #         ]
        #       }
        #

        # Perform the HTTP post request

        response = requests.post(
            url=self.endpoint + '/' + layer + '/geometry',
            json=feature,
            params={
                'startDate': start_date,
                'endDate': end_date
            },
            headers={
                'content-type': 'application/json'
            },
            verify=self.verify
        )

        # Raise an exception if an error occurred

        response.raise_for_status()

        # Otherwise, parse the response body as JSON

        results = response.json()['results']

        # Extract the dates as a pandas DatetimeIndex

        dates = [r['date'] for r in results]
        dates = pd.to_datetime(dates)

        # Extract the averages as a numpy ndarray

        values = [r['result']['average'] for r in results]
        values = pd.to_numeric(values, errors='coerce')

        # And finally create a new pandas Series

        return pd.Series(values, index=dates)

    def _get_ts_multiband(self, feature, layer, start_date, end_date):

        # Uses HTTP request:
        #
        #    POST: /v1.0/ts/<layer>/geometry/multiband
        #
        #    Params
        #    ------
        #      - startDate: <start_date>
        #      - endDate:   <end_date>
        #
        #    Body
        #    ----
        #      A GeoJSON feature or geometry
        #
        #    Result
        #    ------
        #       Returns a JSON object of the form:
        #
        #       {
        #         'results': [
        #           {
        #             'date': '2017-01-01',
        #             'result': [
        #               {
        #                 'average': 0.0,
        #                 'totalCount': 0,
        #                 'validCount': 0
        #               }, {
        #                 'average': 0.0,
        #                 'totalCount': 0,
        #                 'validCount': 0
        #               },
        #               ...
        #             ]
        #           },{
        #             'date': '2017-01-02',
        #             'result': [
        #               {
        #                 'average': 0.0,
        #                 'totalCount': 0,
        #                 'validCount': 0
        #               }, {
        #                 'average': 0.0,
        #                 'totalCount': 0,
        #                 'validCount': 0
        #               },
        #               ...
        #             ]
        #           },
        #           ...
        #         ]
        #       }
        #

        # Perform the HTTP post request

        response = requests.post(
            url=self.endpoint + '/' + layer + '/geometry/multiband',
            json=feature,
            params={
                'startDate': start_date,
                'endDate': end_date
            },
            headers={
                'content-type': 'application/json'
            },
            verify=self.verify
        )

        # Raise an exception if an error occurred

        response.raise_for_status()

        # Otherwise, parse the response body as JSON

        results = response.json()['results']

        # Extract the dates as a pandas DatetimeIndex

        dates = [r['date'] for r in results]
        dates = pd.to_datetime(dates)

        # Extract the values as a 2D numpy ndarray, with one column per band

        values = [[b['average'] for b in r['result']] for r in results]
        values = np.array(values, dtype=np.float)

        # And finally create a list, with one pandas Series per band/column

        return [pd.Series(col, index=dates) for col in values.T]

    def _get_hist(self, feature, layer, start_date, end_date):

        # Uses HTTP request:
        #
        #    POST: /v1.0/ts/<layer>/geometry/histogram
        #
        #    Params
        #    ------
        #      - startDate: <start_date>
        #      - endDate:   <end_date>
        #
        #    Body
        #    ----
        #       A GeoJSON feature or geometry
        #
        #    Result
        #    ------
        #       Returns a JSON object of the form
        #
        #       {
        #         'results': [
        #           {
        #             'date': '2017-01-01',
        #             'result': [
        #               {'value': 1.0, 'count': 100},
        #               {'value': 4.0, 'count': 200},
        #               {'value': 6.0, 'count': 50}
        #             ]
        #           }, {
        #             'date': '2017-01-01',
        #             'result': [ ... ]
        #           }, {
        #             'date': '2017-01-01',
        #             'result': [ ... ]
        #           }
        #         ]
        #       }

        # Perform the HTTP post request

        response = requests.post(
            url=self.endpoint + '/' + layer + '/geometry/histogram',
            json=feature,
            params={
                'startDate': start_date,
                'endDate': end_date
            },
            headers={
                'content-type': 'application/json'
            },
            verify=self.verify
        )

        # Raise an exception if an error occurred

        response.raise_for_status()

        # Otherwise, parse the response body as JSON

        results = response.json()['results']

        # Build a 'prototype' histogram based on the bins available

        hist = {x['value']: 0 for r in results for x in r['result']}

        bins = sorted(hist.keys())

        # For every date, extract a histogram

        hists = {}

        for r in results:
            date = r['date']

            h = hist.copy()

            for x in r['result']:
                value = x['value']
                count = x['count']

                h[value] = count

            hists[date] = [h[bin] for bin in bins]

        #  And finally convert this to a pandas DataFrame

        df = pd.DataFrame.from_dict(hists, orient='index', columns=bins)
        df.index = pd.to_datetime(df.index)

        return df


##############################################################################
# SentinelHub Timeseries Service

class TSServiceSentinelHub(TSService):
    def __init__(self, instance_id=None):
        if instance_id is None:
            instance_id = '69db9d34-5fc2-4583-adb9-7b1079e0fe67'

        self.instance_id = instance_id

    def get_timeseries(self, features, layers, start_date, end_date):

        # TODO: remove/take care of empty features...

        features_arg = features
        layers_arg = layers

        # Make sure we have a list of layer names

        if isinstance(layers, str):
            layers = [layers]

        # Make sure we have a FeatureCollection

        features = _as_geojson_features(features)

        # Make sure we have string dates

        start_date = _as_date_str(start_date)
        end_date   = _as_date_str(end_date)

        # Pre-allocate a list with one dict per feature

        results = [{} for _ in features['features']]

        # This dict will serve as a cache for multiband layers

        multiband = {}

        # Loop over all requested layers

        for layer in layers:

            # 'multiband' layers need special handling

            if layer.startswith('S1_GRD_GAMMA0_'):

                # Cut off the extension (_VH, _VV or _ANGLE), and derive
                # the base name for multiband layer

                ext = layer.split('_')[-1]
                base = layer[:-len(ext)-1]

                # Check the multiband cache to see whether we have already
                # fetched this layer before

                if layer not in multiband:

                    # If not, query the service for the multiband layer
                    # and store all bands as separate layers in the
                    # multiband cache.  Note that we don't really get an
                    # _ANGLE band, so we just fill it with 0's!
                    #
                    # (As we request already gamma0 from sentinelhub, 0 as
                    # angle is the logical choice, since the conversion
                    # function from sigma to gamma does not change a thing
                    # in that case)

                    result = self._get_ts(features['features'],
                                          base,
                                          start_date,
                                          end_date)

                    multiband[base + '_VV']    = [r[0]     for r in result]
                    multiband[base + '_VH']    = [r[1]     for r in result]
                    multiband[base + '_ANGLE'] = [r[0]*0.0 for r in result]

                # Retrieve the layer data from the multiband cache

                result = multiband[layer]

                for i in range(len(results)):
                    results[i][layer] = result[i]
            else:

                # Query the service using a standard timeseries request

                result = self._get_ts(features['features'],
                                      layer,
                                      start_date,
                                      end_date)

                for i in range(len(results)):
                    results[i][layer] = result[i][0]

        # Finally restructure the list of results depending on the type of
        # arguments passed by the user

        return _pack_results(results, features_arg, layers_arg)

    def get_histogram(self, features, layers, start_date, end_date):

        # TODO: remove/take care of empty features...

        features_arg = features
        layers_arg = layers

        # Make sure we have a list of layer names

        if isinstance(layers, str):
            layers = [layers]

        # Make sure we have a FeatureCollection

        features = _as_geojson_features(features)

        # Make sure we have string dates

        start_date = _as_date_str(start_date)
        end_date   = _as_date_str(end_date)

        # Pre-allocate a list with one dict per feature

        results = [{} for f in features['features']]

        # Loop over all requested layers

        for layer in layers:

            # Retrieve histograms for all features at once

            result = self._get_hist(features['features'],
                                    layer,
                                    start_date,
                                    end_date)

            for i in range(len(results)):
                results[i][layer] = result[i][0]

        # Finally restructure the list of results depending on the type of
        # arguments passed by the user

        return _pack_results(results, features_arg, layers_arg)

    def _get_ts(self, features, layer, start_date, end_date):

        import sentinelhub as shub

        # Construct a list of geometries, one per feature

        geometries = [shub.Geometry(f['geometry'], shub.CRS.WGS84)
                      for f in features]

        # Perform the HTTP request

        request = shub.FisRequest(layer=layer,
                                  geometry_list=geometries,
                                  time=(start_date, end_date),
                                  resolution='10m',
                                  instance_id=self.instance_id)

        data = request.get_data()

        # Finally create a pandas Series per band, per feature

        return [[self._parse_ts(feature_data[channel])
                 for channel in self._get_channels(feature_data)]
                for feature_data in data]

    def _get_hist(self, features, layer, start_date, end_date):

        # For now we only support S2 scene classifications, because
        # interpretation of the histogram data returned requires us to know
        # some details of the 'layer' requested.

        if layer == 'S2_SCENECLASSIFICATION':
            return self._get_sccl_hist(features, layer, start_date, end_date)

        raise ValueError('Unsupported layer for histograms ' + layer)

    def _get_sccl_hist(self, features, layer, start_date, end_date):

        import sentinelhub as shub

        # Construct a list of geometries, one per feature

        geometries = [shub.Geometry(f['geometry'], shub.CRS.WGS84)
                      for f in features]

        # Perform the HTTP request

        request = shub.FisRequest(layer=layer,
                                  geometry_list=geometries,
                                  time=(start_date, end_date),
                                  histogram_type=shub.HistogramType.EQUIDISTANT,
                                  bins='12',
                                  resolution='20m',
                                  instance_id=self.instance_id)

        data = request.get_data()

        # Finally create a pandas DataFrame per band, per feature

        return [[self._parse_hist(feature_data[channel], 12)
                 for channel in self._get_channels(feature_data)]
                for feature_data in data]

    def _get_channels(self, data):
        return ['C' + str(i) for i in range(len(data))]

    def _parse_ts(self, results):

        # https://sentinelhub-py.readthedocs.io/en/latest/examples/fis_request.html
        #
        #   [
        #     {
        #       'C0': [
        #         {
        #           'date': '2019-09-15',
        #           'basicStats': {
        #             'min': 0.532964289188385,
        #             'max': 0.8887381553649902,
        #             'mean': 0.8253281362406859,
        #             'stDev': 0.04244118355920065
        #           }
        #         }, {
        #           'date': '2019-09-13',
        #           'basicStats': {
        #             'min': 0.34871456027030945,
        #             'max': 0.616974949836731,
        #             'mean': 0.5222235424118445,
        #             'stDev': 0.037613876662721236
        #           }
        #         },
        #         ...
        #       ],
        #       'C1': [
        #           ...
        #       ],
        #       ...
        #     }, {
        #       ...
        #     },
        #     ...
        #  ]

        dates = [r['date'] for r in results]
        dates = pd.to_datetime(dates)

        values = [r['basicStats']['mean'] for r in results]
        values = pd.to_numeric(values, errors='coerce')

        return pd.Series(values, index=dates)

    def _parse_hist(self, results, num_bins):

        # https://sentinelhub-py.readthedocs.io/en/latest/examples/fis_request.html
        #
        #   [
        #     {
        #       'C0': [
        #         {
        #           'date': '2019-09-15',
        #           'basicStats': {
        #             'min': 7.0,
        #             'max': 9.0,
        #             'mean': 8.555555555555554,
        #             'stDev': 0.7617394000445611
        #           },
        #           histogram': {
        #             'bins': [
        #               {'lowEdge': 7.0, count: 3.0},
        #               {'lowEdge': 7.166666666666667, count: 0.0},
        #               {'lowEdge': 7.333333333333333, count: 0.0},
        #               {'lowEdge': 7.5, count: 0.0},
        #               {'lowEdge': 7.666666666666667, count: 0.0},
        #               {'lowEdge': 7.833333333333333, count: 0.0},
        #               {'lowEdge': 8.0, count: 2.0},
        #               {'lowEdge': 8.166666666666666, count: 0.0},
        #               {'lowEdge': 8.333333333333334, count: 0.0},
        #               {'lowEdge': 8.5, count: 0.0},
        #               {'lowEdge': 8.666666666666666, count: 0.0},
        #               {'lowEdge': 8.833333333333334, count: 13.0},
        #             ]
        #           }
        #         },
        #         ...
        #       ],
        #       'C1': [
        #           ...
        #       ],
        #       ...
        #     }, {
        #       ...
        #     },
        #     ...
        #  ]

        # For interpretation of the results, see:
        #
        # https://forum.sentinel-hub.com/t/sentinel-2-scl-histogram/1331/4

        hists = {}

        for r in results:
            date = pd.to_datetime(r['date'])

            minval = float(r['basicStats']['min'])
            maxval = float(r['basicStats']['max'])

            hists[date] = {i: 0 for i in range(num_bins)}

            bins = [b for b in r['histogram']['bins'] if int(b['count']) > 0]

            if minval <= maxval:
                equidist = ((maxval - minval) / num_bins)

                values = [round(float(b['lowEdge']) + equidist/2) for b in bins]
                counts = [        int(b['count'])                 for b in bins]

                hists[date].update({v: c for v, c in zip(values, counts)})

        columns = [float(i) for i in range(num_bins)]

        df = pd.DataFrame.from_dict(hists, orient='index', columns=columns)
        df = df.loc[:, (df != 0).any(axis=0)]

        return df

##############################################################################
# Backward compatibility with proba-v MEP dataclient package

class DataClient(object):

    @staticmethod
    def for_probav(endpoint=None, verify=True):
        return DataClient().use_probav(endpoint, verify)

    @staticmethod
    def for_shub(instance_id=None):
        return DataClient().use_shub(instance_id)

    def __init__(self, tss=None):
        self._tss = tss
        self.tsservice = self # for backward compatibility with dataclient

        if self._tss is None:
            self.use_probav()

    def use(self, tss):
        self._tss = tss
        return self

    def use_probav(self, endpoint=None, verify=True):
        return self.use(TSServiceProbaV(endpoint, verify))

    def use_shub(self, instance_id=None):
        return self.use(TSServiceSentinelHub(instance_id))

    def is_using_probav(self):
        return isinstance(self._tss, TSServiceProbaV)

    def is_using_shub(self):
        return isinstance(self._tss, TSServiceSentinelHub)

    # dataclient interface ---------------------------------------------------

    def get_timeseries(self, geometry, layer, start, end, endpoint=None):
        return self._tss.get_timeseries(geometry, layer, start, end)

    def get_histogram(self, geometry, layer, start, end, endpoint=None):
        return self._tss.get_histogram(geometry, layer, start, end)

    def get_timeseries_multiband(self, geometry, layer, start, end, endpoint=None):
        if layer not in ['S1_GRD_GAMMA0']:
            raise ValueError('Unknown multiband layer')

        multiband = [layer + '_VH',
                     layer + '_VV',
                     layer + '_ANGLE']

        ts = self._tss.get_timeseries(geometry, multiband, start, end)

        return {'VH':    ts[layer + '_VH'],
                'VV':    ts[layer + '_VV'],
                'ANGLE': ts[layer + '_ANGLE']}

    def get_timeseries_n_features(self, df, layer, start, end, endpoint=None):
        ts = self._tss.get_timeseries(df, layer, start, end)

        return {id: data for id, data in zip(df.index, ts)}

    def get_histogram_n_features(self, df, layer, start, end, endpoint=None):
        ts = self._tss.get_histogram(df, layer, start, end)

        return {id: data for id, data in zip(df.index, ts)}


##############################################################################
# This global object is used for compatibility with the 'dataclient' package.
#
# By default it uses the Proba-V timeseries service, to switch to the
# Sentinel-Hub service, call:
#
#   tsservice.dataclient.use_shub()
#
# To switch back:
#
#   tsservice.dataclient.use_probav()
#
# If you don't need the backward compatibility, you can of course create a
# local DataClient instance, or use the TSService classes directly.

dataclient = DataClient.for_shub()
#dataclient = DataClient.for_probav()


# Special instance for watch-it-grow TS service, this provides a layer not
# offered by Sentinel-Hub.
#
# TODO: For now this may disable SSL checks.  In some cases we have to set
# the REQUESTS_CA_BUNDLE environment variable to get the Sentinel-Hub code
# to work (e.g. on the Vito network) but this seems to conflict with the
# Proba-V TS service code.

dataclient_for_wig = DataClient.for_probav(
    endpoint='https://services.terrascope.be/timeseries/v1.0/ts',
    verify=False if 'REQUESTS_CA_BUNDLE' in os.environ else True
)

##############################################################################


