#
#
#
import os
import pandas

"""
wic:
- collect queries the timeseries webservices and writes the (raw) results in csv files
- clean performs operations (currently S1 and S2 cropsar cleaning algorithms) on these csv files, and writes the results as csv files.
- config contains the workspace data (directories etc) and parameters for the collect and clean programs.

collect:
collect - config: 
wiccollectconfig['szparcelsshapefile'] : shape file containing the fields - it is assumed this shape file contains a 'fieldID' attribute
wiccollectconfig['szworkspaceroot'] : an existing 'root' directory
wiccollectconfig['szparcelsrawdatadir'] : a name for the workspace subdirectory in which the "raw" csv's will be stored
wiccollectconfig['dcrawdatalayers'] : dictionary specifying the layers (layerId) to be queried:
wiccollectconfig['dcrawdatalayers'][ layerId ]  :  dictionary specifying how to query:
- 'szlayername'  : layer name to be used in webservice call
- 'bdohistogram' : selection between dataclient.get_timeseries (for layers with 'ordinal' values) and dataclient.get_histogram interfaces (for layers with 'categorical' values)
- 'endpoint' : dataclient.get_.. endpoint

collect - params:
wiccollectparams['datetimedatefirst'] and ['datetimedatelast'] start and end dates for the series
wiccollectparams['dcrawdatalayers'] : dictionary containing the layers (layerId) to be queried - layerId's must match those of wiccollectconfig
wiccollectconfig['dcrawdatalayers'][ layerId ]  :  dictionary containing specifying parameters for the query:
- 'bdocollect' : do or don't collect data for this layer
- 'lsibuffersinmeter': buffer(s) (integer - meter) applied on the field geometry negative values: inward, positive values : outward

the wiccollect results are written as csv files is the warkspace:
szworkspaceroot \ szparcelsrawdatadir \ fieldID \ layerId_InXXm.csv, layerId_Exm.csv, layerId_InXXm_HIS.csv or layerId_Exm_HIS.csv (In/Ex: inward/outward buffer, HIS indicates histogram (get_histogram interface) 

clean:
clean - config:
wiccleanconfig['szparcelsresdatadir'] : a name for the workspace subdirectory in which the "resulting" csv's will be stored
wiccleanconfig['dcpseudolayers'] :  dictionary specifying the pseudo layers (cleanded results, processed results, ...)
wiccleanconfig['dcpseudolayers'][ pseudoId ] : dictionary specifying the (general(?)) parameters of the input csv' to use (eg ...._In1000m.csv vs _in20m.csv)

clean - params:
wiccleanparams['dcpseudolayers'] : dictionary containing the pseudo layers (pseudoId) to be created - pseudoId's must match those of wiccollectconfig
wiccleanparams['dcpseudolayers'][ pseudoId ] : dictionary containg specific parameters for the (hardcoded) functions which create the pseudo layer results 
- 'bdoclean'  : do or don't create data for this pseudo layer
- 'szversion' : some indication (used in output filename) to distinguish results created with different parameter setting 
- ... specific parameters for the (hardcoded) functions which create this pseudo layer

the wicclean results are written as csv files is the warkspace:
szworkspaceroot \ szparcelsresdatadir \ fieldID \ XXXXXX_szversion.csv 

"""


#
#    this is the end.
#
_TS_DEFAULT_ENDPOINT = 'https://services.terrascope.be/timeseries/v1.0/ts'
_TS_SMOOTH_ENDPOINT = 'https://services.terrascope.be/timeseries-smooth/ts'


#
#    what to collect
#
wiccollectconfig = {
    
    'szparcelsshapefile'    : r"O:\data\ref\field_selection\test_fields_sample\2016_TEST_sample.shp",
    'szworkspaceroot'       : r"O:\data\ref\field_selection\test_fields_sample\2016_TEST",
#    'szparcelsshapefile'    : r"O:\data\ref\field_selection\test_fields_sample\2017_TEST_sample.shp",
#    'szworkspaceroot'       : r"O:\data\ref\field_selection\test_fields_sample\2017_TEST",
#    'szparcelsshapefile'    : r"O:\data\ref\field_selection\test_fields_sample\2018_TEST_sample.shp",
#    'szworkspaceroot'       : r"O:\data\ref\field_selection\test_fields_sample\2018_TEST",

#    'szparcelsshapefile'    : "D:\\data\\training\\V000\\CroptypesFlemishParcels_2017_30ha.shp",
#    'szworkspaceroot'       : "D:\\Tmp\\wicappdata",

    'szparcelsrawdatadir'   : "raw",
    'dcrawdatalayers'       : {
        "S1_GRD_SIGMA0_ASCENDING_VV"      : {'szlayername':"S1_GRD_SIGMA0_ASCENDING_VV",     'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT }, 
        "S1_GRD_SIGMA0_ASCENDING_VH"      : {'szlayername':"S1_GRD_SIGMA0_ASCENDING_VH",     'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT }, 
        "S1_GRD_SIGMA0_ASCENDING_ANGLE"   : {'szlayername':"S1_GRD_SIGMA0_ASCENDING_ANGLE",  'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT },
        "S1_GRD_SIGMA0_DESCENDING_VV"     : {'szlayername':"S1_GRD_SIGMA0_DESCENDING_VV",    'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT },
        "S1_GRD_SIGMA0_DESCENDING_VH"     : {'szlayername':"S1_GRD_SIGMA0_DESCENDING_VH",    'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT },
        "S1_GRD_SIGMA0_DESCENDING_ANGLE"  : {'szlayername':"S1_GRD_SIGMA0_DESCENDING_ANGLE", 'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT },
        "S2_SCENECLASSIFICATION"          : {'szlayername':"S2_SCENECLASSIFICATION",         'bdohistogram':True,  'endpoint':_TS_DEFAULT_ENDPOINT },
        "S2_NDVI"                         : {'szlayername':"S2_NDVI",                        'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT }, 
        "S2_FAPAR"                        : {'szlayername':"S2_FAPAR",                       'bdohistogram':False, 'endpoint':_TS_DEFAULT_ENDPOINT },
        "S2_FAPAR_WIG"                    : {'szlayername':"S2_FAPAR",                       'bdohistogram':False, 'endpoint':_TS_SMOOTH_ENDPOINT  }
        }
    }

#
#    how to collect
#
_DEFAULT_IN_BUFFER = -10   #    in most cases we require the same inward buffer for each layer
_DEFAULT_COLLECT   = False #    hack to pervent thoughtless overwriting of valuable data - set to True to do "real" collecting.
wiccollectparams = {
    'datetimedatefirst' : pandas.to_datetime("20150601").date(),
    'datetimedatelast'  : pandas.to_datetime("20161231").date(),
#    'datetimedatefirst' : pandas.to_datetime("20160601").date(),
#    'datetimedatelast'  : pandas.to_datetime("20171231").date(),
#     'datetimedatefirst' : pandas.to_datetime("20170601").date(),
#     'datetimedatelast'  : pandas.to_datetime("20181231").date(),

    'dcrawdatalayers'   : {
        "S1_GRD_SIGMA0_ASCENDING_VV"      : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]}, 
        "S1_GRD_SIGMA0_ASCENDING_VH"      : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S1_GRD_SIGMA0_ASCENDING_ANGLE"   : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S1_GRD_SIGMA0_DESCENDING_VV"     : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S1_GRD_SIGMA0_DESCENDING_VH"     : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S1_GRD_SIGMA0_DESCENDING_ANGLE"  : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S2_SCENECLASSIFICATION"          : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER, 300, 1000]},
        "S2_NDVI"                         : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S2_FAPAR"                        : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]},
        "S2_FAPAR_WIG"                    : {'bdocollect':_DEFAULT_COLLECT, 'lsibuffersinmeter':[_DEFAULT_IN_BUFFER]}
        }
    }

#
#    what to clean, what to process,... how in the world could we make this more reusable?
#    config is supposed to describe somehow the inputs of the pseudo-layer-generation
#
wiccleanconfig = {
    'szparcelsresdatadir'   : "tmp_res",
    'dcpseudolayers'    : {
        "S1_CLEAN"      : {'inarrowfieldbordersinmeter':_DEFAULT_IN_BUFFER}, 
        "S2_CLEAN"      : {'inarrowfieldbordersinmeter':_DEFAULT_IN_BUFFER, 'lstiextendfieldbordersinmeter':[300, 1000, 1000]}
        }
    }

#
#    how to clean, how to process,...
#    params is supposed to describe how to process the inputs. for now 1-1 map with s1 and s2 cleaning parameters
#    ellende is dat dit moeilijk generisch te maken is; we hebben specifieke parameters nodig van specifieke funties
#
_SZVERSION = "V004"  #    in most cases the collection of parameters represent the same version

wiccleanparams = {
    'datetimedatefirst' : None, # not used yet
    'datetimedatelast'  : None, # not used yet
    'dcpseudolayers'    : {
        "S1_CLEAN"      : {
            'bdoclean'  :True, 
            'szversion' : _SZVERSION,
            'imovingaveragewindow': 11
            }, 
        "S2_CLEAN"      : { 
            'bdoclean'  :True,
            'szversion' : _SZVERSION,
            'lstnarrowedfieldvalidscenevalues':[ 4, 5, 6, 7 ], 'inarrowedfieldminpctareavalid': 85,
            'lstlstextendedfieldvalidscenevalues':[[3,8,9], [8,9], [7,10]], 'lstiextendedfieldminpctareavalid': [-5, -5, -50],
            'localminimamaxdip' : 0.01, 'localminimamaxdif' : 0.1, 'localminimamaxgap' : 60, 'localminimamaxpas' : 999
            }
        }
    }

#
#
#
class wicworkspace:
    """
    utilities (in a namespace) to control the setup of the workspace
    """

    @staticmethod 
    def workspacePath():
        return wiccollectconfig['szworkspaceroot']
    @staticmethod 
    def workspacePathExists():
        return os.path.exists(wicworkspace.workspacePath())

    @staticmethod 
    def parcelsRawDataPath():
        return os.path.join(wicworkspace.workspacePath(), wiccollectconfig['szparcelsrawdatadir'])
    @staticmethod 
    def parcelsRawDataPathExists():
        return os.path.exists(wicworkspace.parcelsRawDataPath())
    @staticmethod 
    def parcelsRawDataPathCreate():
        os.mkdir(wicworkspace.parcelsRawDataPath())     # yes, we do want you to throw if it exists, and no, we do not want you to create subdirectories behind our backs
        return wicworkspace.parcelsRawDataPathExists()

    @staticmethod 
    def parcelRawDataPath(pidx):
        return os.path.join(wicworkspace.parcelsRawDataPath(), str(pidx))
    @staticmethod 
    def parcelRawDataPathExists(pidx):
        return os.path.exists(wicworkspace.parcelRawDataPath(pidx))
    @staticmethod 
    def parcelRawDataPathCreate(pidx):
        os.mkdir(wicworkspace.parcelRawDataPath(pidx))  # yes, we want you to throw if it exists, and no, we do not want you to create subdirectories behind our backs
        return wicworkspace.parcelRawDataPathExists(pidx)

    @staticmethod 
    def parcelRawDataSimpleTimeseriesCSV(pidx, szproductid, ibuffer):
        return os.path.join(wicworkspace.parcelRawDataPath(pidx), str(szproductid) + ('' if ibuffer == 0 else ('_In' + str(abs(ibuffer)) + 'm' if ibuffer < 0 else '_Ex' + str(ibuffer) + "m")) + ".csv")
    @staticmethod 
    def parcelRawDataSimpleTimeseriesCSVExists(pidx, szproductid, ibuffer):
        return os.path.exists(wicworkspace.parcelRawDataSimpleTimeseriesCSV(pidx, szproductid, ibuffer))
    @staticmethod 
    def parcelRawDataHistoTimeseriesCSV(pidx, szproductid, ibuffer):
        return os.path.join(wicworkspace.parcelRawDataPath(pidx), str(szproductid) + ('' if ibuffer == 0 else ('_In' + str(abs(ibuffer)) + 'm' if ibuffer < 0 else '_Ex' + str(ibuffer) + "m")) + "_HIS.csv")
    @staticmethod 
    def parcelRawDataHistoTimeseriesCSVExists(pidx, szproductid, ibuffer):
        return os.path.exists(wicworkspace.parcelRawDataHistoTimeseriesCSV(pidx, szproductid, ibuffer))


    @staticmethod 
    def parcelsResDataPath():
        return os.path.join(wicworkspace.workspacePath(), wiccleanconfig['szparcelsresdatadir'])
    @staticmethod 
    def parcelsResDataPathExists():
        return os.path.exists(wicworkspace.parcelsResDataPath())
    @staticmethod 
    def parcelsResDataPathCreate():
        os.mkdir(wicworkspace.parcelsResDataPath())     # yes, we do want you to throw if it exists, and no, we do not want you to create subdirectories behind our backs
        return wicworkspace.parcelsResDataPathExists()

    @staticmethod 
    def parcelResDataPath(pidx):
        return os.path.join(wicworkspace.parcelsResDataPath(), str(pidx))
    @staticmethod 
    def parcelResDataPathExists(pidx):
        return os.path.exists(wicworkspace.parcelResDataPath(pidx))
    @staticmethod 
    def parcelResDataPathCreate(pidx):
        os.mkdir(wicworkspace.parcelResDataPath(pidx))  # yes, we want you to throw if it exists, and no, we do not want you to create subdirectories behind our backs
        return wicworkspace.parcelResDataPathExists(pidx)

    #
    #    TODO: how to generalize ?
    #
    @staticmethod 
    def parcelResDataTimeseriesCSV(pidx, szresname, szresparams):
        #sz = os.path.join(wicworkspace.parcelResDataPath(pidx), str(szresname) + '_' + str(szresparams) + ".csv")
        #print(sz)
        return os.path.join(wicworkspace.parcelResDataPath(pidx), str(szresname) + '_' + str(szresparams) + ".csv")

