#
#
#
import pandas
import geopandas
import numpy
import datetime
import time
import threading
import logging
import traceback

from . import smooth
from . import tsservice


#
#    default dataclient for accessing a timeseries service
#
_DATACLIENT = tsservice.dataclient

#
#    number of retries and interval between retries
#
_MAX_RETRIES   = 10
_SLEEP_SECONDS = 600

#
#    dataclient special: S1 VV, VH and IA layers can be retrieved in one query.
#
_S1_MULTI_BAND_QUERY = False

#
#    (only if NOT _S1_MULTI_BAND_QUERY)
#    at this moment is seems that the (single feature) dataclient.get_timeseries interface
#    is faster than the bulk retrieval, via dataclient.get_timeseries_n_features interface,
#    in case the number of features is small. the threshold seem to vary with the weather.
#    this threshold can be specified in the S1TimeSeries constructor, and defaults to the
#    specified _MIN_FEATURES_FOR_BULK_QUERY constant below.
#
#    mind you: one should not use the dataclient.get_timeseries_n_features too often,
#    because it might consume too much resources.
#
_MIN_FEATURES_FOR_BULK_QUERY = 1000

#
#
#
class S1TimeSeriesBase:
    """
    """

    def __init__(self, szyyyymmddfirst, szyyyymmddlast, verbose = True):
        """
        """
        self._szyyyymmddfirst      = szyyyymmddfirst
        self._szyyyymmddlast       = szyyyymmddlast
        self._verbose              = verbose

    #
    #    assumes all sub-classes implement
    #
    def isvalid(self):
        """
        """
        return False

    #
    #    assumes all sub-classes implement
    #
    def getFieldIds(self):
        """
        """
        return None

    #
    #
    #    assumes all sub-classes instantiate:
    #    : self._s1VVSeriesDict      : dict{ fieldId : pandas.Series( index=pandas.DatetimeIndex, ...), ... }     VV
    #    : self._s1VHSeriesDict      : dict{ fieldId : pandas.Series( index=pandas.DatetimeIndex, ...), ... }     VH
    #    : self._s1IASeriesDict      : dict{ fieldId : pandas.Series( index=pandas.DatetimeIndex, ...), ... }     IA
    #
    #
    def getTimeSeriesData(self, imovingaveragewindow = 0):
        """
        """
        #
        #
        #
        if not self.isvalid():
            logging.error("S1TimeSeries.getTimeSeriesData - moving average(%s) - refused: setup failed" % (str(imovingaveragewindow),))             
            raise Exception("S1TimeSeries.getTimeSeriesData - moving average(%s) - refused: setup failed" % (str(imovingaveragewindow),))  
        #
        #
        #
        datetime_tick_clean_data  = datetime.datetime.now()
        #
        #    check params
        #
        if imovingaveragewindow is None : imovingaveragewindow = 0
        if not isinstance(imovingaveragewindow, int) : raise ValueError("(%s) S1TimeSeries.getTimeSeriesData(...) refused: invalid window for moving average - must be integer"  % (self.getFieldIds()[0],))
        if imovingaveragewindow < 0                  : raise ValueError("(%s) S1TimeSeries.getTimeSeriesData(...) refused: invalid window for moving average - must be positive" % (self.getFieldIds()[0],))
        #
        #
        #
        if self._verbose: logging.info("(%s) S1TimeSeries.getTimeSeriesData - moving average(%s) - starts" % (self.getFieldIds()[0], imovingaveragewindow))

        #
        #    dict { fieldId : 
        #           dict { ['VV','VH','IA','RVI'] :
        #             pandas.Series( index=pandas.DatetimeIndex, ...) } } } }
        #
        #    resultDict[fieldId]['VV']   (pandas.Series( index=pandas.DatetimeIndex, ...))
        #    resultDict[fieldId]['VH']   (pandas.Series( index=pandas.DatetimeIndex, ...))
        #    resultDict[fieldId]['IA']   (pandas.Series( index=pandas.DatetimeIndex, ...))
        #    resultDict[fieldId]['RVI']  (pandas.Series( index=pandas.DatetimeIndex, ...))
        #
        resultDict = dict.fromkeys(self.getFieldIds())

        fullindex = pandas.date_range(start=self._szyyyymmddfirst, end=self._szyyyymmddlast)

        def mavg(orgseries):
            fullseries = pandas.Series(index=fullindex, dtype=float)
            fullseries.loc[orgseries.index] = orgseries
            mavgseries = pandas.Series(data=smooth.movingaverage(fullseries, imovingaveragewindow), index=fullindex, dtype=float)
            return mavgseries.loc[orgseries.loc[orgseries.notnull()].index]

        def todb(x):
            return 10.0 * numpy.log10(x)

        for fieldId in self.getFieldIds():
            #
            #
            #
            s1VVsinglefieldSeries  = None
            s1VHsinglefieldSeries  = None
            s1IAsinglefieldSeries  = None
            s1RVIsinglefieldSeries = None

            if not ((self._s1VVSeriesDict[fieldId] is None) or (self._s1VHSeriesDict[fieldId] is None) or (self._s1IASeriesDict[fieldId] is None)) :

                s1VVsinglefieldSeries    = self._s1VVSeriesDict[fieldId].astype(float, copy=True) # depending on the dataclient interface used,
                s1VHsinglefieldSeries    = self._s1VHSeriesDict[fieldId].astype(float, copy=True) # we get floats, or objects, or ...
                s1IAsinglefieldSeries    = self._s1IASeriesDict[fieldId].astype(float, copy=True)
                s1VVsinglefieldSeries.dropna(inplace=True)
                s1VHsinglefieldSeries.dropna(inplace=True)
                s1IAsinglefieldSeries.dropna(inplace=True)
                #
                #    determine common registrations in all  series. actually, they should be identical, however, ...
                #
                s1commondatetimeindices  = s1VVsinglefieldSeries.index.intersection( s1VHsinglefieldSeries.index.intersection( s1IAsinglefieldSeries.index ) )
                s1VVsinglefieldSeries    = pandas.Series(index=s1commondatetimeindices, data=s1VVsinglefieldSeries.loc[s1commondatetimeindices], name = fieldId)
                s1VHsinglefieldSeries    = pandas.Series(index=s1commondatetimeindices, data=s1VHsinglefieldSeries.loc[s1commondatetimeindices], name = fieldId)
                s1IAsinglefieldSeries    = pandas.Series(index=s1commondatetimeindices, data=s1IAsinglefieldSeries.loc[s1commondatetimeindices], name = fieldId)
                if self._verbose: logging.info("(%s) S1TimeSeries.getTimeSeriesData - common registered timeslots data                         : %s  (field: %s) " % (self.getFieldIds()[0], len(s1commondatetimeindices), fieldId))
                #
                #    calculate Radar Vegetation Index - BEFORE converting to decibel - TODO: can we have math problems here?
                #
                s1RVIsinglefieldSeries  = 4.0 * s1VHsinglefieldSeries  / (s1VVsinglefieldSeries  + s1VHsinglefieldSeries)
                #
                #    apply simple moving average smoothing on VV, VH & RVI's if requested
                #
                if 0 < imovingaveragewindow:
                    s1VVsinglefieldSeries   = mavg(s1VVsinglefieldSeries)
                    s1VHsinglefieldSeries   = mavg(s1VHsinglefieldSeries)
                    s1RVIsinglefieldSeries  = mavg(s1RVIsinglefieldSeries)
                #
                #    convert VV & VH's to decibel
                #
                s1VVsinglefieldSeries  = s1VVsinglefieldSeries.apply(todb)
                s1VHsinglefieldSeries  = s1VHsinglefieldSeries.apply(todb)

            else:
                if self._verbose: logging.info("(%s) S1TimeSeries.getTimeSeriesData - field: %s no S1 data" % (self.getFieldIds()[0], fieldId))

            #
            #    store result as dictionary of series (or None)
            #
            resultDict[fieldId] = {
                    'VV' : s1VVsinglefieldSeries,
                    'VH' : s1VHsinglefieldSeries,
                    'IA' : s1IAsinglefieldSeries,
                    'RVI': s1RVIsinglefieldSeries
                }
        #
        #
        #
        datetime_tock_clean_data  = datetime.datetime.now()
        self._secondscleandata    = int((datetime_tock_clean_data-datetime_tick_clean_data).total_seconds())
        if self._verbose: logging.info("(%s) S1TimeSeries.getTimeSeriesData - moving average(%s) - done: time (seconds) : %s" % (self.getFieldIds()[0], imovingaveragewindow, self._secondscleandata) )
        #
        #
        #
        return resultDict

#
#
#
class S1TimeSeriesFromDataSeries(S1TimeSeriesBase):
    """
    class retrieves the data and scene classifications as a dict containing pandas.Series.
    method getTimeSeriesData(...) masks the data according to parameters passed in.
    can be used for evaluation and tests, without bothering the webservices
    """

    def __init__(self, fieldIds, seriesdict, szyyyymmddfirst, szyyyymmddlast, verbose = True):
        """
        :param seriesdict : dictionary containing S1 layers time series data  
            dict { fieldId : 
                dict { ['VV','VH','IA'] :
                      pandas.Series( index=pandas.DatetimeIndex, ...) } } } }
        """
        super().__init__(szyyyymmddfirst, szyyyymmddlast, verbose)

        self._fieldIds = None
        self._isvalid  = False

        lst_fieldIds = []
        if fieldIds is None:
            try:                        lst_fieldIds.extend(seriesdict.keys())        #
            except:                     lst_fieldIds = []
        elif isinstance(fieldIds, str): lst_fieldIds = [fieldIds]                     # single fieldId specified as string
        else:
            try:                        lst_fieldIds.extend(fieldIds)                 # try fieldIds specified as some iterable
            except TypeError:           lst_fieldIds = [fieldIds]                     # else whatever it is, consider it as a single (non-string) fieldId

        if len(lst_fieldIds) == 0 : raise ValueError("S1TimeSeriesFromSeries(...) no valid fieldIds specified or found")    


        s1VVSeriesDict  = dict.fromkeys(lst_fieldIds)
        s1VHSeriesDict  = dict.fromkeys(lst_fieldIds)
        s1IASeriesDict  = dict.fromkeys(lst_fieldIds)

        for fieldId in lst_fieldIds:
            s1VVSeriesDict[fieldId]  = seriesdict[fieldId]['VV']
            s1VHSeriesDict[fieldId]  = seriesdict[fieldId]['VH']
            s1IASeriesDict[fieldId]  = seriesdict[fieldId]['IA']

        #
        #
        #
        self._s1VVSeriesDict  = s1VVSeriesDict
        self._s1VHSeriesDict  = s1VHSeriesDict
        self._s1IASeriesDict  = s1IASeriesDict

        self._fieldIds = lst_fieldIds
        self._isvalid  = True

    def isvalid(self):
        """
        """
        return self._isvalid

    def getFieldIds(self):
        """
        """
        return self._fieldIds

#
#
#
class S1TimeSeries(S1TimeSeriesBase):
    """
    retrieve time series data for the S1 layers (VV,VH and IA) from the time series service.
    :param fieldIds : single fieldId or list of fieldIds 
    :param fieldsgeodataframe : geopandas GeoDataFrame - expected to contain the specified fieldIds
    :param szyyyymmddfirst : timeseries start date
    :param szyyyymmddlast : timeseries end date
    :param inarrowfieldbordersinmeter
    :param imaxretries   (nrt - suggested imaxretries   = 0, training data creation - suggested: imaxretries = 10)
    :param isleepseconds (nrt - suggested isleepseconds = 0, training data creation - suggested: isleepseconds = 600)
    :param iminfieldsforbulkquery : threshold (nr of fields) to select between (loop of) get_timeseries and get_timeseries_n_features queries
    """

    #
    #    thread querying time series of specified layer
    #
    class Thread_S1_LAYER(threading.Thread):
        """
        queries the specified layer for all fields in the specified geodataframe.
        :param layername : layername, as used by the dataclient. list of available layers can be retrieved via https://proba-v-mep.esa.int/api/timeseries/v1.0/ts
        :param geodataframe : geopandas GeoDataFrame - the specified layer will be queried for each GeoSeries in the GeoDataFrame
        :param szyyyymmddfirst : timeseries start date
        :param szyyyymmddlast : timeseries end date
        :param imaxretries   (nrt - suggested imaxretries   = 0, training data creation - suggested: imaxretries = 10)
        :param isleepseconds (nrt - suggested isleepseconds = 0, training data creation - suggested: isleepseconds = 600)
        :param dobulkquery : selects between (loop of) get_timeseries and get_timeseries_n_features queries
        """

        #
        #
        #
        def __init__(self, layername, geodataframe, szyyyymmddfirst, szyyyymmddlast, imaxretries, isleepseconds, dataclient, dobulkquery = False, verbose = True):
            threading.Thread.__init__(self)
            #
            #
            #
            self.layername            = layername
            self.geodataframe         = geodataframe
            self.szyyyymmddfirst      = szyyyymmddfirst
            self.szyyyymmddlast       = szyyyymmddlast
            self.imaxretries          = imaxretries
            self.isleepseconds        = isleepseconds
            self.dataclient           = dataclient
            self.dobulkquery          = dobulkquery
            self.verbose              = verbose
            self.fieldsSeriesDict     = dict.fromkeys(geodataframe.index)
            #
            #
            #
            if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - init: %s entries in geodataframe" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index)))

        #
        #
        #
        def run(self):

            datetime_tick_run = datetime.datetime.now()

            try:

                if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - %s features(%s ... %s) - starts" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1]))

                if self.dobulkquery:
                    self._run_get_timeseries_bulk()
                else:
                    self._run_get_timeseries_loop()

                if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - %s features(%s ... %s) - ended (%s seconds)" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1], int((datetime.datetime.now()-datetime_tick_run).total_seconds())))

            except Exception:

                logging.error (traceback.format_exc())
                logging.error("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - %s features(%s ... %s) - failed (%s seconds)" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1], int((datetime.datetime.now()-datetime_tick_run).total_seconds())))

        #
        #
        #
        def _run_get_timeseries_loop(self):
            """
            dataclient.get_timeseries is called (in a loop) for each index in the original geodataframe passed in.
            dataclient.get_timeseries returns a pandas.Series
            - index  = DatetimeIndex containing (only) the registrations
            - name   = fieldId (from the 'feature' ?
            - data   = averages (can contain nan's)
            """

            for fieldId in self.fieldsSeriesDict:

                attempt = 0
                while True:

                    attempt += 1

                    datetime_tick_attempt = datetime.datetime.now()

                    try:

                        if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries(%s) attempt(%s) starts" % (self.geodataframe.index[0], self.layername, fieldId, attempt))
                        dataseries = self.dataclient.get_timeseries(self.geodataframe.loc[fieldId], self.layername, self.szyyyymmddfirst, self.szyyyymmddlast)

                    except Exception:
                        dataseries = None
                        if self.verbose: logging.info (traceback.format_exc())

                    datetime_tock_attempt = datetime.datetime.now()

                    if (dataseries is not None):
                        if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries(%s) attempt(%s) success (%s seconds)" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                        self.fieldsSeriesDict[fieldId] = dataseries
                        break

                    if (self.imaxretries <= attempt):
                        logging.error("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries(%s) attempt(%s) failed (%s seconds) - abend" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                        self.fieldsSeriesDict[fieldId] = None
                        break

                    logging.warning("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries(%s) attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                    time.sleep(self.isleepseconds * attempt)
 
        #
        #
        #
        def _run_get_timeseries_bulk(self):
            """
            dataclient.get_timeseries_n_features is called with the complete geodataframe passed in.
            dataclient.get_timeseries_n_features returns ...
            """

            attempt = 0
            while True:

                attempt += 1

                datetime_tick_attempt = datetime.datetime.now()

                try:

                    if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries_n_features(%s ... %s) attempt(%s) starts" % (self.geodataframe.index[0], self.layername, self.geodataframe.index[0], self.geodataframe.index[-1], attempt))
                    dataseries = self.dataclient.get_timeseries_n_features(self.geodataframe, self.layername, self.szyyyymmddfirst, self.szyyyymmddlast)

                except Exception:
                    dataseries = None
                    if self.verbose: logging.info (traceback.format_exc())

                datetime_tock_attempt = datetime.datetime.now()

                if (dataseries is not None):
                    if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries_n_features(%s ... %s) attempt(%s) success (%s seconds)" % (self.geodataframe.index[0], self.layername, self.geodataframe.index[0], self.geodataframe.index[-1], attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    for fieldId in dataseries:
                        self.fieldsSeriesDict[fieldId] = dataseries[fieldId]
                    break

                if (self.imaxretries <= attempt):
                    logging.error("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries_n_features(%s ... %s) attempt(%s) failed (%s seconds) - abend" % (self.geodataframe.index[0], self.layername, self.geodataframe.index[0], self.geodataframe.index[-1], attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                    for fieldId in self.fieldsSeriesDict:
                        self.fieldsSeriesDict[fieldId] = None
                    break

                logging.warning("(%s) S1TimeSeries - Thread_S1_LAYER(%30s) - run - get_timeseries_n_features(%s ... %s) attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.geodataframe.index[0], self.layername, self.geodataframe.index[0], self.geodataframe.index[-1], attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                time.sleep(self.isleepseconds * attempt)

    #
    #    thread querying time series of specified layer
    #
    class Thread_S1_MULTI_BAND_LAYER(threading.Thread):
        """
        queries the specified multiband layer for all fields in the specified geodataframe.
        :param layername : layername, as used by the dataclient. list of available multiband layers can be retrieved via ?
        :param geodataframe : geopandas GeoDataFrame - the specified layer will be queried for each GeoSeries in the GeoDataFrame
        :param szyyyymmddfirst : timeseries start date
        :param szyyyymmddlast : timeseries end date
        :param imaxretries   (nrt - suggested imaxretries   = 0, training data creation - suggested: imaxretries = 10)
        :param isleepseconds (nrt - suggested isleepseconds = 0, training data creation - suggested: isleepseconds = 600)
        :param dobulkquery : selects between (loop of) get_timeseries and get_timeseries_n_features queries
        """

        #
        #
        #
        def __init__(self, layername, geodataframe, szyyyymmddfirst, szyyyymmddlast, imaxretries, isleepseconds, dataclient, verbose = True):
            threading.Thread.__init__(self)
            #
            #
            #
            self.layername            = layername
            self.geodataframe         = geodataframe
            self.szyyyymmddfirst      = szyyyymmddfirst
            self.szyyyymmddlast       = szyyyymmddlast
            self.imaxretries          = imaxretries
            self.isleepseconds        = isleepseconds
            self.dataclient           = dataclient
            self.verbose              = verbose
            self.fieldsSeriesDict     = dict.fromkeys(geodataframe.index)
            #
            #
            #
            if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - init: %s entries in geodataframe" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index)))

        #
        #
        #
        def run(self):

            datetime_tick_run = datetime.datetime.now()

            try:

                if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - %s features(%s ... %s) - starts" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1]))
                self._run_get_timeseries_loop()
                if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - %s features(%s ... %s) - ended (%s seconds)" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1], int((datetime.datetime.now()-datetime_tick_run).total_seconds())))

            except Exception:

                logging.error (traceback.format_exc())
                logging.error("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - %s features(%s ... %s) - failed (%s seconds)" % (self.geodataframe.index[0], self.layername, len(self.geodataframe.index), self.geodataframe.index[0], self.geodataframe.index[-1], int((datetime.datetime.now()-datetime_tick_run).total_seconds())))

        #
        #
        #
        def _run_get_timeseries_loop(self):
            """
            dataclient.get_timeseries_multiband is called (in a loop) for each index in the original geodataframe passed in.
            dataclient.get_timeseries_multiband returns a dict of pandas.Series (one per band)
            #
            # {
            #   'VV': pd.Series(...),
            #   'VH': pd.Series(...),
            #   'ANGLE': pd.Series(...)
            # }
            """

            for fieldId in self.fieldsSeriesDict:

                attempt = 0
                while True:

                    attempt += 1

                    datetime_tick_attempt = datetime.datetime.now()

                    try:

                        if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - get_timeseries_multiband(%s) attempt(%s) starts" % (self.geodataframe.index[0], self.layername, fieldId, attempt))
                        dataseries = self.dataclient.get_timeseries_multiband(self.geodataframe.loc[fieldId], self.layername, self.szyyyymmddfirst, self.szyyyymmddlast)
                    except Exception:
                        dataseries = None
                        if self.verbose: logging.info (traceback.format_exc())
                    datetime_tock_attempt = datetime.datetime.now()

                    if (dataseries is not None):
                        if self.verbose: logging.info("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - get_timeseries_multiband(%s) attempt(%s) success (%s seconds)" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                        self.fieldsSeriesDict[fieldId] = dataseries
                        break

                    if (self.imaxretries <= attempt):
                        logging.error("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - get_timeseries_multiband(%s) attempt(%s) failed (%s seconds) - abend" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds())))
                        self.fieldsSeriesDict[fieldId] = None
                        break

                    logging.warning("(%s) S1TimeSeries - Thread_S1_MULTI_BAND_LAYER(%30s) - run - get_timeseries_multiband(%s) attempt(%s) failed (%s seconds) - retry in %s seconds" % (self.geodataframe.index[0], self.layername, fieldId, attempt, int((datetime_tock_attempt-datetime_tick_attempt).total_seconds()), self.isleepseconds * attempt))
                    time.sleep(self.isleepseconds * attempt)

    #
    #
    #
    def __init__(self, fieldIds, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, inarrowfieldbordersinmeter = 0, imaxretries = _MAX_RETRIES, isleepseconds = _SLEEP_SECONDS, bymultibandquery = _S1_MULTI_BAND_QUERY, dataclient = _DATACLIENT, iminfieldsforbulkquery = _MIN_FEATURES_FOR_BULK_QUERY, verbose=True):
        """
        """
        #
        #
        #
        datetime_tick_setup_data  = datetime.datetime.now()
        #
        #
        #
        self._szyyyymmddfirst            = szyyyymmddfirst
        self._szyyyymmddlast             = szyyyymmddlast
        self._imaxretries                = imaxretries
        self._isleepseconds              = isleepseconds
        self._dataclient                 = dataclient
        self._verbose                    = verbose
        self._valid                      = False
        #
        #
        #
        try:
            #
            #    check GeoDataFrame type
            #
            if not isinstance(fieldsgeodataframe, geopandas.GeoDataFrame): raise ValueError("S1TimeSeries(...) fieldsgeodataframe must be a GeoDataFrame")
            if len(fieldsgeodataframe.index) == 0 :                        raise ValueError("S1TimeSeries(...) empty fieldsgeodataframe")
            #
            #    determine fieldIds
            #
            lst_fieldIds = []
            if fieldIds is None:
                try:                        lst_fieldIds.extend(fieldsgeodataframe.index) # could be actual fieldId strings or integer range
                except:                     lst_fieldIds = []
            elif isinstance(fieldIds, str): lst_fieldIds = [fieldIds]                     # single fieldId specified as string
            else:
                try:                        lst_fieldIds.extend(fieldIds)                 # try fieldIds specified as some iterable
                except TypeError:           lst_fieldIds = [fieldIds]                     # else whatever it is, consider it as a single (non-string) fieldId

            if len(lst_fieldIds) == 0 : raise ValueError("S1TimeSeries(...) no valid fieldIds specified or found")
            #
            #    create index from these fieldIds. in normal cases this would be the same as the list passed in
            #
            self._orgfieldsindex = pandas.Index(lst_fieldIds)
            #
            #    create (sub) GeoDataFrame according to these fieldIds. starting from the index determined by the fields => will keep the order of the fields intact 
            #
            self._orgfieldsgeodataframe = fieldsgeodataframe.loc[self._orgfieldsindex.intersection(fieldsgeodataframe.index), :]
            #
            #    check if all fields are accounted for in the GeoDataFrame
            #
            if len(self._orgfieldsgeodataframe.index) != len(self._orgfieldsindex) : raise ValueError("S1TimeSeries(...) fieldsgeodataframe does not cover all specified fieldIds")
            #
            #    some policy is needed in this cat's cradle
            #
            domultibandquery =  bymultibandquery
            dobulkquery      = (not bymultibandquery) and (iminfieldsforbulkquery <= len(self._orgfieldsindex))
            dotheadperlayer  = not dobulkquery
            #
            #
            #
            if verbose:
                if len(self._orgfieldsindex) > 1 : logging.info("(%s) S1TimeSeries - setup for %s fields - first Id: %s" % (self._orgfieldsindex[0], len(self._orgfieldsindex), self._orgfieldsindex[0]))
                else:                              logging.info("(%s) S1TimeSeries - setup for single field - Id: %s"    % (self._orgfieldsindex[0], self._orgfieldsindex[0]))
                if domultibandquery: logging.info("(%s)              - mode: %s" % (self._orgfieldsindex[0], ("one multi-band query per field") + " - " + ("parallel threads for all S1 (multiband) layers" if dotheadperlayer else "looping over the S1  (multiband) layers")))
                else               : logging.info("(%s)              - mode: %s" % (self._orgfieldsindex[0], ("bulk query all fields" if dobulkquery else "one query per field") + " - " + ("parallel threads for all S1 layers" if dotheadperlayer else "looping over the S1 layers")))
            #
            #    check the distance to narrow the field with (absolute value will be used and negated)
            #
            if   inarrowfieldbordersinmeter is None          : self._inarrowfieldbordersinmeter = 0
            elif isinstance(inarrowfieldbordersinmeter, int) : self._inarrowfieldbordersinmeter = inarrowfieldbordersinmeter
            else : raise ValueError("(%s) S1TimeSeries - invalid inward border value %s" % (self._orgfieldsindex[0], inarrowfieldbordersinmeter))
            #
            #    geometry to "UTM 31 N" - actually any meter-based system would do - on crashes: check GDAL environment variables
            #
            geodataframe = self._orgfieldsgeodataframe.to_crs('epsg:32631')
            #
            #    setup narrowed GeoDataFrame - this one will retrieve the average layer value
            #
            narrowedfieldgeodataframe = geodataframe.copy()                                                                   # copy original field
            narrowedfieldgeodataframe['geometry'] = narrowedfieldgeodataframe.buffer(-abs(self._inarrowfieldbordersinmeter))  # calculate buffer (always inward) 
            #
            # 
            #
            narrowedfieldgeodataframe = narrowedfieldgeodataframe.to_crs('epsg:4326')                               # back to EPSG:4326 (WGS 84) to access data-client (once upon a time this had to be EPSG:3857 - Web Mercator)
            #
            #    obtain the data for the specified GeoDataFrames
            #    (converted back from GeoDataFrames to GeoDataSeries since GeoDataFrames cannot be handled by data-client)
            #
            datetime_tick_retrieve_data = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S1TimeSeries(%s ... %s) - fields data retrieval starts. (inward buff: %s m)" % (self._orgfieldsindex[0], self._orgfieldsindex[0], self._orgfieldsindex[-1], self._inarrowfieldbordersinmeter))
            #
            #
            #
            if domultibandquery :
                threadS1 = self.Thread_S1_MULTI_BAND_LAYER("S1_GRD_GAMMA0", narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, self._dataclient, self._verbose)
                if dotheadperlayer: threadS1.start()
                else:               threadS1.run()

                #
                #
                #
                if dotheadperlayer: threadS1.join()
                s1allfieldsSeriesDict = threadS1.fieldsSeriesDict
                if s1allfieldsSeriesDict is None: raise Exception("S1TimeSeries(%s ... %s) S1_GRD_GAMMA0_timeseries data could be retrieved "%(self._orgfieldsindex[0], self._orgfieldsindex[-1]))

                #
                #
                #
                self._s1VVSeriesDict  = dict( (fid, bands['VV'])    for (fid, bands) in s1allfieldsSeriesDict.items() )
                self._s1VHSeriesDict  = dict( (fid, bands['VH'])    for (fid, bands) in s1allfieldsSeriesDict.items() )
                self._s1IASeriesDict  = dict( (fid, bands['ANGLE']) for (fid, bands) in s1allfieldsSeriesDict.items() )

            else:
                threadS1VV  = self.Thread_S1_LAYER("S1_GRD_GAMMA0_VV",     narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, self._dataclient, dobulkquery, self._verbose)
                if dotheadperlayer: threadS1VV.start()
                else:               threadS1VV.run()
    
                threadS1VH  = self.Thread_S1_LAYER("S1_GRD_GAMMA0_VH",     narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, self._dataclient, dobulkquery, self._verbose)
                if dotheadperlayer: threadS1VH.start()
                else:               threadS1VH.run()
    
                threadS1IA  = self.Thread_S1_LAYER("S1_GRD_GAMMA0_ANGLE",  narrowedfieldgeodataframe, self._szyyyymmddfirst, self._szyyyymmddlast, self._imaxretries, self._isleepseconds, self._dataclient, dobulkquery, self._verbose)
                if dotheadperlayer: threadS1IA.start()
                else:               threadS1IA.run()

                #
                #
                #
                if dotheadperlayer: threadS1VV.join()
                s1VVallfieldsSeriesDict = threadS1VV.fieldsSeriesDict
                if s1VVallfieldsSeriesDict is None: raise Exception("S1TimeSeries(%s ... %s) S1_GRD_GAMMA0_VV timeseries data could be retrieved "%(self._orgfieldsindex[0], self._orgfieldsindex[-1]))
    
                if dotheadperlayer: threadS1VH.join()
                s1VHallfieldsSeriesDict = threadS1VH.fieldsSeriesDict
                if s1VHallfieldsSeriesDict is None: raise Exception("S1TimeSeries(%s ... %s) S1_GRD_GAMMA0_VH timeseries data could be retrieved "%(self._orgfieldsindex[0], self._orgfieldsindex[-1]))
    
                if dotheadperlayer: threadS1IA.join()
                s1IAallfieldsSeriesDict = threadS1IA.fieldsSeriesDict
                if s1IAallfieldsSeriesDict is None: raise Exception("S1TimeSeries(%s ... %s) S1_GRD_GAMMA0_ANGLE timeseries data could be retrieved "%(self._orgfieldsindex[0], self._orgfieldsindex[-1]))

                #
                #
                #
                self._s1VVSeriesDict  = s1VVallfieldsSeriesDict
                self._s1VHSeriesDict  = s1VHallfieldsSeriesDict
                self._s1IASeriesDict  = s1IAallfieldsSeriesDict

            #
            #
            #
            datetime_tock_retrieve_data  = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S1TimeSeries(%s ... %s) - fields data retrieval done - time (seconds)                        : %s" % (self._orgfieldsindex[0], self._orgfieldsindex[0], self._orgfieldsindex[-1], int((datetime_tock_retrieve_data-datetime_tick_retrieve_data).total_seconds())))
            #
            #
            #
            self._valid = True
            #
            #
            #
            datetime_tock_setup_data = datetime.datetime.now()
            #
            #
            #
            if self._verbose: logging.info("(%s) S1TimeSeries - fields total setup time (seconds)                                  : %s" % (self._orgfieldsindex[0], int((datetime_tock_setup_data-datetime_tick_setup_data).total_seconds())))

        except Exception:
            logging.error (traceback.format_exc())
            logging.error("(%s) S1TimeSeries - setup failed" % (self._orgfieldsindex[0],))

    #
    #
    #
    def isvalid(self):
        """
        """
        return self._valid

    #
    #
    #
    def getFieldIds(self):
        """
        """
        try:
            return self._orgfieldsindex.tolist()
        except:
            return None

#
#
#
class CleanS1TimeSeriesService:
    """
    actually, doesn't clean anything at the moment
    """
    #
    #
    #
    def __init__(self, inarrowfieldbordersinmeter, imaxretries = 0, isleepseconds = 0, imovingaveragewindow = 0, bymultibandquery = _S1_MULTI_BAND_QUERY, dataclient = _DATACLIENT):
        """
        :param inarrowfieldbordersinmeter
        :param imaxretries   (nrt - suggested imaxretries   = 0, training data creation - suggested: imaxretries = 10)
        :param isleepseconds (nrt - suggested isleepseconds = 0, training data creation - suggested: isleepseconds = 600)
        :param imovingaveragewindow
        """
        #
        #
        #
        self._inarrowfieldbordersinmeter = inarrowfieldbordersinmeter
        self._imaxretries                = imaxretries
        self._isleepseconds              = isleepseconds
        self._imovingaveragewindow       = imovingaveragewindow
        self._bymultibandquery           = bymultibandquery
        self._dataclient                 = dataclient

    #
    #
    #
    def getTimeSeries(self, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, verbose=True):
        """
        debug method returning the actual TimeSeriesData object, so client can has access to 
        detail data (._originalfielddatatimeseries ._cleanfielddatatimeseries and ._fielddataclassification)
        """
        datetime_tick_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("S1TimeSeriesService.getTimeSeries(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s)" % (fieldId, szyyyymmddfirst, szyyyymmddlast))
        #
        #    setup - it will retrieve raw data
        #
        timeseries = S1TimeSeries(fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, inarrowfieldbordersinmeter = self._inarrowfieldbordersinmeter, imaxretries = self._imaxretries, isleepseconds = self._isleepseconds, bymultibandquery = self._bymultibandquery, dataclient = self._dataclient, verbose=verbose)
        #
        #    have it calculate clean data
        #
        timeseries.getTimeSeriesData(self._imovingaveragewindow)
        #
        #
        #
        datetime_tock_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("S1TimeSeriesService.getTimeSeries(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s) done  - (%s seconds)" % (fieldId, szyyyymmddfirst, szyyyymmddlast, int((datetime_tock_gettimeseriesdata-datetime_tick_gettimeseriesdata).total_seconds())) )
        #
        #    return object; client can access ._orgS1VVallfieldsseries ._orgS1VHallfieldsseries, ._orgS1IAallfieldsseries
        #
        return timeseries        

    #
    #
    #
    def getTimeSeriesData(self, fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, verbose=True):
        """
        """
        datetime_tick_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("S1TimeSeriesService.getTimeSeriesData(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s)" % (fieldId, szyyyymmddfirst, szyyyymmddlast))
        #
        #    setup - it will retrieve raw data
        #
        timeseries = S1TimeSeries(fieldId, fieldsgeodataframe, szyyyymmddfirst, szyyyymmddlast, inarrowfieldbordersinmeter = self._inarrowfieldbordersinmeter, imaxretries = self._imaxretries, isleepseconds = self._isleepseconds, bymultibandquery = self._bymultibandquery, dataclient = self._dataclient, verbose=verbose)
        #
        #    have it calculate clean data
        #
        resultDict = timeseries.getTimeSeriesData(self._imovingaveragewindow)
        #
        #
        #
        datetime_tock_gettimeseriesdata = datetime.datetime.now()
        #
        #
        #
        if verbose: logging.info("S1TimeSeriesService.getTimeSeriesData(fieldId: %s, szyyyymmddfirst: %s, szyyyymmddlast: %s) done  - (%s seconds)" % (fieldId, szyyyymmddfirst, szyyyymmddlast, int((datetime_tock_gettimeseriesdata-datetime_tick_gettimeseriesdata).total_seconds())) )
        #
        #
        #
        return resultDict[fieldId]

