#!/usr/bin/env python3

import logging
import datetime

from json.decoder import JSONDecodeError

import pandas as pd
import shapely.geometry
import shapely.ops

from flask import Blueprint, request, jsonify, json, redirect, url_for
from flask_httpauth import HTTPBasicAuth
from flask_swagger_ui import get_swaggerui_blueprint

from werkzeug.exceptions import BadRequest, \
                                Forbidden, \
                                InternalServerError, \
                                UnprocessableEntity, \
                                HTTPException

import cropsar


log = logging.getLogger(__name__)

blueprint = Blueprint(name='api',
                      import_name=__name__,
                      template_folder='templates',
                      static_folder='static')

docs_blueprint = get_swaggerui_blueprint(blueprint_name='apidocs',
                                         base_url='/api/docs',
                                         api_url='/api/openapi.yaml',
                                         config={
                                             'app_name': 'CropSAR service'
                                         })

auth = HTTPBasicAuth()


def _df_to_dict(df):
    return df.to_dict(orient='index')


def _ts_df_to_dict(df):
    df.index = df.index.strftime('%Y-%m-%d')
    for col in df.columns:
        if pd.api.types.is_float_dtype(df[col]):
            df[col] = df[col].where(df[col].notna(), 'NaN')

    return _df_to_dict(df)


def get_timeseries(product, start, end, geometry,
                   crs='epsg:4326', source='probav-mep',
                   user=None, referrer=None,
                   minimal_input_checks=False):

    data = cropsar.retrieve_timeseries(product,
                                       start,
                                       end,
                                       geometry,
                                       crs,
                                       source,
                                       user,
                                       referrer)

    return _ts_df_to_dict(data)


def get_cropsar_analysis(product, start, end, geometry,
                         crs='epsg:4326', source='probav-mep',
                         params=None, user=None, referrer=None,
                         minimal_input_checks=False):

    data = cropsar.retrieve_cropsar_analysis(product,
                                             start,
                                             end,
                                             geometry,
                                             crs,
                                             source,
                                             params,
                                             user,
                                             referrer,
                                             minimal_input_checks)

    def _update(fun, d, key):
        d[key] = fun(d[key])

    _update(_ts_df_to_dict, data, 'cropsar')
    _update(_ts_df_to_dict, data, 'whittaker')
    _update(_ts_df_to_dict, data['clean'], 's1-data')
    _update(_ts_df_to_dict, data['clean'], 's2-data')
    _update(_ts_df_to_dict, data['sources'], 's1-data')
    _update(_ts_df_to_dict, data['sources'], 's2-data')

    for b in data['sources']['s2-scenes']:
        _update(_ts_df_to_dict, data['sources']['s2-scenes'], b)

    _update(_df_to_dict, data['clean']['.metadata'], 'flags')
    _update(_df_to_dict, data['sources']['.metadata'], 'scene-classes')

    return data


def get_cropsar_timeseries(product, start, end, geometry,
                           crs='epsg:4326', source='probav-mep',
                           params=None, user=None, referrer=None,
                           minimal_input_checks=False):

    data = cropsar.retrieve_cropsar(product,
                                    start,
                                    end,
                                    geometry,
                                    crs,
                                    source,
                                    params,
                                    user,
                                    referrer,
                                    minimal_input_checks)

    data = data[['q50']]
    data = data.rename(columns={'q50': 'data'})

    return _ts_df_to_dict(data)


def _default_timeseries_source():
    return 'probav-mep'


def _supported_timeseries_sources():
    return ['probav-mep', 'sentinel-hub']


def _unrestricted_timeseries_sources():
    return ['probav-mep']


def _unrestricted_users():
    return {'admin':   '4Dm1n!',
            'cropsar': 'Cr0ps4r!',
            'devtest': 'fRXEQc4JTq',
            'niab':    'hUgBfL2jR7',
            'geosys':  'fY2gG4Q9uR',
            'swissre': 'QEpgJLxfZ3',
            'radicle': 'SrZoEHlkuJ',
            'pepsico': 'RjBCaxV5vk'}


def _default_timeseries_product():
    return 'S2_FAPAR'


def _supported_timeseries_products():
    return ['S2_FAPAR',
            'S2_FCOVER',
            'S2_NDVI',
            'S2_LAI',
            'S2_SCENECLASSIFICATION',
            'S1_GRD_GAMMA0']


def _supported_cropsar_products():
    return ['S2_FAPAR',
            'S2_FCOVER',
            'S2_NDVI']


def _map_ts_product(product):
    return product


def _map_fis_product(product):
    return product


def _parse_boolean(value):
    lower = value.lower()

    if lower in ('true', 'yes', '1', 'on', 'enabled', 't', 'y'):
        return True
    if lower in ('false', 'no', '0', 'off', 'disabled', 'f', 'n'):
        return False

    raise ValueError('Unsupported boolean value: ' + value)


def _parse_date(date):
    if date is None:
        return None
    if isinstance(date, datetime.date):
        return date
    if not isinstance(date, datetime.datetime):
        date = date.replace('-', '')
        date = datetime.datetime.strptime(date, '%Y%m%d')

    return datetime.date(date.year, date.month, date.day)


def _complete_date_range(start, end):
    if not end:
        now = datetime.datetime.now()
        end = datetime.date(now.year, now.month, now.day)

    if not start:
        start = datetime.date(end.year-1, end.month, end.day)

    return start, end


def _get_timeseries_request_kwargs():

    # Parse 'Referer' header

    referrer = request.referrer

    # Get requested product

    product = request.args.get('product', _default_timeseries_product())

    product = product.replace('-', '_').upper()

    if product not in _supported_timeseries_products():
        raise BadRequest('Invalid parameter value: product=' + product)

    # Get requested end date

    end = request.args.get('end')

    try:
        end = _parse_date(end)
    except ValueError:
        raise BadRequest('Invalid parameter value: end=' + end)

    # Get requested start date

    start = request.args.get('start')

    try:
        start = _parse_date(start)
    except ValueError:
        raise BadRequest('Invalid parameter value: start=' + start)

    # Determine date range

    start, end = _complete_date_range(start, end)

    if end < start:
        raise BadRequest('Invalid range: start=' + start + ' end=' + end)

    start = start.strftime('%Y-%m-%d')
    end   = end.strftime('%Y-%m-%d')

    # Get requested source

    source = request.args.get('source', _default_timeseries_source())

    if source not in _supported_timeseries_sources():
        raise BadRequest('Invalid parameter value: source=' + source)

    user = None

    if source not in _unrestricted_timeseries_sources():
        user = auth.username()

        if user not in _unrestricted_users():
            raise Forbidden('Invalid parameter value: source=' + source)

    # Check feature

    if request.method == 'POST':
        feature_input = 'request body'
        feature = request.get_data(as_text=True)
    else:
        feature_input = 'parameter: feature'
        feature = request.args.get('feature')

    if not feature:
        raise BadRequest('Missing ' + feature_input)

    try:
        feature = json.loads(feature)
    except JSONDecodeError:
        raise BadRequest('Invalid JSON ' + feature_input)

    if not isinstance(feature, dict):
        raise BadRequest('Invalid GeoJSON ' + feature_input)

    # Get feature geometry

    geometry = feature.get('geometry', feature)

    if not isinstance(geometry, dict):
        raise BadRequest('Invalid GeoJSON ' + feature_input)

    if geometry.get('coordinates') in ([], ()):
        raise UnprocessableEntity(
            'Empty geometry in GeoJSON ' + feature_input)

    try:
        geometry = shapely.geometry.shape(geometry)
    except Exception:
        raise BadRequest('Invalid GeoJSON ' + feature_input)

    # Get feature crs

    if 'crs' in feature:
        crs = feature['crs']
    else:
        crs = request.args.get('crs', 'epsg:4326')

    # Get minimal_input_checks flag

    minimal_input_checks = request.args.get('minimal_input_checks', 'false')

    try:
        minimal_input_checks = _parse_boolean(minimal_input_checks)
    except ValueError:
        raise BadRequest('Invalid parameter value: minimal_input_checks=' + minimal_input_checks)

    return {'product': product,
            'start': start,
            'end': end,
            'geometry': geometry,
            'crs': crs,
            'source': source,
            'user': user,
            'referrer': referrer,
            'minimal_input_checks': minimal_input_checks}


@auth.verify_password
def verify_password(username, password):
    users = _unrestricted_users()

    if username in users:
        return password == users[username]

    return True
    

@blueprint.errorhandler(Exception)
def handle_error(e):
    code = 500
    if isinstance(e, HTTPException):
        code = e.code
    return jsonify(error=str(e)), code


@blueprint.route('openapi.yaml')
def openapi_yaml():
    return redirect(url_for('api.static', filename='openapi.yaml'))


@blueprint.route('v1.0/timeseries/', methods=['GET', 'POST'])
@auth.login_required
def v1_0_timeseries():

    kwargs = _get_timeseries_request_kwargs()

    try:
        results = get_timeseries(**kwargs)
    except cropsar.GeometryError as e:
        raise UnprocessableEntity('Failed to collect timeseries: ' + str(e))
    except Exception as e:
        raise InternalServerError('Failed to collect timeseries: ' + str(e))

    return jsonify(results)


@blueprint.route('v1.0/cropsar/', methods=['GET', 'POST'])
@auth.login_required
def v1_0_cropsar():

    kwargs = _get_timeseries_request_kwargs()

    try:
        results = get_cropsar_timeseries(**kwargs)
    except cropsar.GeometryError as e:
        raise UnprocessableEntity('Failed to generate cropsar timeseries: ' + str(e))
    except cropsar.MaxGapError as e:
        raise UnprocessableEntity('Failed to generate cropsar timeseries: ' + str(e))
    except Exception as e:
        raise InternalServerError('Failed to generate cropsar timeseries: ' + str(e))

    return jsonify(results)


@blueprint.route('v1.0/cropsar-analysis/', methods=['GET', 'POST'])
@auth.login_required
def v1_0_cropsar_analysis():

    kwargs = _get_timeseries_request_kwargs()

    try:
        results = get_cropsar_analysis(**kwargs)
    except cropsar.GeometryError as e:
        raise UnprocessableEntity('Failed to generate cropsar analysis: ' + str(e))
    except cropsar.MaxGapError as e:
        raise UnprocessableEntity('Failed to generate cropsar analysis: ' + str(e))
    except Exception as e:
        raise InternalServerError('Failed to generate cropsar analysis: ' + str(e))

    return jsonify(results)


def create_app():
    
    import logging

    logging.basicConfig(level=logging.INFO,
                        format='{asctime} {levelname} {name}: {message}',
                        style='{',
                        datefmt='%Y-%m-%d %H:%M:%S')

    logging.getLogger('urllib3').setLevel(logging.DEBUG)

    import http.client

    http.client.HTTPConnection.debuglevel = 1

    from flask import Flask

    app = Flask(__name__)
    app.register_blueprint(blueprint, url_prefix='/api')
    app.register_blueprint(docs_blueprint, url_prefix='/api/docs')

    return app


if __name__ == '__main__':
    app = create_app()
    app.run(debug=True)
