######################################################################################################
# timesat_phenology.py
# ---------------------------
# Purpose
#       Build OpenEO timesat phenology service
#
######################################################################################################

#imports
import json
import pandas as pd
import geopandas as gpd
from nextland_services.helpers import load_markdown
import openeo
import os
from openeo.api.process import Parameter
from openeo.internal.graph_building import PGNode
from openeo.rest.datacube import DataCube
from openeo.rest.udp import build_process_dict

from nextland_services.helper_functions import get_input_datacube_udf_cropsar, apply_inwards_buffering
import logging
import shapely



from nextland_services.constants import *

_log = logging.getLogger("timesat_phenology")

#Generate results paths
OUTPUT_DIR = Path(__file__).with_suffix('.output')
OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
OUTPUT_DIR_GRAPH = RESOURCES
BASEPATH = Path(__file__).parent


# Functions

def _get_epsg(lat, zone_nr):
    if lat >= 0:
        epsg_code = '326' + str(zone_nr)
    else:
        epsg_code = '327' + str(zone_nr)
    return int(epsg_code)

def load_udf(udf):
    with open(Path(__file__).parent / udf, 'r+', encoding= "utf8") as f:
        return f.read()

#Write content to named file
def fwrite(fname, fcontent):
    f = open(fname, 'w')
    f.write(str(fcontent))
    f.close()


def get_input_timeseries_udf_timesat_biopar(eoconn, metric, time_range, geo, metric_type_param =  None, metric_fixed = False, bbox = False):
    if metric_fixed:
        context = {"metric": "{}".format(metric)}
    else:
        context = {"metric": {"from_parameter": "metric"}}

    S2_bands = eoconn.load_collection('SENTINEL2_L2A_SENTINELHUB',
                                      bands=["B03", "B04", "B08", "sunAzimuthAngles", "sunZenithAngles",
                                             "viewAzimuthMean", "viewZenithMean", "SCL"])
    S2_bands_mask = S2_bands.process("mask_scl_dilation", data=S2_bands,
                                     scl_band_name="SCL")

    udf = load_udf(Path(__file__).parent / 'biopar_udf.py')
    biopar_masked = S2_bands_mask.reduce_dimension(dimension="bands", reducer=PGNode(
        process_id="run_udf",
        data={"from_parameter": "data"},
        udf=udf,
        runtime="Python",
        context=context
    ))
    biopar_masked = biopar_masked.add_dimension('bands', label='band_0', type='bands').filter_temporal(time_range).polygonal_mean_timeseries(geo)

    input_datacube = apply_inwards_buffering(biopar_masked)

    return input_datacube


def get_input_timeseries_udf_timesat_NDVI(eoconn, time_range, geo, bbox = False):
    S2_bands = eoconn.load_collection('SENTINEL2_L2A_SENTINELHUB',
                                      bands=["B04", "B08", "SCL"])
    S2_bands_mask = S2_bands.process("mask_scl_dilation", data=S2_bands,
                                     scl_band_name="SCL")
    ndviband=S2_bands_mask.filter_bands(["B04","B08"]).ndvi(red="B04",nir="B08").filter_temporal(time_range).polygonal_mean_timeseries(geo)
    input_datacube = apply_inwards_buffering(ndviband)

    return input_datacube


def create_timeseries_udf(eoconn, time_range, geo, metric, metric_type_param = None, metric_fixed = False):
    if metric in ['FAPAR','LAI','FCOVER']:
        TS_cube = get_input_timeseries_udf_timesat_biopar(eoconn,metric, time_range, geo, metric_type_param ,metric_fixed,)
    elif metric == "NDVI":
        TS_cube = get_input_timeseries_udf_timesat_NDVI(eoconn,time_range,geo)

    return TS_cube

###  request the input datacube data for running the UDF
def load_timeseries_data():
    metric = 'NDVI'
    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    time_range = "2021-04-01", "2021-12-31"
    eoconn = openeo.connect("https://openeo-dev.vito.be/openeo/1.0").authenticate_basic('bontek', 'bontek123')

    TS_cube = create_timeseries_udf(eoconn,time_range,polygon,metric, metric_fixed= True)

    timeseries_input_poly = TS_cube.execute()#.send_job().start_and_wait().get_result().load_json()
    with open(OUTPUT_DIR / "TIMESAT_inputs_poly_{}.json".format(metric), 'w') as f:
        json.dump(timeseries_input_poly, f)



# Test running the udf
def run_udf_timesat():
    metric = 'NDVI'
    bbox = {
        "west": 5.15183687210083,
        "east": 5.153381824493408,
        "south": 51.18192559252128,
        "north": 51.18469636040683,
        "crs": "EPSG:4326"
    }

    polygon = shapely.geometry.Polygon([[bbox['west'], bbox['south']], [bbox['west'], bbox['north']],
                                        [bbox['east'], bbox['north']], [bbox['east'], bbox['south']]])

    time_range = "2021-04-01", "2021-12-31"
    eoconn = openeo.connect("https://openeo-dev.vito.be/openeo/1.0").authenticate_basic('bontek', 'bontek123')

    TS_cube = create_timeseries_udf(eoconn, time_range, polygon, metric, metric_fixed=True)

    timesat_code = load_udf("udf_timesat.py")

    context = {"metric": metric}

    udf = TS_cube.process("run_udf", data = TS_cube, udf = timesat_code, runtime = "Python", context = context)

    res = udf.send_job().start_and_wait().get_result().load_json()
    return res


def debug_timesat_phenology_udf():
    from openeo_udf.api.udf_data import UdfData
    from openeo_udf.api.structured_data import StructuredData
    from udf_timesat import udf_timesat

    metric = 'NDVI'
    context =  {"metric": metric}
    with open(OUTPUT_DIR / "TIMESAT_inputs_poly_{}.json".format(metric),'r') as json_file:
        ts = json.load(json_file)
        udfdata = UdfData({"EPSG":4326}, structured_data_list=[StructuredData(description= "timeseries input", data = ts, type = "dict")])
        udfdata.user_context = context
        res = udf_timesat(udfdata)
        fwrite(os.path.join(OUTPUT_DIR, 'TIMESAT_PHEN_result.res'.format(metric)), res.to_dict())


# Build graph timesat phenology and write it away as a JSON file
def timesat_phenology_build_graph_poly_and_store_udp():

    metric = 'FCOVER'
    eoconn = openeo.connect("https://openeo-dev.vito.be/openeo/1.0").authenticate_basic('bontek', 'bontek123')
    #Define the service parameters
    time_range = date_parameter("Left-closed temporal interval, i.e. an array with exactly two elements:\n\n1. The first element is the start of the temporal interval. The specified instance in time is **included** in the interval.\n2. The second element is the end of the temporal interval. The specified instance in time is **excluded** from the interval.\n\nThe specified temporal strings follow [RFC 3339](https://www.rfc-editor.org/rfc/rfc3339.html). Also supports open intervals by setting one of the boundaries to `null`, but never both.")



    if metric == 'NDVI':
        metric_type = Parameter.string(
            name="metric",
            description='Metric used to calculate the phenology from. As default the NDVI is used',
            default="NDVI")
        process_id = 'phenology_with_timesat_ndvi_graph'
    else:
        metric_type = Parameter.string(
        name="metric", description='Metric used to calculate the phenology from. As default the FAPAR is used. Options between: [FAPAR, LAI, FCOVER]', default="FAPAR")
        process_id = 'phenology_with_timesat_biopar_graph'

    polygon = polygon_param(description="A geojson polygon on which to compute the metric timeseries.")

    udf = timesat_phenology_cube(eoconn, polygon,time_range, metric, metric_type)

    #Build service dict
    timesat_phenology_dict = build_process_dict(
        process_id= process_id,
        description=load_markdown("timesat_phenology.md"),
        process_graph= udf,
        parameters= [time_range, polygon, metric_type]
    )

    #Write service graph to json file

    fwrite(os.path.join(OUTPUT_DIR_GRAPH, '{}.json'.format(process_id)), json.dumps(timesat_phenology_dict, indent = 4))


def timesat_phenology_cube(eoconn, file_polygons, time_range, metric, metric_type_param):
    timeseries_input = create_timeseries_udf(eoconn, time_range, file_polygons, metric, metric_type_param)
    # load UDF
    timesat_code = load_udf("udf_timesat.py")

    context = {'metric': {'from_parameter': 'metric'}}
    udf = timeseries_input.process("run_udf", data=timeseries_input, udf=timesat_code, runtime="Python", context=context)

    return udf


def save_udp():
    # Define the service parameters
    metric = 'FCOVER'
    eoconn = openeo.connect("https://openeo-dev.vito.be/openeo/1.0").authenticate_basic('bontek', 'bontek123')
    # Define the service parameters
    time_range = date_parameter(
        "Left-closed temporal interval, i.e. an array with exactly two elements:\n\n1. The first element is the start of the temporal interval. The specified instance in time is **included** in the interval.\n2. The second element is the end of the temporal interval. The specified instance in time is **excluded** from the interval.\n\nThe specified temporal strings follow [RFC 3339](https://www.rfc-editor.org/rfc/rfc3339.html). Also supports open intervals by setting one of the boundaries to `null`, but never both.")

    if metric == 'NDVI':
        metric_type = Parameter.string(
            name="metric",
            description='Metric used to calculate the phenology from. As default the NDVI is used',
            default="NDVI")
        process_id = 'phenology_with_timesat_ndvi_graph'
    else:
        metric_type = Parameter.string(
            name="metric",
            description='Metric used to calculate the phenology from. As default the FAPAR is used. Options between: [FAPAR, LAI, FCOVER]',
            default="FAPAR")
        process_id = 'phenology_with_timesat_biopar_graph'

    polygon = polygon_param(description="A geojson polygon on which to compute the metric timeseries.")

    with open(os.path.join(OUTPUT_DIR_GRAPH,'{}.json'.format(process_id)),'r') as file:
        graph = json.load(file)
    udp = eoconn.save_user_defined_process(
        process_id,
        graph["process_graph"],
        description= load_markdown("timesat_phenology.md"),
        parameters=[time_range, polygon, metric_type], public=True)


#load_timeseries_data()
#debug_timesat_phenology_udf()
#res = run_udf_timesat()
#timesat_phenology_build_graph_poly_and_store_udp()
#save_udp()