import json

import geopandas as gpd
from openeo.rest.datacube import DataCube

from nextland_services.constants import *


def load_markdown(name: str) -> str:
    with open(MARKDOWN / name, 'r', encoding='utf-8') as file:
        return file.read()


def load_text(path) -> str:
    with open(path, 'r+') as f:
        return f.read()


def load_udf(relative_path):
    with open(relative_path, 'r+') as f:
        return f.read()


def read_input(path) -> dict:
    """
    Read the input file and return the a FeatureCollection
    :param path: Path of the input file
    :return: FeatureCollection
    """

    if '.json' in path or '.geojson' in path:
        return read_geojson(path)
    elif '.shp' in path:
        return read_shapefile(path)
    else:
        raise Exception(f'Can not read input: {path} - format not supported')


def read_shapefile(path) -> dict:
    """
     Read a GeoJSON file from disk
     :param path: Path to the GeoJSON file
     :return: A dictionary representing the contents of the GeoJSON file
     """
    shapes_gdf = gpd.read_file(path, encoding='utf-8')
    return shapes_gdf.__geo_interface__


def read_geojson(path) -> dict:
    """
    Read a GeoJSON file from disk
    :param path: Path to the GeoJSON file
    :return: A dictionary representing the contents of the GeoJSON file
    """
    with open(path) as f:
        geom = json.load(f)
        f.close()
    return geom


def write_json(path, data: dict) -> dict:
    """
    Read a json file to disk
    :param path: Path to the json file
    :param data: Data to write to disk
    """
    with open(path, 'w') as f:
        json.dump(data, f, indent=4)
        f.close()


def execute_batch_job(datacube: DataCube, title: str, target: str) -> dict:
    """
    Execute a batch job through OpenEO and wait for the results
    :param datacube: OpenEO processing datacube to execute
    :param title: Title of the batch job
    :param target: Target folder where results will be stored
    :return: Dict containing the downloaded file path as value and asset metadata
    """
    return datacube.send_job(title=title).start_and_wait().download_results(target=target)


def get_extension(format: str) -> str:
    """
    Get the extension for the given format
    :param format: Format of the output file
    :return: String representing the extension to use
    """
    if format.lower() == 'gtiff':
        return 'tiff'
    elif format.lower() == 'geojson':
        return 'geojson'
    elif format.lower() == 'json':
        return 'json'
    elif format.lower() == 'netcdf':
        return 'nc'
    else:
        raise Exception(f'Format {format} is not supported')
