######################################################################################################
# vegetationwaterindices.py
# ---------------------------
# Purpose
#       Build OpenEO vegetation water indices services
#
######################################################################################################

#imports
import numpy as np
import openeo
from openeo.api.process import Parameter
from openeo.rest.udp import build_process_dict
import scipy.signal
import logging
from pathlib import Path
from pprint import pprint
import os
import json
import shapely

from .constants import *
from .helpers import *

_log = logging.getLogger("vegetation_water_indices")

#Generate results paths
OUTPUT_DIR = Path(__file__).with_suffix('.output')
OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
OUTPUT_DIR_UDP = Path(__file__).with_suffix('.output_UDP')
OUTPUT_DIR_UDP.mkdir(parents=True, exist_ok=True)

#Write content to named file
def fwrite(fname, fcontent):
    f = open(fname, 'w')
    f.write(str(fcontent))
    f.close()


#Get Scene Classification for collection
def get_scene_classification(eoconn, collection_id="SENTINEL2_L2A_SENTINELHUB", band_scene_classification="SCL"):
    s2 = eoconn.load_collection(collection_id, bands=[band_scene_classification])
    return s2.band(0)

#NEX-19 NDWI Service
#----- NDWI ------

#Create ndwi using cropsar cloud mask
def get_ndwi(eoconn, collection_id="SENTINEL2_L2A_SENTINELHUB", band08='B08', band12='B12',
             band_scene_classification="SCL"):

    if(band08==None):
        band08='B08'
    if(band12==None):
        band12='B12'

    s2_l2a = eoconn.load_collection(collection_id, bands=[band08, band12,band_scene_classification])
    s2_l2a_masked = s2_l2a.process("mask_scl_dilation", data=s2_l2a,scl_band_name=band_scene_classification)

    ### NDWI = (B08 - B12)/(B08 + B12)
    B08 = s2_l2a_masked.band(0)
    B12 = s2_l2a_masked.band(1)

    ndwi = (B08 - B12) / (B08 + B12)

    return ndwi

def water_index_build_graph(eoconn,index:str,band1,band2):
    band_classification = Parameter.string(
        name="band_classification", description="Band name of the scene classification band",
        default="SCL"
    )
    collection_id = SENTINEL2_COLLECTION_PARAM
    date = date_parameter()
    '''bbox = Parameter(
        name="bbox", schema="object"
    )'''
    polygon = polygon_param(description="Polygon for which to compute {index}, specified as GEOJSON.".format(index=index))

    ndwi = water_index(index, eoconn, collection_id, date, polygon, band_classification, band1, band2)
    ndwi_tmp = ndwi.to_json()
    ndwi_json = json.loads(ndwi_tmp)
    ndwi_pg = ndwi_json["process_graph"]

    ndwi_descript = load_markdown(index.lower() + ".md")

    ndwi_dict = build_process_dict(
        process_id=index,
        summary=index + " calculation",
        description=ndwi_descript,
        process_graph=ndwi_pg,
        parameters=[collection_id, band1, band2, band_classification, date, polygon]
    )

    # Write JSON graph
    fwrite(os.path.join(RESOURCES, index.lower() + '_graph.json'), json.dumps(ndwi_dict, indent=4))

def ndwi_build_graph(eoconn):

    band08 = Parameter.string(
        name="band08", description="Band name of Sentinel-2 band 08", default="B08"
    )
    band12 = Parameter.string(
        name="band12", description="Band name of Sentinel-2 band 12", default="B12"
    )
    water_index_build_graph(eoconn,"NDWI",band08,band12)


#Using the ndwi directly (no UDP)
def ndwi_direct(eoconn, date, bbox):

    _log.info("Downloading NDWI (direct)")
    ndwi_direct = get_ndwi(eoconn)
    ndwi_direct = ndwi_direct.filter_temporal(date, date).filter_bbox(bbox)
    fwrite(os.path.join(OUTPUT_DIR_UDP, 'ndwi_direct.json'), ndwi_direct.to_json())
    '''with open(OUTPUT_DIR / "ndwi_direct.json", "w") as f:
        f.write(ndwi_direct.to_json())'''
    ndwi_direct.download(OUTPUT_DIR_UDP / "ndwi_direct.tiff", format="GTIFF")

def ndwi_call(eoconn):
    _log.info("Downloading NDWI (call)")

    date = "2018-06-27","2018-06-27"
    bbox = {
        "west": 6.8, "south": 50.54,
        "east": 6.9, "north": 50.61,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndwi = eoconn.datacube_from_process("NDWI", namespace="nextland", date=date,
                                        polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR, 'ndwi_call_0001.json'), ndwi.to_json())
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_0001.tiff"), format="GTIFF")
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_0001.nc"), format="NetCDF")


    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.837040667417134, "south": 42.87743216402034,
        "east": -3.8345952719383525, "north": 42.87994265425226,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndwi = eoconn.datacube_from_process("NDWI", namespace="nextland", date=date, collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band08='TOC-B08_10M',band12='TOC-B12_20M'
                                            ,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'ndwi_call_terra_0002.json'),ndwi.to_json())
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_terra_0002.tiff"), format="GTIFF")
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_terra_0002.nc"), format="NetCDF")


    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.844752324294209, "south": 42.87970306999923,
        "east": -3.8363300630609727, "north": 42.884282967100305,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndwi = eoconn.datacube_from_process("NDWI", namespace="nextland", date=date, collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band08='TOC-B08_10M',band12='TOC-B12_20M'
                                            ,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'ndwi_call_terra_0003.json'),ndwi.to_json())
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_terra_0003.tiff"), format="GTIFF")
    ndwi.download(os.path.join(OUTPUT_DIR,"ndwi_call_terra_0003.nc"), format="NetCDF")

#----- END NDWI ------

#NEX-20 NDII Service
#----- NDII ------

#Create ndwi using cropsar cloud mask
def get_ndii(eoconn, collection_id="SENTINEL2_L2A_SENTINELHUB", band8A='B8A', band11='B11',
             band_scene_classification="SCL"):

    if(band8A==None):
        band8A='B8A'
    if(band11==None):
        band11='B11'
    s2_l2a = eoconn.load_collection(collection_id, bands=[band8A, band11, band_scene_classification])
    s2_l2a_masked = s2_l2a.process("mask_scl_dilation", data=s2_l2a,scl_band_name=band_scene_classification)

    ### NDII = (B8A - B11) /(B8A + B11)
    B8A = s2_l2a_masked.band(0)
    B11 = s2_l2a_masked.band(1)

    ndii = (B8A - B11) / (B8A + B11)

    return ndii


def ndii_build_graph(eoconn):
    band8A = Parameter.string(
        name="band8A", description="Band name of Sentinel-2 band 8A", default="B8A"
    )
    band11 = Parameter.string(
        name="band11", description="Band name of Sentinel-2 band 11", default="B11"
    )
    water_index_build_graph(eoconn,"NDII",band8A,band11)


def water_index(index, eoconn, collection_id, date, polygon, band_classification, band1=None, band2=None):

    if(index == "NDII"):
        cube = get_ndii(
            eoconn=eoconn,
            collection_id=collection_id,
            band_scene_classification=band_classification,
            band8A=band1,
            band11=band2
        )
    elif (index == "MSI"):
        cube = get_msi(
            eoconn=eoconn,
            collection_id=collection_id,
            band_scene_classification=band_classification,
            band08=band1,
            band11=band2
        )
    elif (index == "NDWI"):
        cube = get_ndwi(
            eoconn=eoconn,
            collection_id=collection_id,
            band_scene_classification=band_classification,
            band08=band1,
            band12=band2
        )

    # ndii = ndii.filter_temporal(date).filter_bbox(bbox)
    return cube.filter_temporal(date).mask_polygon(polygon)



def ndii_call(eoconn):

    _log.info("Downloading NDII (call)")

    date = "2018-06-27","2018-06-27"
    bbox = {
        "west": 6.8, "south": 50.54,
        "east": 6.9, "north": 50.61,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndii = eoconn.datacube_from_process("NDII", namespace="nextland", date=date
                                        ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR, 'ndii_call_0001.json'), ndii.to_json())
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_0001.tiff"), format="GTIFF")
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_0001.nc"), format="NetCDF")

    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.837040667417134, "south": 42.87743216402034,
        "east": -3.8345952719383525, "north": 42.87994265425226,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndii = eoconn.datacube_from_process("NDII", namespace="nextland",collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band8A='TOC-B8A_20M',band11='TOC-B11_20M'
                                            ,date=date,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'ndii_call_terra_0002.json'),ndii.to_json())
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_terra_0002.tiff"), format="GTIFF")
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_terra_0002.nc"), format="NetCDF")

    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.844752324294209, "south": 42.87970306999923,
        "east": -3.8363300630609727, "north": 42.884282967100305,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    ndii = eoconn.datacube_from_process("NDII", namespace="nextland",collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band8A='TOC-B8A_20M',band11='TOC-B11_20M'
                                            ,date=date,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'ndii_call_terra_0003.json'),ndii.to_json())
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_terra_0003.tiff"), format="GTIFF")
    ndii.download(os.path.join(OUTPUT_DIR,"ndii_call_terra_0003.nc"), format="NetCDF")

#----- END NDII ------


#NEX-21 MSI Service
#----- MSI ------

#Create msi using cropsar cloud mask
def get_msi(eoconn, collection_id="SENTINEL2_L2A_SENTINELHUB", band08='B08', band11='B11',
             band_scene_classification="SCL"):

    if(band08==None):
        band08='B08'
    if(band11==None):
        band11='B11'
    s2_l2a = eoconn.load_collection(collection_id, bands=[band08, band11,band_scene_classification])
    s2_l2a_masked = s2_l2a.process("mask_scl_dilation", data=s2_l2a,scl_band_name=band_scene_classification)

    ### MSI = B11/B08
    B08 = s2_l2a_masked.band(0)
    B11 = s2_l2a_masked.band(1)

    msi = B11/B08

    return msi


def msi_build_graph(eoconn):


    band08 = Parameter.string(
        name="band08", description="Band name of Sentinel-2 band 08", default="B08"
    )
    band11 = Parameter.string(
        name="band11", description="Band name of Sentinel-2 band 11", default="B11"
    )

    water_index_build_graph(eoconn,"MSI",band08,band11)


def msi_call(eoconn):

    _log.info("Downloading MSI (call)")

    date = "2018-06-27","2018-06-27"
    bbox = {
        "west": 6.8, "south": 50.54,
        "east": 6.9, "north": 50.61,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    msi = eoconn.datacube_from_process("MSI", namespace="nextland", date=date
                                       ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR, 'msi_call_0001.json'), msi.to_json())
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_0001.tiff"), format="GTIFF")
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_0001.nc"), format="NetCDF")

    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.837040667417134, "south": 42.87743216402034,
        "east": -3.8345952719383525, "north": 42.87994265425226,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    msi = eoconn.datacube_from_process("MSI", namespace="nextland",collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band08='TOC-B08_10M',band11='TOC-B11_20M'
                                            ,date=date,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'msi_call_terra_0002.json'),msi.to_json())
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_terra_0002.tiff"), format="GTIFF")
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_terra_0002.nc"), format="NetCDF")

    date = ["2017-04-01","2020-11-15"]
    bbox = {
        "west": -3.844752324294209, "south": 42.87970306999923,
        "east": -3.8363300630609727, "north": 42.884282967100305,
        "crs": "EPSG:4326"
    }
    _log.info(f"Constraints: {date} {bbox}")
    poly = shapely.geometry.box(bbox['east'], bbox['south'], bbox['west'], bbox['north'])

    msi = eoconn.datacube_from_process("MSI", namespace="nextland",collection_id="TERRASCOPE_S2_TOC_V2"
                                            ,band08='TOC-B08_10M',band11='TOC-B11_20M'
                                            ,date=date,band_classification='SCENECLASSIFICATION_20M'
                                            ,polygon=shapely.geometry.mapping(poly))
    fwrite(os.path.join(OUTPUT_DIR,'msi_call_terra_0003.json'),msi.to_json())
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_terra_0003.tiff"), format="GTIFF")
    msi.download(os.path.join(OUTPUT_DIR,"msi_call_terra_0003.nc"), format="NetCDF")

#----- END MSI ------


#Generate graph json files for VITO backend ingestion
def test_generate_graph_services():
    logging.basicConfig(level=logging.INFO)
    _log = logging.getLogger("vegetation_water_indices")
    _log.info("Create Graphs Vegetation Water Indices Services")
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    ndwi_build_graph(eoconn)
    ndii_build_graph(eoconn)
    msi_build_graph(eoconn)


#Test calling OpenEO VITO backend injected process graphs
def test_call_services():
    logging.basicConfig(level=logging.INFO)
    _log.info("Test call Vegetation Water Indices Services")
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()

    ndwi_call(eoconn)
    ndii_call(eoconn)
    msi_call(eoconn)

def test_call_services_shp():
    logging.basicConfig(level=logging.INFO)
    _log.info("Test call Vegetation Water Indices Services using shape file")
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    date = "2018-06-10","2018-06-30"

    import geopandas as gpd

    services = ['NDWI','MSI','NDII']
    shapes = []

    shapes.append(Path(BASEPATH).joinpath('data','parcel-in-retie.shp'))
    shapes.append(Path(BASEPATH).joinpath('data','D0911317_sup_depurada_WGS84.shp'))

    for shp in shapes:
        gpd_shp = gpd.read_file(shp)
        js_shp = json.loads(gpd_shp.to_json())

        shp_nm = Path(shp).stem

        fwrite(os.path.join(OUTPUT_DIR, '%s_geom.json' % (shp_nm)), json.dumps(js_shp, indent=4))

        for poly in js_shp['features']:
            polyid = poly['id']
            polyg = poly['geometry']

            for service in services:
                serv = eoconn.datacube_from_process(service, namespace="nextland", date=date
                                                    , polygon=poly)
                fwrite(os.path.join(OUTPUT_DIR, '%s_CALL_%s_ID-%s_0001.json' % (service,shp_nm,polyid)), serv.to_json())
                serv.download(os.path.join(OUTPUT_DIR, '%s_%s_ID-%s_0001.tiff' % (service,shp_nm,polyid)), format="GTIFF")
                serv.download(os.path.join(OUTPUT_DIR, '%s_%s_ID-%s_0001.nc' % (service,shp_nm,polyid)), format="NetCDF")
                serv.download(os.path.join(OUTPUT_DIR, '%s_%s_ID-%s_0001.json' % (service,shp_nm,polyid)), format="JSON")

def test_ndwi_timeseries():
    logging.basicConfig(level=logging.INFO)
    _log.info("Test call Vegetation Water Indices Services NDWI timeseries")
    eoconn = openeo.connect("https://openeo-dev.vito.be").authenticate_basic()
    date = "2018-06-10","2018-06-30"

    import geopandas as gpd

    services = ['NDWI']
    shapes = []

    shapes.append(Path(BASEPATH).joinpath('data','parcel-in-retie.shp'))

    for shp in shapes:
        gpd_shp = gpd.read_file(shp)
        js_shp = json.loads(gpd_shp.to_json())
        shp_nm = Path(shp).stem

        for poly in js_shp['features']:
            polyid = poly['id']
            polyg = poly['geometry']

            for service in services:
                serv = eoconn.datacube_from_process(service, namespace="nextland", date=date
                                                    , polygon=poly).polygonal_mean_timeseries(polyg)
                serv.download(os.path.join(OUTPUT_DIR, 'TS_%s_%s_ID-%s_0001.json' % (service,shp_nm,polyid)), format="JSON")
