#!/usr/bin/env python3

import argparse
import json
import os
import os.path
import sys
import traceback
import pkgutil
import datetime
import logging
import time
import threading
import queue
import re
import copy

from concurrent.futures import ThreadPoolExecutor

import tkinter as tk
import tkinter.ttk as ttk
import tkinter.font
import tkinter.filedialog
import tkinter.messagebox

import requests
import numpy as np
import pandas as pd
import geopandas as gpd
import shapely.geometry
import rasterio
import rasterio.mask
import rasterio.plot
import matplotlib.pyplot as plt
import matplotlib.dates
import matplotlib.colors
import matplotlib.patches
import matplotlib.gridspec

import cropsar


pd.plotting.register_matplotlib_converters()

plt.style.use('seaborn-darkgrid')

plt.rc('font', size=9)
plt.rc('axes', titlesize=10, labelsize=9)
plt.rc('xtick', labelsize=9)
plt.rc('ytick', labelsize=9)
plt.rc('legend', fontsize=9)
plt.rc('figure', titlesize=11)


log = logging.getLogger(__name__)


##############################################################################
#
# TIMESERIES DATA COLLECTION
#
##############################################################################


class ImageServiceSentinelHub(object):

    # noinspection PyPackageRequirements
    def get_image(self, layer, geometry, crs, date, mask_exterior=False):

        # TODO: remove this hack once we've converted to v3 sentinel-hub
        #  scripts, apparantly minimal scripts like '[SCL]' apparantly don't
        #  work for WCS requests
        if layer == 'S2_SCENECLASSIFICATION':
            layer = 'S2_SCENECLASSIFICATION_VIS'

        # https://sentinel-hub.com/develop/documentation/api/ogc_api/wcs-request
        # https://sentinel-hub.com/develop/documentation/api/ogc_api/wms-parameters
        # https://sentinel-hub.com/develop/documentation/api/custom-url-parameters

        image_crs = 'epsg:3857'
        geometry = cropsar._crs.geometry_to_crs(geometry,
                                                crs,
                                                image_crs)

        # Download the image from the Sentinel-Hub OGC/WCS service

        instance_id = '69db9d34-5fc2-4583-adb9-7b1079e0fe67'

        response = requests.get(url='http://services.sentinel-hub.com/ogc/wcs/' + instance_id,
                                params={'SERVICE': 'WCS',
                                        'VERSION': '1.1.2',
                                        'REQUEST': 'GetCoverage',
                                        'TIME': date + '/' + date,
                                        'COVERAGE': layer,
                                        'FORMAT': 'image/tiff;depth=32f', #'GeoTIFF',
                                        'BBOX': ','.join([str(f) for f in geometry.bounds]),
                                        'CRS': image_crs,
                                        'RESX': '10m',
                                        'RESY': '10m',
                                        'MAXCC': 100.0})

        if response.status_code != 200:
            raise RuntimeError(response.text)

        # Load the image from memory

        with rasterio.io.MemoryFile(response.content) as memfile:
            with memfile.open() as image:
                if not mask_exterior:

                    # Crop to the bounding box of the parcel

                    x0 = geometry.bounds[0]
                    y0 = geometry.bounds[1]
                    x1 = geometry.bounds[2]
                    y1 = geometry.bounds[3]

                    geometry = shapely.geometry.box(x0, y0, x1, y1)

                shapes = [shapely.geometry.mapping(geometry)]

                data, tf = rasterio.mask.mask(dataset=image,
                                              shapes=shapes,
                                              crop=True,
                                              all_touched=True)

                extent = rasterio.plot.plotting_extent(data[0], tf)

                # And return the cropped image data

                return {'data': data,
                        'extent': extent,
                        'crs': image.crs,
                        'transform': tf,
                        'date': date}


class ImageServiceOpenEO(object):

    def __init__(self):
        self.endpoint = 'http://openeo.vgt.vito.be/openeo/0.4.0'
        #self.endpoint = 'http://openeo-dev.vgt.vito.be/openeo/0.4.0'

        resp = requests.get(url=self.endpoint + '/credentials/basic',
                            auth=('cropsar',
                                  'cropsar123')).json()

        self.access_token = resp['access_token']

    # A class that abstracts the retrieval of image data.
    #
    # This implementation is backed by the MEP's OpenEO web service.
    #

    def get_image(self, layer, geometry, crs, date, mask_exterior=False):

        # Returns an image for the specified data layer, parcel and date.
        #
        # The result is a dictionary of the form:
        #
        #   {
        #     'data': <numpy ndarray>,
        #     'extent': (left, right, bottom, top),
        #     'crs': <CRS spec>,
        #     'transform': <Affine transform>
        #   }
        #

        if layer == 'S1_GRD_GAMMA0':
            return self._get_grd_gamma0(geometry,
                                        crs,
                                        date,
                                        mask_exterior)
        elif layer == 'S2_RADIOMETRY':
            return self._get_image('TERRASCOPE_S2_TOC_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['TOC-B02_10M',
                                    'TOC-B03_10M',
                                    'TOC-B04_10M',
                                    'TOC-B08_10M'])
        elif layer == 'S2_SCENECLASSIFICATION':
            return self._get_image('TERRASCOPE_S2_TOC_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['SCENECLASSIFICATION_20M'])
        elif layer == 'S2_FAPAR':
            return self._get_image('TERRASCOPE_S2_FAPAR_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['FAPAR_10M'])
        elif layer == 'S2_NDVI':
            return self._get_image('TERRASCOPE_S2_NDVI_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['NDVI_10M'])
        elif layer == 'S2_FCOVER':
            return self._get_image('TERRASCOPE_S2_FCOVER_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['FCOVER_10M'])
        elif layer == 'S2_LAI':
            return self._get_image('TERRASCOPE_S2_LAI_V2',
                                   geometry,
                                   crs,
                                   date,
                                   mask_exterior,
                                   ['LAI_10M'])

        raise RuntimeError('Invalid layer name')

    def _get_grd_gamma0(self, geometry, crs, date, mask_exterior=False):

        # A GAMMA0 layer isn't available through OpenEO, so we try to
        # derive it on the fly from the S1_GRD_SIGMA0_ASCENDING and
        # S1_GRD_SIGMA0_DESCENDING layers.

        layers = ['S1_GRD_SIGMA0_ASCENDING',
                  'S1_GRD_SIGMA0_DESCENDING']

        with ThreadPoolExecutor(max_workers=2) as pool:
            images = {layer: pool.submit(self._get_image,
                                         layer,
                                         geometry,
                                         crs,
                                         date,
                                         mask_exterior,
                                         ['VV', 'VH', 'angle'])
                       for layer in layers}

            for layer in layers:
                try:
                    images[layer] = images[layer].result()
                except (LookupError, RuntimeError):
                    del images[layer]

        # Convert sigma to gamma

        for layer, image in images.items():
            if image:
                sigma0_vv = image['data'][0]
                sigma0_vh = image['data'][1]
                sigma0_angle = image['data'][2]

                d = np.cos(np.deg2rad(sigma0_angle*0.0005 + 29.0))

                gamma0_vv = sigma0_vv / d
                gamma0_vh = sigma0_vh / d
                gamma0_angle = sigma0_angle

                image['data'][0] = gamma0_vv
                image['data'][1] = gamma0_vh
                image['data'][2] = gamma0_angle

        for layer in layers:
            if layer in images:
                return images[layer]

        raise LookupError('Image not available')

    def _get_image(self, layer, geometry, crs, date, mask_exterior, bands):

        # Download the image from the OpenEO web service

        graph = self._build_graph(layer, geometry, crs, date, bands)

        response = requests.post(url=self.endpoint + '/result',
                                 json=graph,
                                 stream=True,
                                 timeout=5000,
                                 headers={'Authorization': 'Bearer {}'.format(self.access_token)})

        if response.status_code != 200:
            if 'Cannot stitch empty collection' in response.text:
                raise LookupError('Image not available')

            raise RuntimeError(response.text)

        # Load the image from memory (streaming)

        with rasterio.io.MemoryFile(response.raw) as memfile:
            with memfile.open() as image:
                geometry = cropsar._crs.geometry_to_crs(geometry,
                                                        crs,
                                                        image.crs)

                if not mask_exterior:
                    # Crop to the bounding box of the parcel

                    x0 = geometry.bounds[0]
                    y0 = geometry.bounds[1]
                    x1 = geometry.bounds[2]
                    y1 = geometry.bounds[3]

                    geometry = shapely.geometry.box(x0, y0, x1, y1)

                shapes = [shapely.geometry.mapping(geometry)]

                data, tf = rasterio.mask.mask(dataset=image,
                                              shapes=shapes,
                                              crop=True,
                                              all_touched=True)

                extent = rasterio.plot.plotting_extent(data[0], tf)

                # And return the cropped image data

                return {'data': data,
                        'extent': extent,
                        'crs': image.crs,
                        'transform': tf,
                        'date': date}

    # Builds a process_graph as expected by the OpenEO service for
    # downloading image data for the specified layer, parcel and date.

    def _build_graph(self, layer, geometry, crs, date, bands):

        start = datetime.datetime.strptime(date, '%Y-%m-%d')
        end   = start + datetime.timedelta(days=1)

        # Get bounding box coordinates

        x0 = geometry.bounds[0]
        y0 = geometry.bounds[1]
        x1 = geometry.bounds[2]
        y1 = geometry.bounds[3]

        # Creates the process graph

        graph = {
            'process_graph': {
                'loadcollection1': {
                    'process_id': 'load_collection',
                    'arguments': {
                        'id': layer,
                        'bands': bands,
                        'spatial_extent': {
                            'west': x0,
                            'east': x1,
                            'north': y1,
                            'south': y0,
                            'crs': str(crs)
                        },
                        'temporal_extent': [
                            start.strftime('%Y-%m-%d'),
                            end.strftime('%Y-%m-%d'),
                        ]
                    },
                    'result': False
                },
                'saveresult1': {
                    'process_id': 'save_result',
                    'arguments': {
                        'data': {'from_node': 'loadcollection1'},
                        'options': {},
                        'format': 'GTIFF'
                    },
                    'result': True
                }
            }
        }

        return graph


class TimeseriesPlotDataSource(object):

    # This class is responsible for collecting all data needed by the
    # timeseries visualization

    def __init__(self, parcel_id, parcel_df, product, start_date, end_date, params, source=None):

        # Select the requested parcel by id, and convert it to lat/lon

        parcel_df = parcel_df.loc[parcel_df.fieldID == parcel_id]
        parcel_df = parcel_df.set_index(parcel_df.fieldID, drop=True)
        parcel_df = parcel_df.to_crs('epsg:4326')

        # Setup the object

        self._parcel_id = parcel_id
        self._parcel_df = parcel_df
        self._product = product
        self._start_date = start_date
        self._end_date = end_date
        self._params = params
        self._source = source

        if source == 'sentinel-hub':
            self._image_service = ImageServiceSentinelHub()
        else:
            self._image_service = ImageServiceOpenEO()

    def get_timeseries(self):

        # Collect inputs and generate CropSAR analysis

        data = cropsar.retrieve_cropsar_analysis(self._product,
                                                 self._start_date,
                                                 self._end_date,
                                                 self._parcel_df.geometry.iloc[0],
                                                 self._parcel_df.crs,
                                                 self._source,
                                                 self._params)

        return data

    def get_images(self, date_layer_tuples):

        # Returns a collection of images for the given date.
        #
        # date_layer_tuples should contain a list of (date, layer) tuples.
        #
        # The following layers are available:

        #   S2_RADIOMETRY
        #   S2_SCENECLASSIFICATION
        #   S2_FAPAR
        #   S2_NDVI
        #   S2_FCOVER
        #   S2_LAI
        #   S1_GRD_GAMMA0
        #
        # The result is a dictionary of the form:
        #
        #   {
        #     '<image>': {
        #       'data': <numpy ndarray>,
        #       'extent': (left, right, bottom, top),
        #       'crs': <CRS spec>
        #     },
        #     ...
        #   }

        # Take the largest buffer around the parcel polygon

        crs = self._parcel_df.crs

        buffer_geoms = self.get_parcel_buffer_shapes(crs)
        buffer_geoms = sorted(buffer_geoms, key=lambda g: g.area)

        geometry = buffer_geoms[-1]

        # Now retrieve the images for all layers/datasets in parallel

        with ThreadPoolExecutor(max_workers=len(date_layer_tuples)) as pool:
            futures = {layer: pool.submit(self._time_call,
                                         'Retrieving ' + layer + ' image',
                                         self._image_service.get_image,
                                         layer,
                                         geometry,
                                         crs,
                                         date)
                       for date, layer in date_layer_tuples}

            result = {}

            for date, layer in date_layer_tuples:
                try:
                    result[layer] = futures[layer].result()
                except:
                    log.exception('Failed to retrieve image for ' + layer)

            return result

    def get_source(self):
        return self._source

    def get_cropsar_product(self):
        return self._product

    def get_cropsar_model(self):
        return self._params['cropsar']['model']

    def get_parcel_info(self, crs=None):

        # Returns the parcel info as a dict

        df = self._parcel_df
        df = df.loc[df.fieldID == self._parcel_id]

        if crs is not None:
            df.geometry = df.geometry.to_crs(crs)

        return df.iloc[0].to_dict()

    def get_parcel_shape(self, crs):

        # Returns the parcel geometry

        return self._parcel_df.to_crs(crs).geometry.iloc[0]

    def get_parcel_buffer_shapes(self, crs):

        # Returns the parcel geometry with inward/outward buffers

        g = self.get_parcel_shape(crs)

        buffer_geoms = [
            cropsar.prepare_s1_data_geometry(g, self._params, crs),
            cropsar.prepare_s2_data_geometry(g, self._params, crs),
            *cropsar.prepare_s2_scene_geometries(g, self._params, crs).values()
        ]

        return buffer_geoms

    def _time_call(self, logname, fun, *args, **kwargs):

        # Logs the running time of a function call

        log.debug('{} started'.format(logname))
        t0 = time.time()
        try:
            return fun(*args, **kwargs)
        finally:
            t1 = time.time()
            log.info('{} took {:0.2f}s'.format(logname, t1-t0))


class BackgroundPlotDataCollector(object):
    def __init__(self, fig, interval=1000):

        self._work_queue = queue.Queue()
        self._notify_queue = queue.Queue()
        self._thread = threading.Thread(target=self._thread_fun, daemon=True)
        self._thread.start()

        self._timer = fig.canvas.new_timer(interval=interval)
        self._timer.add_callback(self._timer_fun)
        self._timer.start()

        fig.canvas.draw()

    def schedule(self, fun, args, notify):
        self._work_queue.put((fun, args, notify))

    def _thread_fun(self):
        while True:
            fun, args, notify = self._work_queue.get()
            try:
                try:
                   self._notify_queue.put((notify, fun(*args)))
                except Exception as e:
                    self._notify_queue.put((notify, e))
            finally:
                self._work_queue.task_done()

    def _timer_fun(self):
        try:
            while True:
                notify, result = self._notify_queue.get_nowait()
                try:
                    notify(result)
                except Exception as e:
                    traceback.print_exc()
                finally:
                    self._notify_queue.task_done()
        except queue.Empty:
            pass


##############################################################################
#
# TIMESERIES VISUALIZATION
#
##############################################################################

class TimeseriesPlot(object):

    # This class creates an interactive timeseries visualization

    def __init__(self, fig, data_source):

        # Initializes the plot using a figure and a data source.  The
        # data source provides access to the timeseries data and
        # corresponding images

        self._data_source = data_source
        self._fig = fig
        self._ts = None
        self._im = None
        self._dates = None

        # These variables are used for handling plot interaction

        self._show_ts = [tk.StringVar(value='cropsar'),
                         tk.StringVar(value='vv,vh')]
        self._show_im = tk.StringVar(value='s2')
        self._show_margins = tk.BooleanVar(value=False)
        self._show_legends = tk.BooleanVar(value=True)

        self._context_dates = None

        # Setup the plotting axes

        self._setup_figure()

        # Start a background thread

        self._data_collector = BackgroundPlotDataCollector(fig, 1000)

        # Start retrieving the timeseries data in the background

        self._schedule_timeseries_update()

    def _schedule_timeseries_update(self):

        # Starts retrieving timeseries data in the background thread

        # Invalidate the current timeseries data (->Loading...)

        self._ts = None

        # This calls data_source.get_timeseries in the background thread,
        # with no arguments,
        # and calls _on_timeseries_update when ready

        self._data_collector.schedule(self._data_source.get_timeseries,
                                      (),
                                      self._on_timeseries_update)

    def _schedule_images_update(self):

        # Starts retrieving images in the background thread

        # Invalidate the current images (->Loading...)

        self._im = None

        # Determine which images to load

        s1_date = self._dates['s1'].strftime('%Y-%m-%d')
        s2_date = self._dates['s2'].strftime('%Y-%m-%d')

        product = self._data_source.get_cropsar_product()

        date_layer_tuples = [(s2_date, product),
                             (s2_date, 'S2_RADIOMETRY'),
                             (s2_date, 'S2_SCENECLASSIFICATION'),
                             (s1_date, 'S1_GRD_GAMMA0')]

        # This calls data_source.get_images in the background thread,
        # with a date and a list of images as arguments,
        # and calls _on_images_update when ready

        self._data_collector.schedule(self._data_source.get_images,
                                      (date_layer_tuples,),
                                      self._on_images_update)

    def _on_timeseries_update(self, result):

        # This function is called when the background thread is
        # ready retrieving timeseries data

        # Store the result and redraw the plots

        self._ts = result
        self._update_figure()

        # If the result is an exception, raise it in the current thread

        if isinstance(result, Exception):
            raise RuntimeError from result

    def _on_images_update(self, result):

        # This function is called when the background thread is
        # ready retrieving images

        # Store the result and redraw the plots

        self._im = result
        self._update_figure()

        # If the result is an exception, raise it in the current thread

        if isinstance(result, Exception):
            raise RuntimeError from result

    def _select_date(self, date):

        # Tries to select the specified date and loads images.

        # Find the S1 and S2 date in the currently loaded timeseries
        # that are closest to the given date.

        dates = self._nearest_timeseries_dates(date)

        # Skip if they are identical to the currently active dates

        if dates != self._dates:

            # Update the current state

            self._dates = dates

            # Start reloading images in the background

            if self._dates is not None:
                self._schedule_images_update()

    def _export_ts_json(self, filename):

        # Note: based on the service.api code for conversion to json...

        def _df_to_dict(df):
            return df.to_dict(orient='index')

        def _ts_df_to_dict(df):
            df.index = df.index.strftime('%Y-%m-%d')
            for col in df.columns:
                if pd.api.types.is_float_dtype(df[col]):
                    df[col] = df[col].where(df[col].notna(), 'NaN')

            return _df_to_dict(df)

        def _update(fun, d, key):
            d[key] = fun(d[key])

        data = copy.deepcopy(self._ts)

        _update(_ts_df_to_dict, data, 'cropsar')
        _update(_ts_df_to_dict, data, 'whittaker')
        _update(_ts_df_to_dict, data['clean'], 's1-data')
        _update(_ts_df_to_dict, data['clean'], 's2-data')
        _update(_ts_df_to_dict, data['sources'], 's1-data')
        _update(_ts_df_to_dict, data['sources'], 's2-data')

        for b in data['sources']['s2-scenes']:
            _update(_ts_df_to_dict, data['sources']['s2-scenes'], b)

        _update(_df_to_dict, data['clean']['.metadata'], 'flags')
        _update(_df_to_dict, data['sources']['.metadata'], 'scene-classes')

        with open(filename, 'w') as fp:
            json.dump(data, fp, indent=4)

    def _export_ts_csv(self, filename):

        # Exports the currently loaded timeseries data to CSV

        s1_product = self._ts['sources']['.metadata']['s1-product'].lower()
        s2_product = self._ts['sources']['.metadata']['s2-product'].lower()

        s1 = self._ts['clean']['s1-data']
        s2 = self._ts['clean']['s2-data']
        cropsar = self._ts['cropsar']
        whittaker = self._ts['whittaker']

        # Rename columns for export

        s1 = s1.rename(columns={'vv': s1_product + '_vv',
                                'vh': s1_product + '_vh',
                                'angle': s1_product + '_angle'})

        s2 = s2.rename(columns={'data': s2_product + '_orign',
                                'clean': s2_product + '_clean',
                                'flag': s2_product + '_flag'})

        cropsar = cropsar.rename(columns={'q10': 'cropsar_q10',
                                          'q50': 'cropsar_q50',
                                          'q90': 'cropsar_q90'})

        whittaker = whittaker.rename(columns={'smooth': 'whittaker'})

        # Combine columns into one DataFrame

        df = pd.concat([cropsar, whittaker, s1, s2], axis=1, sort=True)

        # And export to the specified filename

        df.to_csv(filename, date_format='%Y-%m-%d')

    def _export_ts(self, filename):
        
        if (filename.lower().endswith('.json')):
            self._export_ts_json(filename)
        else:
            self._export_ts_csv(filename)

    def _export_shp(self, filename):

        info = self._data_source.get_parcel_info(crs='epsg:4326')

        df = pd.DataFrame.from_records([info], index=[info['fieldID']])
        df = gpd.GeoDataFrame(df, crs='epsg:4326')

        if filename.lower().endswith('.shp'):
            df.to_file(filename)
        else:
            with open(filename, 'w') as fp:
                fp.write(df.to_json(indent=4))


    def _view_image(self, type='rgb'):

        # Displays the specified image in a separate window

        info = self._data_source.get_parcel_info()

        # Create a new figure window

        frame = FigureFrame(self._fig.canvas.get_tk_widget())
        frame.wm_title('Parcel Inspector: {}'.format(info['fieldID']))
        frame.geometry('800x600')

        # Setup the figure and axes

        if 'croptype' in info:
            frame.fig.suptitle('{} (croptype {})'.format(info['fieldID'],
                                                         info['croptype']))
        else:
            frame.fig.suptitle(info['fieldID'])

        ax = frame.fig.add_subplot(1, 1, 1)

        # And finally draw the image

        self._draw_image(ax, type)

    def _show_context_menu(self, ax, x, y, date):

        nearest_dates = self._nearest_timeseries_dates(date)

        # Highlight the timeseries date to which the menu applies

        self._context_dates = nearest_dates
        self._update_figure()

        # Build the context menu

        menu = tk.Menu(self._fig.canvas.get_tk_widget(), tearoff=False)

        # Add menu items specific to the timeseries axes

        if ax in self._ax_ts:

            index = self._ax_ts.index(ax)

            # Add 'Load images' if a date was selected

            if date is not None:

                if self._show_ts[index].get() == 'cropsar':
                    label_date = nearest_dates['s2']
                else:
                    label_date = nearest_dates['s1']

                label = 'Load images for {:%Y-%m-%d}'.format(label_date)

                command = lambda: self._select_date(date)

                # Disable if we're already waiting for images

                state = 'normal'

                if self._dates is not None and self._im is None:
                    state = 'disabled'

                menu.add_command(label=label,
                                 command=command,
                                 state=state)
                menu.add_separator()

            # Add 'Figure' submenu

            figure_menu = tk.Menu(menu, tearoff=False)

            figure_menu.add_radiobutton(label='CropSAR',
                                        variable=self._show_ts[index],
                                        value='cropsar')
            figure_menu.add_radiobutton(label='VV  VH',
                                        variable=self._show_ts[index],
                                        value='vv,vh')
            figure_menu.add_radiobutton(label='RVI',
                                        variable=self._show_ts[index],
                                        value='rvi')

            menu.add_cascade(label='Figure', menu=figure_menu)

        # Add menu items specific to the images axes

        if ax in self._ax_im:

            product = self._data_source.get_cropsar_product()

            # If images are available, add a 'Show in new window' item

            if self._im:
                index = self._ax_im.index(ax)

                if self._show_im.get() == 's1':
                    images = ['vv', 'vh', 'vh/vv', 'rvi']
                else:
                    images = ['rgb', 'nir', '<product>', 'sccl']

                command = lambda: self._view_image(images[index])

                menu.add_command(label='Show in new window', command=command)
                menu.add_separator()

            # Add 'Images' submenu

            images_menu = tk.Menu(menu, tearoff=False)

            images_menu.add_radiobutton(label='RGB  NIR  ' + product.upper() + '  SCCL',
                                        variable=self._show_im,
                                        value='s2')
            images_menu.add_radiobutton(label='VV  VH  VH/VV  RVI',
                                        variable=self._show_im,
                                        value='s1')

            menu.add_cascade(label='Images', menu=images_menu)

        # Add menu items not specific to any axes

        options_menu = tk.Menu(menu, tearoff=False)

        options_menu.add_checkbutton(label='Show legends',
                                     variable=self._show_legends,
                                     onvalue=True,
                                     offvalue=False)

        options_menu.add_checkbutton(label='Show margins',
                                     variable=self._show_margins,
                                     onvalue=True,
                                     offvalue=False)

        menu.add_cascade(label='Options', menu=options_menu)

        menu.add_separator()

        # Add a 'Export timeseries data' item

        def show_save_ts_as():
            filename = tk.filedialog.asksaveasfilename(
                parent=self._fig.canvas.get_tk_widget(),
                title='Export timeseries data',
                filetypes=[('JSON files', '*.json'),
                           ('CSV files', '*.csv'),
                           ('All files',  '*.*')],
                defaultextension='.csv')

            if filename:
                self._export_ts(filename)

        menu.add_command(label='Export timeseries data...',
                         command=show_save_ts_as,
                         state='disabled' if self._ts is None else 'normal')

        # Add a 'Export parcel geometry' item

        def show_save_shp_as():
            filename = tk.filedialog.asksaveasfilename(
                parent=self._fig.canvas.get_tk_widget(),
                title='Export parcel geometry',
                filetypes=[('GeoJSON files',   '*.geojson;*.json'),
                           ('ESRI Shapefiles', '*.shp'),
                           ('All files',       '*.*')],
                defaultextension='.geojson')

            if filename:
                self._export_shp(filename)

        menu.add_command(label='Export parcel geometry...',
                         command=show_save_shp_as)

        # Show the menu relative to its parent window

        win_x = self._fig.canvas.get_tk_widget().winfo_rootx()
        win_y = self._fig.canvas.get_tk_widget().winfo_rooty()
        win_h = self._fig.canvas.get_tk_widget().winfo_height()

        menu.tk_popup(win_x + x, win_y - y + win_h)

        # Explicitly process events.  We want to handle menu events
        # and trigger callback functions before returning!

        self._fig.canvas.get_tk_widget().update()

        # Remove timeseries date highlighting

        self._context_dates = None
        self._update_figure()

    def _on_button_press(self, event):

        # If a double-click happened in one of the timeseries axes,
        # update the currently selected date and reload images

        if event.button == 1 and event.dblclick:
            if event.inaxes in self._ax_ts:

                # Allow only if we're not already waiting for images

                if self._dates is None or self._im is not None:

                    # Get the date at the mouse cursor

                    try:
                        date = matplotlib.dates.num2date(event.xdata)
                        date = date.replace(tzinfo=None)
                    except ValueError:
                        date = None

                    # Select the new date and update the plot

                    if date is not None:

                        # If a user clicked on the CropSAR plot, we want to
                        # use the nearest S2 date, and the S1 date nearest to
                        # _that_ date.  Otherwise if the user specifically
                        # clicked in a S1 plot, we want the S1 and S2 dates
                        # closest to the clicked point.

                        if event.inaxes in self._ax_ts:
                            index = self._ax_ts.index(event.inaxes)

                            if self._show_ts[index].get() == 'cropsar':
                                dates = self._nearest_timeseries_dates(date)
                                date = dates['s2']

                        self._select_date(date)
                        self._update_figure()

    def _on_button_release(self, event):

        # If a right-click in any of the axes, show the context menu

        if event.button == 3:

            # If in one of the timeseries axes, get the date at
            # the mouse cursor

            if event.inaxes in self._ax_ts:
                try:
                    date = matplotlib.dates.num2date(event.xdata)
                    date = date.replace(tzinfo=None)
                except ValueError:
                    date = None
            else:
                date = None

            if date is not None:

                # If a user clicked on the CropSAR plot, we want to
                # use the nearest S2 date, and the S1 date nearest to
                # _that_ date.  Otherwise if the user specifically
                # clicked in a S1 plot, we want the S1 and S2 dates
                # closest to the clicked point.

                if event.inaxes in self._ax_ts:
                    index = self._ax_ts.index(event.inaxes)

                    if self._show_ts[index].get() == 'cropsar':
                        dates = self._nearest_timeseries_dates(date)
                        date = dates['s2']

            # Show the context menu and block waiting for input

            self._show_context_menu(event.inaxes, event.x, event.y, date)

    def _nearest_timeseries_dates(self, date):

        # Returns the timeseries dates that are the nearest to the given date

        if date is None:
            return None
        if self._ts is None:
            return None
        if isinstance(self._ts, Exception):
            return None

        # Collect reference timeseries

        s1 = self._ts['clean']['s1-data'].dropna()
        s2 = self._ts['clean']['s2-data'].data.dropna() # uncleaned data

        # Find the nearest date for each timeseries

        return {'s1': s1.index[s1.index.get_loc(date, method='nearest')],
                's2': s2.index[s2.index.get_loc(date, method='nearest')]}

    def _on_resize(self, event):

        # Recalculate layout when the plot window is resized

        self._fig.tight_layout(rect=[0, 0, 1, 0.95])
        self._fig.canvas.draw()

    def _setup_figure(self):

        parcel = self._data_source.get_parcel_info()

        if 'croptype' in parcel:
            self._fig.suptitle('{} (croptype {})'.format(parcel['fieldID'],
                                                         parcel['croptype']))
        else:
            self._fig.suptitle(parcel['fieldID'])

        # Creates the plotting axes

        gs0 = matplotlib.gridspec.GridSpec(3, 4)

        # Timeseries axes

        ax_ts0 = self._fig.add_subplot(gs0[0, 0:4])
        ax_ts1 = self._fig.add_subplot(gs0[1, 0:4], sharex=ax_ts0)

        self._ax_ts = [ax_ts0, ax_ts1]

        # Image axes

        ax_im0 = self._fig.add_subplot(gs0[2, 0])
        ax_im1 = self._fig.add_subplot(gs0[2, 1], sharex=ax_im0, sharey=ax_im0)
        ax_im2 = self._fig.add_subplot(gs0[2, 2], sharex=ax_im0, sharey=ax_im0)
        ax_im3 = self._fig.add_subplot(gs0[2, 3], sharex=ax_im0, sharey=ax_im0)

        self._ax_im = [ax_im0, ax_im1, ax_im2, ax_im3]

        # Draw the initial plots on these axes

        self._update_figure()

        # Connect event handlers

        self._fig.canvas.mpl_connect('button_press_event', self._on_button_press)
        self._fig.canvas.mpl_connect('button_release_event', self._on_button_release)
        self._fig.canvas.mpl_connect('resize_event', self._on_resize)

    def _update_figure(self):

        # Redraws the plotting axes using the currently loaded
        # timeseries and images data

        # Timeseries axes

        self._draw_timeseries(self._ax_ts[0], self._show_ts[0].get())
        self._draw_timeseries(self._ax_ts[1], self._show_ts[1].get())

        # Image axes

        if self._show_im.get() == 's1':
            self._draw_image(self._ax_im[0], 'vv')
            self._draw_image(self._ax_im[1], 'vh')
            self._draw_image(self._ax_im[2], 'vh/vv')
            self._draw_image(self._ax_im[3], 'rvi')
        else:
            self._draw_image(self._ax_im[0], 'rgb')
            self._draw_image(self._ax_im[1], 'nir')
            self._draw_image(self._ax_im[2], '<product>')
            self._draw_image(self._ax_im[3], 'sccl')

            if self._show_legends.get():
                self._draw_image_s2_sccl_legend(self._ax_im[3])

        # Recalculate the plotting axes layout and update canvas

        self._fig.tight_layout(rect=[0, 0, 1, 0.95])
        self._fig.canvas.draw()

    def _draw_text(self, ax, text):
        ax.text(0.5, 0.5, text,
                horizontalalignment='center',
                verticalalignment='center',
                transform=ax.transAxes)

    def _get_ts(self):

        # If we don't have any data yet, it is currently loading

        if self._ts is None:
            return None, 'Loading...'

        # Check for loading errors

        if isinstance(self._ts, Exception):
            return None, 'N/A'

        return self._ts, ''

    def _draw_timeseries(self, ax, type):
        if type == 'cropsar':
            self._draw_timeseries_s2_cropsar(ax, *self._get_ts())
        elif type == 'vv,vh':
            self._draw_timeseries_s1_vv_vh(ax, *self._get_ts())
        elif type == 'rvi':
            self._draw_timeseries_s1_rvi(ax, *self._get_ts())

    def _draw_timeseries_s2_cropsar(self, ax, ts, alt_text='N/A'):

        # Draws Sentinel-2 timeseries data on the specified axis

        ax.clear()
        ax.set_title('Sentinel-2 timeseries')

        if ts is None:
            self._draw_text(ax, alt_text)
            ax.xaxis.set_ticks([])
            ax.yaxis.set_ticks([])
            return

        # Collect the timeseries data returned by the data source

        model = self._data_source.get_cropsar_model()
        source = self._data_source.get_source()

        product = ts['sources']['.metadata']['s2-product']

        series_label = re.sub('^S2_', '', product)

        s2 = ts['clean']['s2-data']
        cropsar = ts['cropsar']
        whittaker = ts['whittaker']

        # Extract the original FAPAR points that were removed by the
        # cleaning algorithm, and classify them based on the flags data

        removed_nar = s2.loc[s2.flag == 21].data
        removed_ext = s2.loc[s2.flag == 22].data
        removed_min = s2.loc[s2.flag == 51].data

        # Add product and source info to the title

        ax.set_title('{}: {} ({})'.format(ax.get_title(), product, source))

        # Plot the various timeseries curves

        ax.plot_date(cropsar.q50.index, cropsar.q50.values,
                     'C0-', label='CropSAR q50 ({})'.format(model))

        ax.plot_date(whittaker.smooth.index, whittaker.smooth.values,
                     'C0:', label='{} (whittaker)'.format(series_label))

        ax.plot_date(s2.clean.index, s2.clean.values,
                     'C0.', mew=1, label=series_label)

        ax.plot_date(removed_nar.index, removed_nar.values,
                     'C2+', mew=1,
                     label='{} removed (narrowed field)'.format(series_label))

        ax.plot_date(removed_ext.index, removed_ext.values,
                     'C3+', mew=1,
                     label='{} removed (extended field)'.format(series_label))

        ax.plot_date(removed_min.index, removed_min.values,
                     'C4+', mew=1,
                     label='{} removed (local minima)'.format(series_label))

        # Draw CropSAR uncertainty 'range'

        ax.fill_between(cropsar.index,
                        cropsar.q10.values.flatten(),
                        cropsar.q90.values.flatten(),
                        color='C0', alpha=0.3,
                        label='CropSAR q10-q90 ({})'.format(model))

        # Configure timeseries axes labels, ticks, ...

        self._configure_timeseries_axes(ax)

        # Expect values to be in range [0, 1] but add some margin

        ax.set_ylim(-0.2, 1.2)
        ax.set_yticks([0.0, 0.2, 0.4, 0.6, 0.8, 1.0])

        # And finally add margins, dates markers and legend

        self._draw_timeseries_margins(ax, ts)
        self._draw_timeseries_dates(ax, s2.data)

        if self._show_legends.get():
            ax.legend(loc='upper left', bbox_to_anchor=(1.0, 1.0))

    def _draw_timeseries_s1_vv_vh(self, ax, ts, alt_text):

        # Draws Sentinel-2 timeseries data on the specified axis

        ax.clear()
        ax.set_title('Sentinel-1 timeseries')

        if ts is None:
            self._draw_text(ax, alt_text)
            ax.xaxis.set_ticks([])
            ax.yaxis.set_ticks([])
            return

        # Collect the timeseries data returned by the data source

        source = self._data_source.get_source()

        product = self._ts['sources']['.metadata']['s1-product']

        s1 = self._ts['clean']['s1-data']
        s1_vv_sm = self._smooth(s1.vv, in_db=True)
        s1_vh_sm = self._smooth(s1.vh, in_db=True)

        # Add product and source info to the title

        ax.set_title('{}: {} ({})'.format(ax.get_title(), product, source))

        # Plot the various timeseries curves

        ax.plot_date(s1.vv.index, s1.vv.values, 'C0.', mew=0.3, label='VV')
        ax.plot_date(s1_vv_sm.index, s1_vv_sm.values, 'C0-', label='VV (smoothed)')
        ax.plot_date(s1.vh.index, s1.vh.values, 'C3.', mew=0.3, label='VH')
        ax.plot_date(s1_vh_sm.index, s1_vh_sm.values, 'C3-', label='VH (smoothed)')

        # Configure timeseries axes labels, ticks, ...

        self._configure_timeseries_axes(ax)

        # Expect values to be in range [-25, 0] but add some margin

        ax.set_ylim(-30.0, 5.0)
        ax.set_yticks([-25.0, -20.0, -15.0, -10.0, -5.0, 0.0])

        # And finally add margins, dates markers and legend

        self._draw_timeseries_margins(ax, ts)
        self._draw_timeseries_dates(ax, [s1.vv, s1.vh])

        if self._show_legends.get():
            ax.legend(loc='upper left', bbox_to_anchor=(1.0, 1.0))

    def _draw_timeseries_s1_rvi(self, ax, ts, alt_text):

        # Draws Sentinel-2 timeseries data on the specified axis

        ax.clear()
        ax.set_title('Sentinel-1 timeseries')

        if ts is None:
            self._draw_text(ax, alt_text)
            ax.xaxis.set_ticks([])
            ax.yaxis.set_ticks([])
            return

        # Collect the timeseries data returned by the data source

        source = self._data_source.get_source()

        product = self._ts['sources']['.metadata']['s1-product']

        s1 = self._ts['clean']['s1-data'].copy()

        s1_lin_vv = np.power(10, s1.vv/10)
        s1_lin_vh = np.power(10, s1.vh/10)

        s1_rvi = (4.0 * s1_lin_vh) / (s1_lin_vv + s1_lin_vh)
        s1_rvi_sm = self._smooth(s1_rvi)

        # Add product and source info to the title

        ax.set_title('{}: {} ({})'.format(ax.get_title(), product, source))

        # Plot the various timeseries curves

        ax.plot_date(s1_rvi.index, s1_rvi.values, 'C0.', mew=0.3, label='RVI')
        ax.plot_date(s1_rvi_sm.index, s1_rvi_sm.values, 'C0-', label='RVI (smoothed)')

        # Configure timeseries axes labels, ticks, ...

        self._configure_timeseries_axes(ax)

        # Expect values to be in range [0, 1] but add some margin

        ax.set_ylim(0.0, 1.6)
        ax.set_yticks([0.2, 0.4, 0.6, 0.8, 1.0, 1.2, 1.4])

        # And finally add margins, dates markers and legend

        self._draw_timeseries_margins(ax, ts)
        self._draw_timeseries_dates(ax, s1_rvi)

        if self._show_legends.get():
            ax.legend(loc='upper left', bbox_to_anchor=(1.0, 1.0))

    def _configure_timeseries_axes(self, ax):

        # Show a label only every 3 months, but gridlines every month

        ax.xaxis.set_major_locator(matplotlib.dates.MonthLocator(bymonth=[1,4,7,10]))
        ax.xaxis.set_major_formatter(matplotlib.dates.DateFormatter('%Y-%m'))
        ax.xaxis.set_minor_locator(matplotlib.dates.MonthLocator())
        ax.format_coord = self._format_ts_coord
        ax.grid(True, which='minor')
        ax.grid(True)

    def _draw_timeseries_dates(self, ax, series_list):

        # This marks the currently selected date, as well as the nearest
        # samples in the specified timeseries.

        if isinstance(series_list, pd.Series):
            series_list = [series_list]

        # Find out what type of timeseries we're plotting

        index = self._ax_ts.index(ax)

        type = self._show_ts[index].get()

        # Draw marker lines, these always show the Sentinel-2 date

        if self._dates is not None:
            date = self._dates['s2']
            ax.axvline(date, linestyle=':', color='C1')

        if self._context_dates is not None:
            date = self._context_dates['s2']
            ax.axvline(date, alpha=0.6, linestyle=':', color='C1')

        # Draw text, this matches the Sentinel-2 date of the marker line

        if self._dates is not None:
            date = self._dates['s2']

            y_min, y_max = ax.get_ylim()
            y_pos = y_max - 0.025*(y_max - y_min)

            ax.text(x=date,
                    y=y_pos,
                    s=' {}'.format(date.strftime('%Y-%m-%d')),
                    color='C1',
                    horizontalalignment='left',
                    verticalalignment='top')

        # Draw marker bullets, these match the specified timeseries

        for series in series_list:
            if self._dates is not None:
                if type == 'cropsar':
                    date = self._dates['s2']
                else:
                    date = self._dates['s1']

                series = series.dropna()

                if date in series.index:
                    val = series.loc[date]

                    ax.plot_date(date, val,
                                 marker='o',
                                 markerfacecolor='none',
                                 markeredgecolor='C1',
                                 markersize=10)

            if self._context_dates is not None:
                if type == 'cropsar':
                    date = self._context_dates['s2']
                else:
                    date = self._context_dates['s1']

                series = series.dropna()

                if date in series.index:
                    val = series.loc[date]

                    ax.plot_date(date, val,
                                 alpha=0.6,
                                 marker='o',
                                 markerfacecolor='none',
                                 markeredgecolor='C1',
                                 markersize=10)

    def _draw_timeseries_margins(self, ax, ts):

        # Optionally crop the Sentinel-2 timeseries to the CropSAR timeseries
        # date range (as normally it includes some extra margins)

        s2 = ts['clean']['s2-data']

        cropsar = ts['cropsar']

        if not self._show_margins.get():
            ax.set_xlim(cropsar.index[0], cropsar.index[-1])
        else:
            ax.set_xlim(s2.index[0], s2.index[-1])

            x0, x3 = cropsar.index[0], cropsar.index[-1]
            x1, x2 = s2.index[0], s2.index[-1]

            if (x0, x3) != (x1, x2):
                y0, y1 = ax.get_ylim()

                ax.fill_between([x0, x1], [y0, y0], [y1, y1],
                                color='C1', alpha=0.1,
                                label='CropSAR margin')
                ax.fill_between([x2, x3], [y0, y0], [y1, y1],
                                color='C1', alpha=0.1)

    def _format_ts_coord(self, x, y):

        # Formats a timeseries x,y coordinate as 'x=yyyy-mm-dd y=val'

        date = matplotlib.dates.num2date(x)
        date = date.replace(tzinfo=None)

        return 'x={} y={:0.4f}'.format(date.strftime('%Y-%m-%d'), y)

    def _smooth(self, series, window=7, in_db=False):

        # If the series is in dB, we first convert it to linear units
        # before calculating the rolling mean, then convert it back
        # again

        if in_db:
            series = np.power(10.0, series / 10.0)

        series = series.replace((np.inf, -np.inf), np.nan)
        series = series.dropna()
        series = series.rolling(window=window, center=True).mean()

        if in_db:
            series = 10.0 * np.log10(series)

        return series

    def _get_image(self, type):

        # If we don't have a date yet, just show empty

        if self._dates is None:
            return None, ''

        # If we have a date, but no images, they are currently loading

        if self._im is None:
            return None, 'Loading...'

        # Check for loading errors

        if isinstance(self._im, Exception):
            return None, 'N/A'

        # Check for availability

        if type not in self._im:
            return None, 'N/A'

        return self._im[type], type

    def _draw_image(self, ax, type):

        product = self._data_source.get_cropsar_product()

        if type == '<product>':
            if product.startswith('S2_FAPAR'):
                self._draw_image_s2_fapar(ax, *self._get_image(product))
            elif product.startswith('S2_FCOVER'):
                self._draw_image_s2_fcover(ax, *self._get_image(product))
        elif type == 'rgb':
            self._draw_image_s2_rgb(ax, *self._get_image('S2_RADIOMETRY'))
        elif type == 'nir':
            self._draw_image_s2_nir(ax, *self._get_image('S2_RADIOMETRY'))
        elif type == 'sccl':
            self._draw_image_s2_sccl(ax, *self._get_image('S2_SCENECLASSIFICATION'))
        elif type == 'vv':
            self._draw_image_s1_vv(ax, *self._get_image('S1_GRD_GAMMA0'))
        elif type == 'vh':
            self._draw_image_s1_vh(ax, *self._get_image('S1_GRD_GAMMA0'))
        elif type == 'vh/vv':
            self._draw_image_s1_vh_vv(ax, *self._get_image('S1_GRD_GAMMA0'))
        elif type == 'rvi':
            self._draw_image_s1_rvi(ax, *self._get_image('S1_GRD_GAMMA0'))

    def _draw_image_s2_rgb(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('RGB')

        if image is None:
            self._draw_text(ax, alt_text)
        else:
            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            rgb = np.stack((np.clip(data[2] / 2000.0, 0.0, 1.0),
                            np.clip(data[1] / 2000.0, 0.0, 1.0),
                            np.clip(data[0] / 2000.0, 0.0, 1.0)),
                           axis=2)

            ax.imshow(rgb, extent=ext)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s2_nir(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('NIR')

        if image is None:
            self._draw_text(ax, alt_text)
        else:
            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            nir = np.stack((np.clip(data[3] / 4000.0, 0.0, 1.0),
                            np.clip(data[2] / 2000.0, 0.0, 1.0),
                            np.clip(data[1] / 2000.0, 0.0, 1.0)),
                           axis=2)

            ax.imshow(nir, extent=ext)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s2_fapar(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('FAPAR')

        if image is None:
            self._draw_text(ax, alt_text)
        else:
            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            cmap, norm = self._fapar_colors()

            fapar = data[0] / 200.0
            fapar[fapar > 1.0] = np.nan

            ax.imshow(fapar, extent=ext, cmap=cmap, norm=norm)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s2_fcover(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('FCOVER')

        if image is None:
            self._draw_text(ax, alt_text)
        else:
            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            cmap, norm = self._fapar_colors()

            fcover = data[0] / 200.0
            fcover[fcover > 1.0] = np.nan

            ax.imshow(fcover, extent=ext, cmap=cmap, norm=norm)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s2_sccl(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('SCCL')

        if image is None:
            self._draw_text(ax, alt_text)
        else:

            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            cmap, norm, labels = self._sccl_colors()

            sccl = data[0]

            ax.imshow(sccl, extent=ext, cmap=cmap, norm=norm)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s2_sccl_legend(self, ax):

        from matplotlib.patches import Patch

        if self._ts is not None:

            cmap, norm, labels = self._sccl_colors()

            handles = []
            for label, rgb in labels:
                handles.append(Patch(color=rgb, label=label))

            ax.legend(handles=handles,
                      loc='upper left',
                      bbox_to_anchor=(1.0, 1.0),
                      ncol=2)

    def _draw_image_s1_vv(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('VV')

        if image is None:
            self._draw_text(ax, alt_text)
        else:

            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{} \n{}'.format(ax.get_title(), date))

            vv_lin = data[0]
            vv = 10.0 * np.log10(vv_lin)

            ax.imshow(vv, extent=ext, cmap='binary', vmin=-25.0, vmax=3.0)

            self._draw_buffers(ax, crs,)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s1_vh(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('VH')

        if image is None:
            self._draw_text(ax, alt_text)
        else:

            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            vh_lin = data[1]
            vh = 10.0 * np.log10(vh_lin)

            ax.imshow(vh, extent=ext, cmap='binary', vmin=-30.0, vmax=-2.0)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s1_vh_vv(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('VH/VV')

        if image is None:
            self._draw_text(ax, alt_text)
        else:

            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            vv_lin = data[0]
            vh_lin = data[1]

            vh_vv_lin = vh_lin / vv_lin

            vv = 10.0 * np.log10(vv_lin)
            vh = 10.0 * np.log10(vh_lin)

            def norm(x, x0, x1):
                return np.clip((x-x0) / (x1-x0), 0, 1.0)

            img = np.stack((norm(vv,      -25.0,  3.0),
                            norm(vh,      -30.0, -2.0),
                            norm(vh_vv_lin, 0.0,  0.6)),
                           axis=2)

            ax.imshow(img, extent=ext)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_image_s1_rvi(self, ax, image, alt_text='N/A'):

        ax.clear()
        ax.set_title('RVI')

        if image is None:
            self._draw_text(ax, alt_text)
        else:

            data = image['data']
            ext = image['extent']
            crs = image['crs']
            date = image['date']

            ax.set_title('{}\n{}'.format(ax.get_title(), date))

            vv_lin = data[0]
            vh_lin = data[1]

            rvi = (4.0 * vh_lin) / (vv_lin + vh_lin)

            vv = 10.0 * np.log10(vv_lin)
            vh = 10.0 * np.log10(vh_lin)

            def norm(x, x0, x1):
                return np.clip((x-x0) / (x1-x0), 0, 1.0)

            img = np.stack((norm(vv, -25.0,  3.0),
                            norm(vh, -30.0, -2.0),
                            norm(rvi,  0.2,  1.2)),
                           axis=2)

            ax.imshow(img, extent=ext)

            self._draw_buffers(ax, crs)

        ax.xaxis.set_ticks([])
        ax.yaxis.set_ticks([])
        ax.grid(False)

    def _draw_polygon(self, ax, polygon, **kwargs):
        if polygon.geom_type == 'MultiPolygon':
            for geom in polygon.geoms:
                self._draw_polygon(ax, geom, **kwargs)
        else:
            x, y = polygon.exterior.coords.xy

            ax.plot(x, y, **kwargs)

            for interior in polygon.interiors:
                x, y = interior.coords.xy

                ax.plot(x, y, **kwargs)

    def _draw_buffers(self, ax, crs):
        shape = self._data_source.get_parcel_shape(crs)

        self._draw_polygon(ax, shape, color='k', linestyle='-')

        for shape in self._data_source.get_parcel_buffer_shapes(crs):
            self._draw_polygon(ax, shape, color='k', linestyle='--')

    def _fapar_colors(self):
        colors = [(168, 80, 0),
                  (189, 124, 0),
                  (211, 167, 0),
                  (233, 211, 0),
                  (255, 255, 0),
                  (200, 222, 0),
                  (145, 189, 000),
                  (91, 157, 000),
                  (36, 124, 000),
                  (51, 102, 000),
                  (0, 0, 0)]

        colors = [(r/255, g/255, b/255) for (r, g, b) in colors]

        bounds = [0, 20, 40, 60, 80, 100, 120, 140, 160, 180, 200, 255]
        bounds = [b/200 for b in bounds]

        cmap = matplotlib.colors.ListedColormap(colors)
        norm = matplotlib.colors.BoundaryNorm(bounds, cmap.N)

        return cmap, norm

    def _sccl_colors(self):
        colors = [('no_data',        (0, 0, 0)),
                  ('sat_or_def',     (255, 0, 0)),
                  ('dark_feat/shad', (51, 51, 51)),
                  ('shadows',        (102, 51, 0)),
                  ('vegetation',     (0, 255, 0)),
                  ('bare_soils',     (255, 255, 0)),
                  ('water',          (0, 0, 255)),
                  ('cloud_low',      (128, 128, 128)),
                  ('cloud_medium',   (200, 200, 200)),
                  ('cloud_high',     (255, 255, 255)),
                  ('cirrus',         (102, 204, 255)),
                  ('snow/ice',       (255, 153, 255))]

        colors = [(s, (r/255, g/255, b/255)) for (s, (r, g, b)) in colors]

        bounds = range(len(colors)+1)

        cmap = matplotlib.colors.ListedColormap([rgb for (s, rgb) in colors])
        norm = matplotlib.colors.BoundaryNorm(bounds, cmap.N)

        return cmap, norm, colors


##############################################################################
#
# USER INTERFACE UILITY CLASSES
#
##############################################################################

class BoldLabel(ttk.Label):
    def __init__(self, master, **kwargs):
        super().__init__(master, **kwargs)

        bold_font = tk.font.Font(font=self['font'])
        bold_font['weight'] = 'bold'

        self.config(font=bold_font)


class ProgressDialog(tk.Toplevel):
    def __init__(self, master, title=None, mode='determinate', maximum=None, value=0, variable=None, phase=None, **kwargs):
        super().__init__(master, **kwargs)

        self._master = master

        if title is not None:
            self.wm_title(title)

        self.protocol('WM_DELETE_WINDOW', lambda: None)
        self.geometry('300x60')
        self.transient(master)

        self.frame = ttk.Frame(self)

        self.label = ttk.Label(self.frame)
        self.progressbar = ttk.Progressbar(self.frame , mode=mode, value=value)

        if maximum is not None:
            self.progressbar.configure(maximum=maximum)
        if phase is not None:
            self.progressbar.configure(phase=phase)
        if variable is not None:
            self.progressbar.configure(variable=variable)

        self.label.grid(row=0, column=0, sticky='new', padx=5, pady=5)
        self.progressbar.grid(row=1, column=0, sticky='new', padx=5, pady=5)
        self.frame.columnconfigure(0, weight=1)

        self.frame.pack(expand=True, fill='both', padx=10, pady=10)

        self.center()

    def center(self):

        p_w = self._master.winfo_width()
        p_h = self._master.winfo_height()
        p_x = self._master.winfo_rootx()
        p_y = self._master.winfo_rooty()
        w = 400
        h = 80

        x = p_x + (p_w//2 - w//2)
        y = p_y + (p_h//2 - h//2)

        self.geometry('{}x{}+{}+{}'.format(w, h, x, y))

    def set_text(self, text):
        self.label.configure(text=text)
        self.update_idletasks()

    def set_maximum(self, maximum):
        self.progressbar.configure(maximum=maximum)

    def set_value(self, value):
        self.progressbar.configure(value=value)

    def start(self, interval=None):
        self.progressbar.start(interval=interval)

    def step(self, amount=None):
        self.progressbar.step(amount=amount)

    def stop(self):
        self.progressbar.stop()

    def __enter__(self):
        self.grab_set()
        self.focus_set()
        self.update()

        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.grab_release()
        self.stop()
        self.destroy()


class FigureFrame(tk.Toplevel):
    def __init__(self, parent, **kwargs):

        from matplotlib.figure import Figure
        from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
        from matplotlib.backends.backend_tkagg import NavigationToolbar2Tk

        super().__init__(master=parent, **kwargs)

        self.fig = Figure()

        self._canvas = FigureCanvasTkAgg(self.fig, master=self)
        self._canvas.draw()
        self._canvas.get_tk_widget().pack(side='top', expand=True, fill='both')

        self._toolbar = NavigationToolbar2Tk(self._canvas, self)
        self._toolbar.update()
        self._canvas._tkcanvas.pack(side='top', expand=True, fill='both')

        self._canvas.mpl_connect('key_press_event', self._on_key_press)

        self._attached = []

    def _on_key_press(self, event):

        from matplotlib.backend_bases import key_press_handler

        key_press_handler(event, self._canvas, self._toolbar)

    def keep_alive(self, obj):
        self._attached.append(obj)


class ScrollableFrame(ttk.Frame):
    def __init__(self, parent, **kwargs):
        super().__init__(parent, **kwargs)

        self._canvas = tk.Canvas(self, borderwidth=0, highlightthickness=0)
        self._scroll = ttk.Scrollbar(self, orient='vertical', command=self._canvas.yview)
        self._frame = ttk.Frame(self._canvas)

        self._canvas.config(yscrollcommand=self._scroll.set)
        self._canvas.bind('<Configure>', self._on_canvas_configure)

        self._canvas.grid(row=0, column=0, sticky='nsew')
        self._scroll.grid(row=0, column=1, sticky='ns')

        self.columnconfigure(0, weight=1)
        self.columnconfigure(1, weight=0)
        self.rowconfigure(0, weight=1)

        self._frame_item = self._canvas.create_window((0, 0), window=self._frame, anchor='nw')

        self._frame.bind('<Enter>', self._on_frame_enter)
        self._frame.bind('<Leave>', self._on_frame_leave)

    def scrolled_frame(self):
        return self._frame

    def _on_canvas_configure(self, event):
        self._canvas.itemconfig(self._frame_item, width=event.width)
        self._canvas.configure(scrollregion=self._canvas.bbox('all'))

    def _on_frame_enter(self, event):
        self._canvas.bind_all('<MouseWheel>', self._on_mousewheel)
        self._canvas.bind_all('<Button-4>', self._on_button_4)
        self._canvas.bind_all('<Button-5>', self._on_button_5)

    def _on_frame_leave(self, event):
        self._canvas.unbind_all('<MouseWheel>')
        self._canvas.unbind_all('<Button-4>')
        self._canvas.unbind_all('<Button-5>')

    def _on_mousewheel(self, event):
        if os.name == 'nt':
            self._canvas.yview_scroll(int(-event.delta/120), 'units')
        else:
            self._canvas.yview_scroll(-event.delta, 'units')

    def _on_button_4(self, event):
        self._canvas.yview_scroll(-1, 'units')

    def _on_button_5(self, event):
        self._canvas.yview_scroll(1, 'units')



##############################################################################
#
# USER INTERFACE
#
##############################################################################

class Model(object):
    def __init__(self, app):
        self.app = app

        self._df = None

        today = datetime.datetime.now()
        a_year_ago = today.replace(year=today.year-1)

        self.shapefile = tk.StringVar(value='')
        self.parcel = tk.StringVar(value='')
        self.start_date = tk.StringVar(value=a_year_ago.strftime('%Y-%m-%d'))
        self.end_date = tk.StringVar(value=today.strftime('%Y-%m-%d'))

        self.source = tk.StringVar(value=self.get_sources()[0])

        self.cropsar_product = tk.StringVar(
            value=self.get_cropsar_products()[0])

        self.cropsar_model = tk.StringVar()
        self.cropsar_simulate_nrt = tk.BooleanVar()
        self.s1_clean_inarrowfieldbordersinmeter = tk.StringVar()
        self.s2_clean_inarrowfieldbordersinmeter = tk.StringVar()
        self.s2_clean_lstnarrowedfieldvalidscenevalues = tk.StringVar()
        self.s2_clean_inarrowedfieldminpctareavalid = tk.StringVar()
        self.s2_clean_lstiextendfieldbordersinmeter = tk.StringVar()
        self.s2_clean_lstlstextendedfieldvalidscenevalues = tk.StringVar()
        self.s2_clean_lstiextendedfieldminpctareavalid = tk.StringVar()
        self.s2_clean_localminimamaxdip = tk.StringVar()
        self.s2_clean_localminimamaxdif = tk.StringVar()
        self.s2_clean_localminimamaxgap = tk.StringVar()
        self.s2_clean_localminimamaxpas = tk.StringVar()

    def reload_params(self):

        import cropsar._generate

        params = cropsar._generate.DEFAULT_PARAMS

        self.cropsar_model.set(
            self._get_param('cropsar', 'model', params))

        self.cropsar_simulate_nrt.set(
            self._get_param('cropsar', 'simulate_nrt', params))

        self.s1_clean_inarrowfieldbordersinmeter.set(
            self._get_param('s1_clean',
                            'inarrowfieldbordersinmeter',
                            params))

        self.s2_clean_inarrowfieldbordersinmeter.set(
            self._get_param('s2_clean',
                            'inarrowfieldbordersinmeter',
                            params))

        self.s2_clean_lstnarrowedfieldvalidscenevalues.set(
            self._get_param('s2_clean',
                            'lstnarrowedfieldvalidscenevalues',
                            params))

        self.s2_clean_inarrowedfieldminpctareavalid.set(
            self._get_param('s2_clean',
                            'inarrowedfieldminpctareavalid',
                            params))

        self.s2_clean_lstiextendfieldbordersinmeter.set(
            self._get_param('s2_clean',
                            'lstiextendfieldbordersinmeter',
                            params))

        self.s2_clean_lstlstextendedfieldvalidscenevalues.set(
            self._get_param('s2_clean',
                            'lstlstextendedfieldvalidscenevalues',
                            params))

        self.s2_clean_lstiextendedfieldminpctareavalid.set(
            self._get_param('s2_clean',
                            'lstiextendedfieldminpctareavalid',
                            params))

        self.s2_clean_localminimamaxdip.set(
            self._get_param('s2_clean', 'localminimamaxdip', params))

        self.s2_clean_localminimamaxdif.set(
            self._get_param('s2_clean', 'localminimamaxdif', params))

        self.s2_clean_localminimamaxgap.set(
            self._get_param('s2_clean', 'localminimamaxgap', params))

        self.s2_clean_localminimamaxpas.set(
            self._get_param('s2_clean', 'localminimamaxpas', params))

    def load_shapefile(self, filename, progress=None, interval=0.2):

        if progress is not None:
            with ThreadPoolExecutor(max_workers=1) as pool:
                f = pool.submit(gpd.read_file, filename)

                while not f.done():
                    time.sleep(interval)
                    progress.update()

                df = f.result()
        else:
            df = gpd.read_file(filename)

        # We expect the following columns, but allow for alternative names

        cols = ['fieldID', 'croptype', 'area', 'geometry']

        alts = {'fieldID': ['id', 'CODE_OBJ', 'name', 'OBJECTID_', 'OBJECTID'],
                'croptype': ['GWSCOD_H'],
                'area': ['REF_OPP', 'GRAF_OPP'],
                'geometry': []}

        log.info('Available fields: {}'.format(df.columns))

        for col in cols:
            if col not in df.columns:
                for alt in alts[col]:
                    if alt in df.columns:
                        log.info(' Found {} as {}'.format(col, alt))
                        df.rename(columns={alt: col}, inplace=True)
                        break
            else:
                log.info(' Found {}'.format(col))

        # If we still don't have a fieldID column, simply use the first
        # non-geometry column

        if 'fieldID' not in df.columns:
            if df.columns[0] != 'geometry':
                alt = df.columns[0]
            else:
                alt = df.columns[1]

            df = df.rename(columns={alt: 'fieldID'})

        # If this column contains duplicate values, fallback to the
        # original index.  Also, make sure fieldID and croptype are
        # strings.

        if df.fieldID.is_unique:
            df['fieldID'] = df['fieldID'].astype(str)
        else:
            df['fieldID'] = df.index.astype(str)

        if 'croptype' in df.columns:
            df.croptype = df.croptype.astype(str)

        # Keep only the columns we're interested in

        df = df.filter(items=cols, axis=1)

        self._df = df

        self.shapefile.set(filename)
        self.parcel.set('')

    def get_parcel_data(self):
        return self._df

    def get_sources(self):
        return ['sentinel-hub', 'probav-mep']

    def get_cropsar_products(self):
        return ['S2_FAPAR',
                'S2_FCOVER',
                'S2_NDVI']

    def get_cropsar_models(self):
        return cropsar.get_available_model_types()

    def get_cropsar_params(self):
        return {
            'cropsar': {
                'model': self.cropsar_model.get(),
                'simulate_nrt': self.cropsar_simulate_nrt.get()
            },
            'whittaker': {
            },
            's1_clean': {
                'inarrowfieldbordersinmeter': self._get_json_var(
                    self.s1_clean_inarrowfieldbordersinmeter),
            },
            's2_clean': {
                'inarrowfieldbordersinmeter': self._get_json_var(
                    self.s2_clean_inarrowfieldbordersinmeter),
                'lstnarrowedfieldvalidscenevalues': self._get_json_var(
                    self.s2_clean_lstnarrowedfieldvalidscenevalues),
                'inarrowedfieldminpctareavalid': self._get_json_var(
                    self.s2_clean_inarrowedfieldminpctareavalid),
                'lstiextendfieldbordersinmeter': self._get_json_var(
                    self.s2_clean_lstiextendfieldbordersinmeter),
                'lstlstextendedfieldvalidscenevalues': self._get_json_var(
                    self.s2_clean_lstlstextendedfieldvalidscenevalues),
                'lstiextendedfieldminpctareavalid': self._get_json_var(
                    self.s2_clean_lstiextendedfieldminpctareavalid),
                'localminimamaxdip': self._get_json_var(
                    self.s2_clean_localminimamaxdip),
                'localminimamaxdif': self._get_json_var(
                    self.s2_clean_localminimamaxdif),
                'localminimamaxgap': self._get_json_var(
                    self.s2_clean_localminimamaxgap),
                'localminimamaxpas': self._get_json_var(
                    self.s2_clean_localminimamaxpas),
            }
        }

    def _get_json_var(self, var):
        val = var.get()

        if val:
            try:
                return json.loads(val)
            except json.decoder.JSONDecodeError:
                pass
        else:
            val = None

        return val

    def _get_param(self, section, name, params):

        val = params[section][name]

        if not isinstance(val, str):
            if val is None:
                val = ''
            else:
                val = json.dumps(val)

        return val


class GeneralParameterFrame(ttk.Frame):
    def __init__(self, parent,  model, **kwargs):
        super().__init__(parent, **kwargs)

        today = datetime.datetime.today()

        start_dates = []
        end_dates = []

        for year in range(2015, today.year+1):
            start_dates.append(datetime.datetime(year, 1, 1).strftime('%Y-%m-%d'))
            end_dates.append(datetime.datetime(year, 12, 31).strftime('%Y-%m-%d'))

        title_label = BoldLabel(self, text='General')
        source_label = ttk.Label(self, text='Data Source')
        source_combo = ttk.Combobox(self, textvariable=model.source, width=30, state='readonly', values=model.get_sources())
        start_label = ttk.Label(self, text='Start Date')
        start_combo = ttk.Combobox(self, textvariable=model.start_date, width=20, values=start_dates)
        end_label = ttk.Label(self, text='End Date')
        end_combo = ttk.Combobox(self, textvariable=model.end_date, width=20, values=end_dates)

        title_label.grid(row=0, column=0, sticky='w', padx=5, pady=5, columnspan=2)
        source_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        source_combo.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        start_label.grid(row=2, column=0, sticky='e', padx=5, pady=5)
        start_combo.grid(row=2, column=1, sticky='w', padx=5, pady=5)
        end_label.grid(row=3, column=0, sticky='e', padx=5, pady=5)
        end_combo.grid(row=3, column=1, sticky='w', padx=5, pady=5)


class CropSARParameterFrame(ttk.Frame):
    def __init__(self, parent, model, **kwargs):
        super().__init__(parent, **kwargs)

        title_label = BoldLabel(self, text='CropSAR')
        product_label = ttk.Label(self, text='Product')
        product_combo = ttk.Combobox(self, textvariable=model.cropsar_product, width=30, state='readonly', values=model.get_cropsar_products())
        model_label = ttk.Label(self, text='Model')
        model_combo = ttk.Combobox(self, textvariable=model.cropsar_model, width=30, state='readonly', values=model.get_cropsar_models())
        simulate_nrt_label = ttk.Label(self, text='Simulate NRT')
        simulate_nrt_check = ttk.Checkbutton(self, variable=model.cropsar_simulate_nrt, text='(Ignore margin after End Date)')

        title_label.grid(row=0, column=0, sticky='w', padx=5, pady=5, columnspan=2)
        product_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        product_combo.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        model_label.grid(row=2, column=0, sticky='e', padx=5, pady=5)
        model_combo.grid(row=2, column=1, sticky='w', padx=5, pady=5)
        simulate_nrt_label.grid(row=3, column=0, sticky='e', padx=5, pady=5)
        simulate_nrt_check.grid(row=3, column=1, sticky='w', padx=5, pady=5)


class S1CleaningParameterFrame(ttk.Frame):
    def __init__(self, parent, model, **kwargs):
        super().__init__(parent, **kwargs)

        title_label = BoldLabel(self, text='Sentinel-1 Cleaning')

        nar_frame = ttk.LabelFrame(self, text='Narrowed Field')

        nar_border_label = ttk.Label(nar_frame, text='Border [m]')
        nar_border_entry = ttk.Entry(nar_frame, textvariable=model.s1_clean_inarrowfieldbordersinmeter)

        nar_border_label.grid(row=0, column=0, sticky='e', padx=5, pady=5)
        nar_border_entry.grid(row=0, column=1, sticky='w', padx=5, pady=5)

        title_label.grid(row=0, column=0, sticky='w', padx=5, pady=5)
        nar_frame.grid(row=1, column=0, sticky='nsew', padx=5, pady=5)

        self.columnconfigure(0, weight=1)


class S2CleaningParameterFrame(ttk.Frame):
    def __init__(self, parent,  model, **kwargs):
        super().__init__(parent, **kwargs)

        title_label = BoldLabel(self, text='Sentinel-2 Cleaning')

        nar_frame = ttk.LabelFrame(self, text='Narrowed Field')

        nar_border_label = ttk.Label(nar_frame, text='Border [m]')
        nar_border_entry = ttk.Entry(nar_frame, textvariable=model.s2_clean_inarrowfieldbordersinmeter)
        nar_vals_label = ttk.Label(nar_frame, text='Valid Scene Values')
        nar_vals_entry = ttk.Entry(nar_frame, textvariable=model.s2_clean_lstnarrowedfieldvalidscenevalues, width=40)
        nar_area_label = ttk.Label(nar_frame, text='Min Area Valid [%]')
        nar_area_entry = ttk.Entry(nar_frame, textvariable=model.s2_clean_inarrowedfieldminpctareavalid)

        nar_border_label.grid(row=0, column=0, sticky='e', padx=5, pady=5)
        nar_border_entry.grid(row=0, column=1, sticky='w', padx=5, pady=5)
        nar_vals_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        nar_vals_entry.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        nar_area_label.grid(row=2, column=0, sticky='e', padx=5, pady=5)
        nar_area_entry.grid(row=2, column=1, sticky='w', padx=5, pady=5)

        ext_frame = ttk.LabelFrame(self, text='Extended Field')

        ext_border_label = ttk.Label(ext_frame, text='Border [m]')
        ext_border_entry = ttk.Entry(ext_frame, textvariable=model.s2_clean_lstiextendfieldbordersinmeter, width=40)
        ext_vals_label = ttk.Label(ext_frame, text='Valid Scene Values')
        ext_vals_entry = ttk.Entry(ext_frame, textvariable=model.s2_clean_lstlstextendedfieldvalidscenevalues, width=40)
        ext_area_label = ttk.Label(ext_frame, text='Min Area Valid [%]')
        ext_area_entry = ttk.Entry(ext_frame, textvariable=model.s2_clean_lstiextendedfieldminpctareavalid, width=40)

        ext_border_label.grid(row=0, column=0, sticky='e', padx=5, pady=5)
        ext_border_entry.grid(row=0, column=1, sticky='w', padx=5, pady=5)
        ext_vals_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        ext_vals_entry.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        ext_area_label.grid(row=2, column=0, sticky='e', padx=5, pady=5)
        ext_area_entry.grid(row=2, column=1, sticky='w', padx=5, pady=5)

        min_frame = ttk.LabelFrame(self, text='Local Minima')

        min_dip_label = ttk.Label(min_frame, text='Max Dip')
        min_dip_entry = ttk.Entry(min_frame, textvariable=model.s2_clean_localminimamaxdip)
        min_dif_label = ttk.Label(min_frame, text='Max Dif')
        min_dif_entry = ttk.Entry(min_frame, textvariable=model.s2_clean_localminimamaxdif)
        min_gap_label = ttk.Label(min_frame, text='Max Gap')
        min_gap_entry = ttk.Entry(min_frame, textvariable=model.s2_clean_localminimamaxgap)
        min_pas_label = ttk.Label(min_frame, text='Max Pas')
        min_pas_entry = ttk.Entry(min_frame, textvariable=model.s2_clean_localminimamaxpas)

        min_dip_label.grid(row=0, column=0, sticky='e', padx=5, pady=5)
        min_dip_entry.grid(row=0, column=1, sticky='w', padx=5, pady=5)
        min_dif_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        min_dif_entry.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        min_gap_label.grid(row=2, column=0, sticky='e', padx=5, pady=5)
        min_gap_entry.grid(row=2, column=1, sticky='w', padx=5, pady=5)
        min_pas_label.grid(row=3, column=0, sticky='e', padx=5, pady=5)
        min_pas_entry.grid(row=3, column=1, sticky='w', padx=5, pady=5)

        title_label.grid(row=0, column=0, sticky='w', padx=5, pady=5)
        nar_frame.grid(row=1, column=0, sticky='nsew', padx=5, pady=5)
        ext_frame.grid(row=2, column=0, sticky='nsew', padx=5, pady=5)
        min_frame.grid(row=3, column=0, sticky='nsew', padx=5, pady=5)

        self.columnconfigure(0, weight=1)


class ParcelFilterFrame(ttk.Frame):
    def __init__(self, parent, app, model, page_size=100, **kwargs):
        super().__init__(parent, **kwargs)

        self.app = app
        self.model = model
        self._page_size = page_size

        self._page = tk.IntVar(value=1)
        self._id = tk.StringVar()
        self._croptype = tk.StringVar()
        self._area_min = tk.DoubleVar()
        self._area_max = tk.DoubleVar()

        self._area_min.set('')
        self._area_max.set('')

        filter_frame = ttk.Frame(self, relief='flat')

        filter_label = ttk.Label(filter_frame, text='Filter parcels by:')
        id_label = ttk.Label(filter_frame, text='id')
        id_entry = ttk.Entry(filter_frame, textvariable=self._id, width=20)
        crop_label = ttk.Label(filter_frame, text='croptype')
        self._crop_entry = ttk.Entry(filter_frame, textvariable=self._croptype, width=5)
        area_min_label = ttk.Label(filter_frame, text='min area')
        self._area_min_entry = ttk.Entry(filter_frame, textvariable=self._area_min, width=10)
        area_max_label = ttk.Label(filter_frame, text='max area')
        self._area_max_entry = ttk.Entry(filter_frame, textvariable=self._area_max, width=10)

        filter_label.grid(row=0, column=0, sticky='w', padx=5, pady=5)
        id_label.grid(row=0, column=1, sticky='w', padx=5, pady=5)
        id_entry.grid(row=0, column=2, sticky='ew', padx=5, pady=5)
        crop_label.grid(row=0, column=3, sticky='w', padx=5, pady=5)
        self._crop_entry.grid(row=0, column=4, sticky='ew', padx=5, pady=5)
        area_min_label.grid(row=0, column=5, sticky='w', padx=5, pady=5)
        self._area_min_entry.grid(row=0, column=6, sticky='ew', padx=5, pady=5)
        area_max_label.grid(row=0, column=7, sticky='w', padx=5, pady=5)
        self._area_max_entry.grid(row=0, column=8, sticky='ew', padx=5, pady=5)

        filter_frame.columnconfigure(2, weight=3)
        filter_frame.columnconfigure(4, weight=1)
        filter_frame.columnconfigure(6, weight=1)
        filter_frame.columnconfigure(8, weight=1)

        pager_frame = ttk.Frame(self, relief='flat')

        page_label = ttk.Label(pager_frame, text='Page')
        self._first_button = ttk.Button(pager_frame, text='<<', command=self._on_first, width=3)
        self._prev_button = ttk.Button(pager_frame, text='<', command=self._on_prev, width=3)
        page_entry = tk.Entry(pager_frame, textvariable=self._page, width=5)
        self._next_button = ttk.Button(pager_frame, text='>', command=self._on_next, width=3)
        self._last_button = ttk.Button(pager_frame, text='>>', command=self._on_last, width=3)
        self._range_label = ttk.Label(pager_frame, text='')
        self._records_label = ttk.Label(pager_frame, text='')

        page_label.grid(row=0, column=0, sticky='w', padx=2, pady=0)
        self._first_button.grid(row=0, column=1, sticky='w', padx=2, pady=0)
        self._prev_button.grid(row=0, column=2, sticky='w', padx=2, pady=0)
        page_entry.grid(row=0, column=3, sticky='w', padx=0, pady=2)
        self._next_button.grid(row=0, column=4, sticky='w', padx=2, pady=0)
        self._last_button.grid(row=0, column=5, sticky='w', padx=2, pady=0)
        self._range_label.grid(row=0, column=6, sticky='w', padx=2, pady=0)
        self._records_label.grid(row=0, column=7, sticky='e', padx=2, pady=0)

        pager_frame.columnconfigure(7, weight=1)

        table_frame = ttk.Frame(self, relief='flat')

        self._table = ttk.Treeview(table_frame, selectmode='browse', height=5)
        self._table.bind('<<TreeviewSelect>>', self._table_select)
        self._table['columns'] = ('id', 'croptype', 'area')

        self._table.heading('#0', text='#', anchor='c')
        self._table.heading('id', text='id', anchor='c')
        self._table.heading('croptype', text='croptype', anchor='c')
        self._table.heading('area', text='area', anchor='c')

        self._table.column('#0', width=50, anchor='e')
        self._table.column('id', width=300, anchor='w')
        self._table.column('croptype', width=50, anchor='e')
        self._table.column('area', width=100, anchor='e')

        self._table.tag_configure('monospace', font=tk.font.nametofont('TkFixedFont'))

        scroll = ttk.Scrollbar(table_frame, orient='vertical', command=self._table.yview)

        self._table.configure(yscrollcommand=scroll.set)
        self._table.grid(row=0, column=0, sticky='nsew', padx=0, pady=0)

        scroll.grid(row=0, column=1, sticky='ns')

        table_frame.rowconfigure(0, weight=1)
        table_frame.columnconfigure(0, weight=1)

        filter_frame.grid(row=0, column=0, sticky='new', padx=5, pady=5)
        table_frame.grid(row=1, column=0, sticky='nsew', padx=5, pady=5)
        pager_frame.grid(row=2, column=0, sticky='new', padx=5, pady=5)

        self.columnconfigure(0, weight=1)
        self.rowconfigure(1, weight=1)

        self._page.trace('w', self._on_filter_update)
        self._id.trace('w', self._on_filter_update)
        self._croptype.trace('w', self._on_filter_update)
        self._area_min.trace('w', self._on_filter_update)
        self._area_max.trace('w', self._on_filter_update)

        self.model.shapefile.trace('w', self._on_filter_update)

        self._on_filter_update()

    def _get_df(self):

        df = self.model.get_parcel_data()

        if df is not None:

            id = self._id.get()

            if id:
                df = df.loc[df.fieldID.str.contains(id, case=False, na=False)]

            if 'croptype' in df.columns:
                croptype = self._croptype.get()

                if croptype:
                    df = df.loc[df.croptype.str.contains(croptype, case=False, na=False)]

            if 'area' in df.columns:

                try:
                    area_min = self._area_min.get()
                except:
                    area_min = None

                try:
                    area_max = self._area_max.get()
                except:
                    area_max = None

                if area_min:
                    df = df.loc[df['area'] >= area_min]
                if area_max:
                    df = df.loc[df['area'] <= area_max]

        return df

    def _on_first(self):
        self._page.set(1)

    def _on_prev(self):
        if self._page.get() > 1:
            self._page.set(self._page.get()-1)

    def _on_next(self):
        df = self._get_df()
        pages = (df.shape[0] - 1) // self._page_size + 1
        if self._page.get() < pages:
            self._page.set(self._page.get()+1)

    def _on_last(self):
        df = self._get_df()
        pages = (df.shape[0] - 1) // self._page_size + 1
        self._page.set(pages)

    def _on_filter_update(self, *args):

        df = self._get_df()

        if df is None:
            return

        if 'croptype' in df.columns:
            self._crop_entry.config(state='enabled')
        else:
            self._crop_entry.config(state='disabled')

        if 'area' in df.columns:
            self._area_min_entry.config(state='enabled')
            self._area_max_entry.config(state='enabled')
        else:
            self._area_min_entry.config(state='disabled')
            self._area_max_entry.config(state='disabled')

        pages = (df.shape[0] - 1) // self._page_size + 1

        try:
            page = self._page.get() - 1
        except:
            page = None

        if page is not None:
            if page > 0:
                self._prev_button.config(state='enabled')
                self._first_button.config(state='enabled')
            else:
                self._prev_button.config(state='disabled')
                self._first_button.config(state='disabled')

            if page < pages-1:
                self._next_button.config(state='enabled')
                self._last_button.config(state='enabled')
            else:
                self._next_button.config(state='disabled')
                self._last_button.config(state='disabled')

            from_ = max(page * self._page_size, 0)
            to = min(from_ + self._page_size, df.shape[0])

            self._range_label.config(text=' of {}'.format(pages))
            self._records_label.config(text='Matches {} parcels'.format(df.shape[0]))

            self._fill_table(df, from_, to)

    def _fill_table(self, df, from_, to):

        self._table.delete(*self._table.get_children())

        self._table_items = {}

        for i in range(from_, to):
            index = str(df.index[i])
            info = df.iloc[i]

            rec = ('{}'.format(info['fieldID']),
                   '{}'.format(info['croptype']) if 'croptype' in info else '',
                   '{:0.2f}'.format(info['area']) if 'area' in info else '')

            self._table_items[index] = rec

            self._table.insert('', 'end', text=index, values=rec, tag='monospace')

    def _table_select(self, event):
        sel = self._table.selection()

        if sel:
            index = self._table.item(sel, 'text')
            item = self._table_items[index]
            self.model.parcel.set(item[0])
        else:
            self.model.parcel.set('')


class ParcelFrame(ttk.Frame):
    def __init__(self, parent, app, model, **kwargs):
        super().__init__(parent, **kwargs)

        self.app = app
        self.model = model

        self._file_label = ttk.Label(self, text='Shapefile')
        self._file_entry = ttk.Entry(self, width=30, textvariable=self.model.shapefile, state='readonly')
        self._file_button = ttk.Button(self, text='Browse...', command=self._on_pick_file)

        self._parcel_label = ttk.Label(self, text='Selected Parcel')
        self._parcel_entry = ttk.Entry(self, width=25, textvariable=self.model.parcel, state='readonly')

        self._filter_frame = ParcelFilterFrame(self, app, model, relief='solid', borderwidth=1)

        self._file_label.grid(row=0, column=0, sticky='e', padx=5, pady=5)
        self._file_entry.grid(row=0, column=1, sticky='ew', padx=5, pady=5, columnspan=2)
        self._file_button.grid(row=0, column=3, sticky='w', padx=5, pady=5)
        self._parcel_label.grid(row=1, column=0, sticky='e', padx=5, pady=5)
        self._parcel_entry.grid(row=1, column=1, sticky='w', padx=5, pady=5)
        self._filter_frame.grid(row=2, column=0, sticky='ew', padx=5, pady=5, columnspan=4)

        self.columnconfigure(0, weight=0)
        self.columnconfigure(1, weight=0)
        self.columnconfigure(2, weight=1)
        self.columnconfigure(3, weight=0)

    def _on_pick_file(self):
        self.app.pick_shapefile()


class ConfigFrame(ttk.Frame):
    def __init__(self, parent, app, model, **kwargs):
        super().__init__(parent, **kwargs)

        self.app = app
        self.model = model

        scroll_frame = ScrollableFrame(self, relief='solid', borderwidth=1)
        sections_frame = ttk.Frame(scroll_frame.scrolled_frame())

        general_frame = GeneralParameterFrame(sections_frame, model)
        cropsar_frame = CropSARParameterFrame(sections_frame, model)
        s1_clean_frame = S1CleaningParameterFrame(sections_frame, model)
        s2_clean_frame = S2CleaningParameterFrame(sections_frame, model)

        general_frame.grid(row=0, column=0, sticky='nsew', padx=5, pady=5)
        cropsar_frame.grid(row=1, column=0, sticky='nsew', padx=5, pady=5)
        s1_clean_frame.grid(row=2, column=0, sticky='nsew', padx=5, pady=5)
        s2_clean_frame.grid(row=3, column=0, sticky='nsew', padx=5, pady=5)

        sections_frame.pack(expand=True, fill='both')
        scroll_frame.grid(row=1, column=0, sticky='nsew', padx=5, pady=5)

        self.columnconfigure(0, weight=1)
        self.rowconfigure(1, weight=1)


class ActionFrame(ttk.Frame):
    def __init__(self, parent, app, model, **kwargs):
        super().__init__(parent, **kwargs)

        self.app = app
        self.model = model

        self._plot_button = ttk.Button(self, text='Inspect', command=self._on_inspect)
        self._plot_button.grid(row=0, column=1, padx=5, pady=5, ipadx=10, ipady=5)

        self.columnconfigure(0, weight=1)

        self.model.parcel.trace('w', self._on_parcel_update)

        self._on_parcel_update()

    def _on_inspect(self):
        self.app.inspect()

    def _on_parcel_update(self, *args):
        if self.model.parcel.get():
            self._plot_button.config(state='enabled')
        else:
            self._plot_button.config(state='disabled')


class HeaderFrame(ttk.Frame):
    def __init__(self, parent, app, model, **kwargs):
        super().__init__(parent, relief='solid', borderwidth=1, **kwargs)

        title_label = ttk.Label(self, text='Parcel Inspector',
                                justify='center',
                                anchor='w',
                                padding=20,
                                foreground='black',
                                background='white')

        title_font = tk.font.Font(font=title_label['font'])
        title_font['weight'] = 'bold'
        title_font['size'] *= 3

        title_label.config(font=title_font)
        title_label.grid(row=0, column=0, sticky='nsew')

        try:
            self._logo = app.load_logo() # store reference!
        except:
            self._logo = None

        if self._logo is not None:
            logo_label = ttk.Label(self,
                                   image=self._logo,
                                   padding=5,
                                   foreground='black',
                                   background='white',)

            logo_label.grid(row=0, column=1, sticky='nse')

        self.columnconfigure(0, weight=1)


class FooterFrame(ttk.Frame):
    def __init__(self, parent, app, model, **kwargs):
        super().__init__(parent, **kwargs)

        grip = ttk.Sizegrip(self)
        grip.grid(row=0, column=0, stick='e')

        self.columnconfigure(0, weight=1)


class Application(object):
    def __init__(self, shapefile=None, fontscale=1.0):

        self._root = tk.Tk()
        self._root.wm_title('Parcel Inspector')
        self._root.protocol('WM_DELETE_WINDOW', self._on_quit)

        if fontscale != 1.0:
            self._scale_fonts(fontscale)

        self.model = Model(self)

        main_frame = ttk.Frame(self._root)
        main_frame.pack(expand=True, fill='both')

        header_frame = HeaderFrame(main_frame, self, self.model)
        parcel_frame = ParcelFrame(main_frame, self, self.model)
        config_frame = ConfigFrame(main_frame, self, self.model)
        action_frame = ActionFrame(main_frame, self, self.model)
        footer_frame = FooterFrame(main_frame, self, self.model)

        header_frame.grid(column=0, row=0, sticky='nsew')
        parcel_frame.grid(column=0, row=1, sticky='nsew')
        config_frame.grid(column=0, row=2, sticky='nsew')
        action_frame.grid(column=0, row=3, sticky='nsew')
        footer_frame.grid(column=0, row=4, sticky='nsew')

        main_frame.rowconfigure(2, weight=1)
        main_frame.columnconfigure(0, weight=1)

        if shapefile:
            self.load_shapefile(shapefile)

        self._root.update()

        types = cropsar.get_available_model_types()

        with ProgressDialog(self._root, 'Loading CropSAR models') as dialog:
            dialog.set_maximum(len(types))

            for type in types:
                dialog.set_text('Loading {} model...'.format(type))

                log.info('Loading {} model...'.format(type))
                cropsar.get_model(type)

                dialog.step()

            log.info('Done')

    def _scale_fonts(self, scale):

        for name in tk.font.names():
            font = tk.font.nametofont(name)
            size = round(font['size'] * scale)
            if size <= 0:
                size = 10
            font.configure(size=size)

    def load_logo(self):

        data = pkgutil.get_data('cropsar', 'resources/logo.png')
        if data is not None:
            return tk.PhotoImage(data=data)
        return None

    def pick_shapefile(self):

        filename = tk.filedialog.askopenfilename(
            title='Select a shapefile',
            filetypes=[('ESRI Shapefiles', '*.shp'),
                       ('GeoJSON files',   '*.geojson;*.json'),
                       ('All files',       '*.*')])

        if filename:
            self.load_shapefile(filename)

    # TODO: Remove this function after V102 support is removed
    def confirm_reload_params(self):

        self._root.grab_set()
        try:
            return tk.messagebox.askyesno('Reload parameters',
                                          'Do you want to reload parameters?')
        finally:
            self._root.grab_release()

    def load_shapefile(self, filename):

        try:
            with ProgressDialog(self._root,
                                title='Loading shapefile',
                                mode='indeterminate') as dialog:

                dialog.set_text('Loading {}'.format(os.path.basename(filename)))
                dialog.start()

                log.info('Loading shapefile ' + filename)
                self.model.load_shapefile(filename, dialog)
                log.info('Done')

        except Exception as e:
            traceback.print_exc()
            tk.messagebox.showerror('Error loading shapefile', str(e))

    def inspect(self):

        parcel_id = self.model.parcel.get()
        parcel_df = self.model.get_parcel_data()
        product = self.model.cropsar_product.get()
        start_date = self.model.start_date.get()
        end_date = self.model.end_date.get()
        params = self.model.get_cropsar_params()
        source = self.model.source.get()

        log.info('Inspecting parcel {}'.format(parcel_id))
        log.info('Using product: {}'.format(product))
        log.info('Using start_date: {}'.format(start_date))
        log.info('Using end_date: {}'.format(end_date))
        log.info('Using source: {}'.format(source))
        log.info('Using params: {}'.format(params))

        frame = FigureFrame(self._root)
        frame.wm_title('Parcel Inspector: {}'.format(parcel_id))
        frame.geometry('1024x768')

        try:
            data = TimeseriesPlotDataSource(parcel_id, parcel_df, product, start_date, end_date, params, source)
            plot = TimeseriesPlot(frame.fig, data)

            frame.keep_alive(plot)

        except Exception as e:
            traceback.print_exc()
            tk.messagebox.showerror('Error initializing plot', str(e))

    def run(self):
        self._root.mainloop()

    def _on_quit(self):
        self._root.quit()
        self._root.destroy()


def main(argv=sys.argv):

    logging.basicConfig(level=logging.INFO,
                        format='{asctime} {levelname} {name}: {message}',
                        style='{',
                        datefmt='%Y-%m-%d %H:%M:%S')

    log.setLevel(logging.INFO)

    logging.getLogger('urllib3').setLevel(logging.DEBUG)

    parser = argparse.ArgumentParser(prog=argv[0])

    parser.add_argument('-f', '--shapefile', type=str, required=False, default=None)
    parser.add_argument(      '--fontscale', type=float, required=False, default=1.0)

    args = parser.parse_args(argv[1:])

    app = Application(shapefile=args.shapefile,
                      fontscale=args.fontscale)

    app.run()


if __name__ == '__main__':
    main()

