import json
import pkg_resources
import openeo
from openeo.api.process import Parameter

from cropsar_px.udf_gan import load_cropsar_px_udf
from cropsar_px.preprocessing_openeo import cropsar_pixel_inputs


def get_connection():
    return openeo.connect("https://openeo-dev.vito.be").authenticate_basic()

def get_process_graph():
    stream = pkg_resources.resource_stream("cropsar_px", "resources/udp.json")
    return json.load(stream)


if __name__ == "__main__":
    param_extent = Parameter(
        name="extent",
        description="Region of interest",
        schema={
            "type": "object",
            "properties": {
                "west": {"type": "number"},
                "south": {"type": "number"},
                "east": {"type": "number"},
                "north": {"type": "number"},
                "crs": {"type": "string"}
            }
        }
    )

    param_startdate = Parameter.string(name="startdate")
    param_enddate = Parameter.string(name="enddate")

    connection = get_connection()

    input_cube = cropsar_pixel_inputs(connection, param_extent, param_startdate, param_enddate)

    gan_udf_code = load_cropsar_px_udf()

    cropsar_pixel_cube = input_cube.apply_neighborhood(
        lambda data: data.run_udf(udf=gan_udf_code, runtime='Python', context={
            "startdate": {"from_parameter": "startdate"},
            "enddate": {"from_parameter": 'enddate'}
        }),
        size=[
            {'dimension': 'x', 'value': 224, 'unit': 'px'},
            {'dimension': 'y', 'value': 224, 'unit': 'px'}
        ],
        overlap=[
            {'dimension': 'x', 'value': 16, 'unit': 'px'},
            {'dimension': 'y', 'value': 16, 'unit': 'px'}
        ]
    )

    cropsar_pixel_cube = cropsar_pixel_cube.add_dimension("bands", "ndvi", type="bands")
    cropsar_pixel_cube = cropsar_pixel_cube.filter_temporal(param_startdate, param_enddate).filter_bbox(param_extent)
    cropsar_pixel_cube = cropsar_pixel_cube.linear_scale_range(-0.08, 1, 0, 250)

    spec = {
        "id": "CropSAR_px",
        "summary": "CropSAR_px",
        "description": pkg_resources.resource_string("cropsar_px", "resources/udp_description.md").decode("utf-8"),
        "parameters": [
            param_extent.to_dict(),
            param_startdate.to_dict(),
            param_enddate.to_dict()
        ],
        "process_graph": cropsar_pixel_cube.graph
    }

    # write the UDP to a file
    with open("resources/udp.json", "w") as f:
        json.dump(spec, f, indent=4)

    # save the UDP in the back-end for the current user
    # connection.save_user_defined_process(spec['id'], cropsar_pixel_cube, [param_extent, param_startdate, param_enddate])