#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Aug 24 09:46:26 2022

@author: bertelsl

Algorithm:
    
    - group the pre-processed data per crop type:
        
        'fIndir': r'/data/EEA_HRL_VLCC/user/luc/data/LUCAS2018/01_LUCAS_preprocessed/',
        'fOutdir': r'/data/EEA_HRL_VLCC/user/luc/data/LUCAS2018/03_LUCAS_crops/',
        
Version: 02/09/2022

"""
import os
import glob
import netCDF4
import json
import xarray as xr
import numpy as np
import pandas as pd


#================================================================================================================
class cGroupLUCAS(object):
#================================================================================================================
    def __init__(self, Info):        
#================================================================================================================
        
        self.fIndir = Info['fIndir']
        self.fOutdir = Info['fOutdir']
        self.overwrite = Info['overwrite']
        
        if not os.path.isdir(self.fOutdir):
            os.mkdir(self.fOutdir)        
            
#================================================================================================================
    def start_processing(self):        
#================================================================================================================
        
        self.get_years()
        
        for year in self.aYears:
            
            self.get_data('S1', year)
            self.group_crops('S1', year)
            
            self.get_data('S2', year)
            self.group_crops('S2', year)

#================================================================================================================
    def group_crops(self, forS, year):        
#================================================================================================================
        
        aCrops = self.aXinput['labels'].values
        aUnique_crops = np.unique(aCrops)
        
        for crop in aUnique_crops:

            fOut = os.path.join(self.fOutdir, '{}_{}_{}.nc'.format(forS, year, crop.replace('/', '_')))
            
            if os.path.isfile(fOut):
                if self.overwrite:
                    os.remove(fOut)
                else:
                    continue

            print('\r * Writing data for {}'.format(crop), 
                  end='                                                                                                                                               ')
            
            xCrop = self.aXinput.where(self.aXinput['labels'] == crop, drop=True)
            xCrop.to_netcdf(fOut)
            
        print('')

#================================================================================================================
    def get_data(self, forS, for_year):        
#================================================================================================================
        
        afS = glob.glob(self.fIndir + '{}*.nc'.format(forS))
        afS.sort()
        
        count = 0
        
        for fS in afS:
            print('\r * Reading data {} / {} for {} - {}'.format(count, len(afS)-1, for_year, os.path.basename(fS).split('.')[0]),
                  end='                                                                                                                                               ')

            xInput = xr.open_dataset(fS)
            
            aDate = xInput['date'].values
            year = str(aDate[0])[0:4]
            
            if year != for_year:
                continue
            
            if count == 0:
                self.aXinput = xInput
            else:
                self.aXinput = xr.concat([self.aXinput, xInput], dim='labels')

            count += 1

        print('')

#================================================================================================================
    def get_years(self):        
#================================================================================================================
        
        afS = glob.glob(self.fIndir + '*.nc')
        afS.sort()
        
        aYears = []
        
        for fS in afS:
            aYears.append(os.path.basename(fS).split('_')[1])
        
        self.aYears = np.unique(aYears)
        
#================================================================================================================
if __name__ == '__main__':
#================================================================================================================

    Info = {
        'fIndir': r'/data/EEA_HRL_VLCC/user/luc/data/LPIS/02_LPIS_preprocessed/',
        'fOutdir': r'/data/EEA_HRL_VLCC/user/luc/data/LPIS/04_LPIS_crops/',
        'overwrite': True
        }

    oGroupLUCAS = cGroupLUCAS(Info)
    oGroupLUCAS.start_processing()
