from openeo.api.process import Parameter
from openeo_processes.utils.udfstring import UDFString
import os
import openeo
from openeo_processes.utils.mask import create_advanced_mask


def getImageCollection(eoconn,layerid,startdate,enddate,bands,bbox,epsg):
    coll=eoconn.load_collection(layerid)\
    .filter_temporal(startdate,enddate)\
    .filter_bbox(crs=epsg, **dict(zip(["west", "south", "east", "north"], bbox)))
    if bands is not None: coll=coll.filter_bands(bands)
    return coll


def build_vectorization_processgraph(
        eoconn,
        maxlayers=16, # number of least clouded observations to use
        nblocks=5, # how many windows to group per executor (performance improvement+centerpoint delineation allows to use larger contiguous area) 
        overlap=32, # property of the model, do not change
        windowsize=128, # property of the model, do not change
    ):

    # set parameters
    parameters={
        "start_date": Parameter(
            name="start_date", description="Start of time window",
            schema="string"
        ),
        "end_date": Parameter(
            name="end_date", description="End of time window",
            schema="string"
        ),
        "epsg": Parameter(
            name="epsg", description="Coordinate system to use as string.",
            schema="string",
            default="EPSG:4326"
        ),
        "bbox_west": Parameter(
            name="bbox_west", description="Bounding box left",
            schema="number"
        ),
        "bbox_south": Parameter(
            name="bbox_south", description="Bounding box bottom",
            schema="number"
        ),
        "bbox_east": Parameter(
            name="bbox_east", description="Bounding box right",
            schema="number"
        ),
        "bbox_north": Parameter(
            name="bbox_north", description="Bounding box top",
            schema="number"
        ),
    }
    
    # get needed bands
    # SENTINEL2_L2A_SENTINELHUB
    # TERRASCOPE_S2_TOC_V2
    bands=getImageCollection(
        eoconn,
        "TERRASCOPE_S2_TOC_V2",
        parameters["start_date"],
        parameters["end_date"],
        ["B04","B08","SCL"],
        bbox=[{'from_parameter':'bbox_west'},{'from_parameter':'bbox_south'}, {'from_parameter':'bbox_east'}, {'from_parameter':'bbox_north'}],
        epsg={'from_parameter':'epsg'}
    )

    # compute the mask
    maskband=create_advanced_mask(bands.filter_bands(["SCL"]).band("SCL"),parameters["start_date"],parameters["end_date"])
    
    # compute ndvi
    ndviband=bands.filter_bands(["B04","B08"]).ndvi(red="B04",nir="B08")

    # set NaN where mask is active
    ndviband=ndviband.mask(maskband)

    # assemble segmentation code
    # * reduce images select the top usable images
    # * segmentation core runs segmentation
    segmentation_udf_code = UDFString()\
        .load(os.path.dirname(__file__)+'/../udf/reduce_images.py')\
        .load(os.path.dirname(__file__)+'/../udf/segmentation.py')\
        .load(os.path.dirname(__file__)+'/../udf/sobel_felzenszwalb.py')\
        .prepare()
    #print(segmentation_udf_code)

    # run on overlapping tiles
    segmentationband = ndviband.apply_neighborhood(openeo.UDF(
        code=segmentation_udf_code, 
        runtime="Python",
        data={'from_parameter': 'data'},
        context={
            'maxlayers': maxlayers,
            'modeldir': '/data/users/Public/kristofvt/BelgianCropMap/models'
        }
    ), size=[
        {'dimension': 'x', 'value': (windowsize-2*overlap)*nblocks, 'unit': 'px'},
        {'dimension': 'y', 'value': (windowsize-2*overlap)*nblocks, 'unit': 'px'}
    ], overlap=[
        {'dimension': 'x', 'value': overlap, 'unit': 'px'},
        {'dimension': 'y', 'value': overlap, 'unit': 'px'}
    ])
    segmentationband=segmentationband.add_dimension("bands", "segmentation", type="bands").band("segmentation")

    # vectorization
    vectorization=segmentationband.raster_to_vector()

    # fill the necessary parts
    pg={
        'process_graph': vectorization,
        'parameters': [ i.to_dict() for i in parameters.values() ],
    }
    return pg
