import logging
import os
import openeo
import json
from openeo_processes.utils.pg_vectorization import build_vectorization_processgraph
from openeo_processes.utils.pg_helper import pg_save, pg_load, pg_upload
import geopandas
from shapely.geometry.polygon import Polygon
import pandas
from matplotlib import pyplot
from openeo_processes.utils.pg_centerpolygon_select import add_centerpolygon_process
import contextily

#openeo_url='http://localhost:8080/openeo/1.0.0/'
openeo_url='http://openeo-dev.vgt.vito.be/openeo/1.0.0/'
#openeo_url='http://openeo.vgt.vito.be/openeo/1.0.0/'

openeo_user=os.environ.get('OPENEO_USER',os.environ['USER'])
openeo_pass=os.environ.get('OPENEO_PASS',os.environ['USER']+'123')

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

if __name__ == '__main__':

    ##################################################################
    # connect to openeo
    ##################################################################
    eoconn=openeo.connect(openeo_url)
    eoconn.authenticate_basic(openeo_user,openeo_pass)

    ##################################################################
    # build_the generic process graph, that can go to:
    # https://git.vito.be/projects/APPL/repos/nextland/browse/src/nextland_services/resources/process_graphs
    ##################################################################
    # this is the generic delineation over an area
    vpg=build_vectorization_processgraph(eoconn, nblocks=1)
    # this is the extra process that selects a group of interconnected polygons starting from a point
    vpg=add_centerpolygon_process(vpg)
    pg_save("delineation_centerpoint_graph.json", "delineation_centerpoint", "Delineate around centerpoint", "Find the vector representation of a field enclosing the given coordinate", vpg)

    ##################################################################
    # if you need to manually upload the UDP
    ##################################################################
    vpg=pg_load("delineation_centerpoint_graph.json")
    pg_upload(eoconn, vpg)

    ##################################################################
    # example run
    ##################################################################
    epsg="EPSG:32631"
    startdate='2018-10-01'
    enddate='2019-08-31'
    cpx=554300
    cpy=5444800
    boxsize=1000
    
    eoconn.datacube_from_process(
        vpg['id'],
        start_date=startdate,
        end_date=enddate,
        epsg=epsg,
        bbox_west=cpx-0.5*boxsize,
        bbox_south=cpy-0.5*boxsize,
        bbox_east=cpx+0.5*boxsize,
        bbox_north=cpy+0.5*boxsize,
        centerpoint_x=cpx,
        centerpoint_y=cpy,
    ).download("vectorselect.json")
    
    ##################################################################
    # visualization
    ##################################################################
    
    # polygon
    with open('vectorselect.json') as f: v=json.load(f)
    for i in range(len(v)): v[i]['coordinates']=v[i]['coordinates'][0]
    df=geopandas.GeoDataFrame.from_dict(v,crs='EPSG:32631')
    df['geometry'] = df.coordinates.apply(Polygon)
    df.to_crs('EPSG:3857',inplace=True)
    ax=df.plot(color='#FF000044')
     
    # center point that was used
    box='POINT('+str(cpx)+' '+str(cpy)+')'
    bdf=pandas.DataFrame({'geom':geopandas.GeoSeries.from_wkt([box])})
    bgdf=geopandas.GeoDataFrame(bdf,geometry='geom')
    bgdf.crs=epsg
    bgdf.to_crs('EPSG:3857',inplace=True)
    bgdf.plot(ax=ax,color='#FF0000')

    # some satellite imagery for reference
    img,ext=contextily.bounds2img(*df.scale(2,2).total_bounds, source=contextily.providers.Esri.WorldImagery)
    grayimg=0.2989*img[:,:,0]+0.5870*img[:,:,1]+0.1140*img[:,:,2]
    grayimg=grayimg-grayimg.min()
    grayimg=grayimg/grayimg.max()
    pyplot.imshow(grayimg,cmap='gray',vmin=-1,vmax=1,extent=ext)
    
    pyplot.show()

    logger.info('FINISHED')






