import logging
import os
import openeo
import json
from openeo_processes.utils.pg_vectorization import build_vectorization_processgraph
from openeo_processes.utils.pg_helper import pg_save, pg_load, pg_upload
import geopandas
from shapely.geometry.polygon import Polygon
import shapely.geometry
import pandas
from matplotlib import pyplot
import contextily
from openeo_processes.utils.pg_filter_smallpolygons import add_filtersmallpolygons_process

#openeo_url='http://localhost:8080/openeo/1.0.0/'
openeo_url='http://openeo-dev.vgt.vito.be/openeo/1.0.0/'
#openeo_url='http://openeo.vgt.vito.be/openeo/1.0.0/'

openeo_user=os.environ.get('OPENEO_USER',os.environ['USER'])
openeo_pass=os.environ.get('OPENEO_PASS',os.environ['USER']+'123')

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

if __name__ == '__main__':

    ##################################################################
    # connect to openeo
    ##################################################################
    eoconn=openeo.connect(openeo_url)
    eoconn.authenticate_basic(openeo_user,openeo_pass)

    ##################################################################
    # build_the generic process graph, that can go to:
    # https://git.vito.be/projects/APPL/repos/nextland/browse/src/nextland_services/resources/process_graphs
    ##################################################################
    # this is the generic delineation over an area
    vpg=build_vectorization_processgraph(eoconn)
    # this is the extra process that selects a group of interconnected polygons starting from a point
    vpg=add_filtersmallpolygons_process(vpg)
    pg_save("delineation_area_graph.json", "delineation_area", "Delineation within a bounding box", "Based on a series of ndvi images, find the vector representation of the agricultural fields", vpg)

    ##################################################################
    # if you need to manually upload the UDP
    ##################################################################
    vpg=pg_load("delineation_area.json")
    pg_upload(eoconn, vpg)

    ##################################################################
    # example run
    ##################################################################
    epsg="EPSG:32631"
    startdate='2018-10-01'
    enddate='2019-08-31'
    bbox=(552300,5442800,556300,5446800)
    
    eoconn.datacube_from_process(
        vpg['id'],
        start_date=startdate,
        end_date=enddate,
        epsg=epsg,
        bbox_west=bbox[0],
        bbox_south=bbox[1],
        bbox_east=bbox[2],
        bbox_north=bbox[3],
    ).download("vectorselect.json")
    
    ##################################################################
    # visualization
    ##################################################################
    
    # polygon
    with open('vectorselect.json') as f: v=json.load(f)
    for i in range(len(v)): v[i]['coordinates']=v[i]['coordinates'][0]
    df=geopandas.GeoDataFrame.from_dict(v,crs='EPSG:32631')
    df['geometry'] = df.coordinates.apply(Polygon)
    df.to_crs('EPSG:3857',inplace=True)
    df['idx']=df.index
    ax=df.plot(column='idx')
     
    # center point that was used
    box=shapely.geometry.box(*bbox)
    bdf=pandas.DataFrame({'geom':[box]})
    bgdf=geopandas.GeoDataFrame(bdf,geometry='geom')
    bgdf.crs=epsg
    bgdf.to_crs('EPSG:3857',inplace=True)
    bgdf.plot(ax=ax,color='#FF000044')

    # some satellite imagery for reference
    img,ext=contextily.bounds2img(*df.scale(2,2).total_bounds, source=contextily.providers.Esri.WorldImagery)
    grayimg=0.2989*img[:,:,0]+0.5870*img[:,:,1]+0.1140*img[:,:,2]
    grayimg=grayimg-grayimg.min()
    grayimg=grayimg/grayimg.max()
    pyplot.imshow(grayimg,cmap='gray',vmin=-1,vmax=1,extent=ext)
    
    pyplot.show()

    logger.info('FINISHED')






