""""
Functions that will deal with the conversion of the delineation output to
an intiantiazed output and vectorized output 
"""
from loguru import logger as log
import os
import numpy as np
from skimage.filters import sobel
from skimage import segmentation
from skimage.future import graph
import cv2


def _apply_felzenswalb(image, min_px_field=30):

    os.environ['GDAL_DATA'] = '/usr/share/gdal'

    log.info('-' * 75)
    log.info('START FELZENSWALB SEGMENTATION')

    ###################################################
    # Segmentation
    ###################################################

    # Calculate the edges using sobel
    log.info('Calculating edges ...')
    edges = sobel(image)

    # Perform felzenszwalb segmentation
    log.info('Performing Felzenszwalb segmentation ...')
    segments = np.array(segmentation.felzenszwalb(
        image, scale=1, channel_axis=None,
        sigma=0., min_size=min_px_field)).astype(int)

    # Perform the rag boundary analysis and merge the segments
    log.info('Performing RAG analysis ...')
    g = graph.rag_boundary(segments, edges)
    log.info('Merging segments ...')
    mergedSegments = graph.cut_threshold(
        segments.astype(int), g, 0.15, in_place=False)

    # Add 1 to mergedsegments otherwise the zero ID will be ignored
    mergedSegments += 1

    # Felzenswalb does delineate fields in zones with no prob of a field
    # putting zero probabilty to nan does give other artefacts
    # therefore as post-processing the id with the highes occurance should
    # be removed

    unique_seg, counts_seg = np.unique(mergedSegments, return_counts=True)
    idx_max = np.argmax(counts_seg)
    id_remove = unique_seg[idx_max]
    mergedSegments[mergedSegments == id_remove] = 0

    # expand predictions needed as the model always leaves
    #  a few pixels boundary with low probabilities
    dilate_segm = [
        cv2.dilate(x.astype(np.float32),
                   np.ones((3, 3)), iterations=1)
        for x in mergedSegments]
    dilate_segm = np.array(dilate_segm).astype(int).reshape(segments.shape)

    # remove small fields

    # Get the ID counts of non-zero labels
    ids, counts = np.unique(dilate_segm, return_counts=True)
    ids, counts = ids[1:], counts[1:]

    # Calculate the removal mask
    mask = np.where(
        np.isin(dilate_segm, ids[counts < 25]), True, False
    )

    dilate_segm[mask] = 0

    # We currently take 0.3 as the binary threshold to distinguish between
    # segments of fields and other segments.
    # This could definitely be improved and made more objective.

    # NOTE: new implementation uses scaled data, so threshold needs to be
    # scaled as well!

    # log.info('Cleaning segments (threshold = 0.3) ...')
    dilate_segm = dilate_segm.astype(float)
    # should be disabled if want to
    # expand the field boundaries
    # dilate_segm[image < 0.3 * 255] = np.nan
    dilate_segm[dilate_segm < 0] = 0
    dilate_segm[np.isnan(dilate_segm)] = 0

    return dilate_segm
