import pyproj
import geopandas as gpd
import xarray as xr
import numpy as np
from loguru import logger as log
import requests
import json
import rasterio


def reproj_geom(from_epsg, to_epsg, geom):
    import shapely
    proj_from = pyproj.Proj(from_epsg)
    proj_to = pyproj.Proj(to_epsg)

    def project(x, y, z=0):
        return pyproj.transform(proj_from, proj_to, x, y, always_xy=True)

    # And apply to all coordinates in the geometry

    return shapely.ops.transform(project, geom)


def get_crs_from_xr(dir_ds):
    from osgeo import osr

    proj_from = osr.SpatialReference(
        xr.open_dataset(dir_ds).crs.crs_wkt)
    crs = int(proj_from.GetAttrValue('AUTHORITY', 1))

    return crs


def get_crs_from_rasterio(dir_raster):
    from osgeo import osr, gdal

    ds_reference = gdal.Open(dir_raster)
    proj = osr.SpatialReference(wkt=ds_reference.GetProjection())
    crs = int(proj.GetAttrValue('AUTHORITY', 1))

    return crs


def get_shape_target(patch_dir, target_dir, window=None):
    """
    Function that will derive a bounding box from the selected window
    :param ds:
    :param window:
    :return:
    """
    from shapely.geometry import Polygon

    # get epsg target
    crs_to = get_crs_from_rasterio(target_dir)

    # Get epsg source
    crs_from = get_crs_from_xr(patch_dir)

    # Now define the xarray bounds
    ds = xr.open_dataset(patch_dir, engine='h5netcdf')
    if window is None:
        coord_list = np.array(np.meshgrid(
            ds.x.values, ds.y.values)).T.reshape(-1, 2)
    else:
        # Define the different coordinates used to build the shape of the window
        coord_list = np.array(np.meshgrid(ds.x.values[window[1][0]: window[1][1]],
                                          ds.y.values[window[0][0]: window[0][1]])).T.reshape(-1, 2)

    # Create polygon from bounds
    geom = Polygon(coord_list).convex_hull
    # reproject the polygon now
    geom_reproj = reproj_geom(crs_from, crs_to, geom)
    gpd_geom = gpd.GeoDataFrame(geometry=[geom])
    gpd_geom.crs = crs_from
    gpd_geom.to_file(
        r'/vitodata/EEA_HRL_VLCC/user/kasper/tmp/bounds_window.shp')
    gpd_geom_LAEA = gpd.GeoDataFrame(geometry=[geom_reproj])
    gpd_geom_LAEA.crs = crs_to
    gpd_geom_LAEA.to_file(
        r'/vitodata/EEA_HRL_VLCC/user/kasper/tmp/bounds_window_LAEA.shp')
    return geom_reproj


def get_tile_bounds(tile):
    _TS_S2_BASEURL = ('https://services.terrascope.be/catalogue/'
                      'products?collection=urn%3Aeop%3AVITO%3A'
                      'TERRASCOPE_S2')
    log.info('Getting tile bounds and crs for tile {} ...'.format(tile))

    # Make a query to the catalog to retrieve an example file
    response = requests.get((f'{_TS_S2_BASEURL}_TOC_V2&tileId={tile}'
                             '&cloudCover=[0,100]'
                             '&start=2022-01-01&end=2022-01-31'
                             '&accessedFrom=MEP'))
    response = json.loads(response.text)
    ex_file = str(response['features'][0]['properties']
                  ['links']['data'][0]['href'][7:])

    with rasterio.open(ex_file, 'r') as src:
        bounds = src.bounds
        crs = src.crs.to_string()
    return bounds, crs
