
import datetime

from concurrent.futures import ThreadPoolExecutor

import geopandas
import numpy
import rasterio
import rasterio.io
import rasterio.plot
import requests


# https://www.sentinel-hub.com/faq/where-can-i-get-list-available-dates-imagery-specific-area/
# https://www.sentinel-hub.com/develop/api/ogc/standard-parameters/wms/
# https://www.sentinel-hub.com/develop/api/ogc/standard-parameters/wfs/
# https://www.sentinel-hub.com/develop/api/ogc/standard-parameters/wcs/
# https://www.sentinel-hub.com/develop/api/ogc/custom-parameters/


DEFAULT_INSTANCE_ID = 'f068f42b-fdc8-49c3-a92b-8c633f967a3b'


def get_http_session():

    from requests.packages.urllib3 import Retry
    from requests.adapters import HTTPAdapter

    # first retry is immediate (sleep=0), from then on:
    #
    # sleep = {backoff factor} * (2 ** ({number of total retries} - 1))

    retry_strategy = Retry(
        total=5,
        backoff_factor=1,   # 0, 2, 4, 8, 16
        status_forcelist=[429,  # Too Many Requests  (rate-limiting)
                          502,  # Bad Gateway
                          503,  # Service Unavailable
                          504], # Gateway Timeout
        allowed_methods=['GET']
    )

    adapter = HTTPAdapter(max_retries=retry_strategy)

    session = requests.Session()
    session.mount('http://', adapter=adapter)
    session.mount('https://', adapter=adapter)

    return session


def get_features(layer, start, end, bbox, crs='epsg:4326',
                 instance_id=DEFAULT_INSTANCE_ID):

    # bbox = minx, miny, maxx, maxy
    # end = exclusive

    layer = {'S2_L1C': 'DSS1',
             'S2_L2A': 'DSS2',
             'S1':     'DSS3'}.get(layer, layer)
       
    df = geopandas.GeoDataFrame()

    while True:
        
        url = 'https://services.sentinel-hub.com/ogc/wfs/' + instance_id

        params = {'SERVICE': 'WFS',
                  'VERSION': '2.0.0',
                  'REQUEST': 'GetFeature',
                  'TIME': start + 'T00:00:00Z/' + end + 'T00:00:00Z',
                  'TYPENAMES': layer,
                  'MAXFEATURES': 100,
                  'BBOX': ','.join(str(x) for x in bbox),
                  'srsName': crs,
                  'OUTPUTFORMAT': 'application/json',
                  'FEATURE_OFFSET': df.shape[0]}

        http = get_http_session()
        response = http.get(url, params=params, timeout=10)
        response.raise_for_status()

        df_part = geopandas.GeoDataFrame.from_features(response.json())

        if df_part.empty:
            break

        df = df.append(df_part, ignore_index=True, sort=False)

    return df


def get_acquisition_dates(layer, start, end, bbox, crs='epsg:4326',
                          maxcc=100,
                          instance_id=DEFAULT_INSTANCE_ID):

    # maxcc = percentage [0-100]

    df = get_features(layer, start, end, bbox, crs, instance_id)

    if df.empty:
        return []

    df = df.loc[df.cloudCoverPercentage <= maxcc]

    return list(df.date.sort_values().unique())


def get_image(layer, date, bbox, crs='epsg:4326',
              resolution='10m', maxcc=100, priority='leastCC',
              instance_id=DEFAULT_INSTANCE_ID):

    start = date

    end = datetime.datetime.strptime(date, '%Y-%m-%d')
    end = end + datetime.timedelta(days=1)
    end = end.strftime('%Y-%m-%d')

    url = 'https://services.sentinel-hub.com/ogc/wcs/' + instance_id

    params = {'SERVICE': 'WCS',
              'VERSION': '1.1.2',
              'REQUEST': 'GetCoverage',
              'TIME': date + 'T00:00:00Z/' + end + 'T00:00:00Z',
              'COVERAGE': layer,
              'FORMAT': 'image/tiff',
              'BBOX': ','.join(str(x) for x in bbox),
              'CRS': crs,
              'RESX': resolution,
              'RESY': resolution,
              'MAXCC': maxcc,
              'PRIORITY': priority}

    http = get_http_session()
    response = http.get(url, params=params, timeout=10)
    response.raise_for_status()

    with rasterio.io.MemoryFile(response.content) as memfile:
        with memfile.open() as image:
            return image.read(), image.profile


def get_images(layer, dates, bbox, crs='epsg:4326',
               resolution='10m', maxcc=100, priority='leastCC',
               max_workers=8, instance_id=DEFAULT_INSTANCE_ID):

    with ThreadPoolExecutor(max_workers=max_workers) as pool:
        def fun(date):
            return get_image(layer, date, bbox, crs,
                             resolution, maxcc, priority,
                             instance_id)

        return list(pool.map(fun, dates))


def get_datacube(start, end, bbox, crs='epsg:4326',
                 resolution='10m', maxcc=100, priority='leastCC',
                 max_workers=8, instance_id=DEFAULT_INSTANCE_ID):

    dates = get_acquisition_dates('S2_L2A', start, end, bbox,
                                  crs, maxcc, instance_id)
    
    images = get_images('S2_NDVI', dates, bbox, crs,
                        resolution, maxcc, priority,
                        max_workers, instance_id)

    profile = images[0][1].copy()
    profile.update(count=len(images))

    dtype = images[0][0].dtype
    h = images[0][0].shape[1]
    w = images[0][0].shape[2]
    
    b04 = numpy.empty((len(images), h, w), dtype)
    b08 = numpy.empty((len(images), h, w), dtype)
    scl = numpy.empty((len(images), h, w), dtype)

    for i, (data, prof) in enumerate(images):
        b04[i,:,:] = data[0,:,:]
        b08[i,:,:] = data[1,:,:]
        scl[i,:,:] = data[2,:,:]

    return {'B04': b04,
            'B08': b08,
            'SCL': scl,
            'profile': profile,
            'dates': dates}
