from copy import deepcopy
from enum import Enum
from pathlib import Path

# ---------------------------------------------------
# DYNAMIC SETTINGS

STARTMONTH = 1
ENDMONTH = 12
YEAR = 2021
S1_ORBITDIRECTION = 'DESCENDING'

# ---------------------------------------------------
# Processing options for crop type map generation

PROCESSING_OPTIONS = {
    "start_month": STARTMONTH,
    "end_month": ENDMONTH,
    "year": YEAR,
    "masking": "mask_scl_dilation",
    "model_tag": "20230622T164738-MultiHeadResUnet3D_ts24_noAugm.zip",
    "model_dir": "https://artifactory.vgt.vito.be/auxdata-public/hrlvlcc/delineation",
    "sample_size": 128,
    "run_local": False,
    "cache_dir": None,
    # TODO: would it be better to transparently create a .cache folder in the current dir for this?
    # "cache_dir": ".cache",
    "temp_aggr": True,
    "target_crs": 3035
}


# ---------------------------------------------------
# Job options for OpenEO inference


DEFAULT_JOB_OPTIONS = {
    "driver-memory": "2G",
    "driver-memoryOverhead": "1G",
    "driver-cores": "1",
    "executor-memory": "1G",
    "executor-memoryOverhead": "2900m",
    "executor-cores": "1",
    "executor-request-cores": "600m",
    "max-executors": "22",
    "executor-threads-jvm": "7",
    "udf-dependency-archives": [
        "https://artifactory.vgt.vito.be/auxdata-public/hrlvlcc/delineation/lot_delineation_dep.zip#tmp/venv_static_del",
        "https://artifactory.vgt.vito.be/auxdata-public/hrlvlcc/openeo-dependencies/hrl.zip#tmp/venv_static",
        "https://artifactory.vgt.vito.be/auxdata-public/hrlvlcc/delineation/repositories_final_test.zip#tmp/venv_model",
    ],
    "logging-threshold": "info",
    "mount_tmp": False,
    "goofys": "false",
    "node_caching": True,
    "sentinel-hub": {
        "client-alias": "vito"
    }
}


TERRASCOPE_JOB_OPTIONS = {
    "task-cpus": 1,
    "executor-cores": 1,
    "max-executors": 100,
    "executor-threads-jvm": 12  # terrascope reads from geotiff instead of jp2, so no threading issue there
}

CREO_JOB_OPTIONS = {
    "driver-memory": "6G",
    "driver-memoryOverhead": "512M",
    "driver-cores": "2",
    "executor-memory": "3G", # seems to be required at the moment
    "executor-memoryOverhead": "5G",  # Seems to be required at the moment
    "executor-cores": "1",
    "executor-request-cores": "600m",
    "max-executors": "22",
    "logging-threshold": "info",
    "mount_tmp": False,
    "goofys": "false",
    "node_caching": True,
    "executor-threads-jvm":"7"
}



# ---------------------------------------------------
# Collection options for OpenEO inference

# Collection definitions on Terrascope
_TERRASCOPE_COLLECTIONS = {
    'S2_collection': "TERRASCOPE_S2_TOC_V2",
    'METEO_collection': None,
}

# Collection definitions on CREO
_CREO_COLLECTIONS = {
    'S2_collection': "SENTINEL2_L2A",
    'METEO_collection': None,
}



def _get_default_job_options(task: str = 'inference'):
    if task == 'inference':
        return deepcopy(DEFAULT_JOB_OPTIONS)
    elif task == 'preproces':
        return deepcopy(DEFAULT_JOB_OPTIONS)
    else:
        raise ValueError(f'Task `{task}` not known.')


def get_job_options(provider: str = None,
                    task: str = 'inference'):

    job_options = deepcopy(_get_default_job_options(task))

    if task == 'inference':
        if provider.lower() == 'terrascope':
            job_options.update(TERRASCOPE_JOB_OPTIONS)
        elif provider.lower() == 'creodias':
            job_options.update(CREO_JOB_OPTIONS)
        elif provider is None:
            pass
        else:
            raise ValueError(f'Provider `{provider}` not known.')

    return deepcopy(job_options)


def _get_default_processing_options():
    return deepcopy(PROCESSING_OPTIONS)


def get_processing_options(provider: str = 'terrascope'):
    provider = normalize_provider_str(provider)
    processing_options = _get_default_processing_options()
    processing_options.update({'provider': provider})
    return processing_options


class Provider(Enum):
    TERRASCOPE = 1
    CREODIAS = 2
    SENTINELHUB = 3


def normalize_provider_str(provider):
    prov = normalize_provider_enum(provider)
    return prov.name.lower()


def normalize_provider_enum(provider):
    if isinstance(provider, Provider):
        return provider

    if provider.lower() in ("terra", "terrascope"):
        return Provider.TERRASCOPE
    elif provider.lower() in ("creo", "creodias"):
        return Provider.CREODIAS
    elif provider.lower() in ("sentinelhub", "sentinel_hub", "shub"):
        return Provider.SENTINELHUB
    else:
        raise ValueError(f'Provider `{provider}` not known.')


def get_collection_options(provider: str):

    provider = normalize_provider_enum(provider)
    if provider == Provider.TERRASCOPE:
        return _TERRASCOPE_COLLECTIONS
    elif provider == Provider.CREODIAS:
        return _CREO_COLLECTIONS
