import streamlit as st
from pathlib import Path
import pandas as pd
import math
from tqdm import tqdm
pd.options.mode.chained_assignment = None  # default='warn'

def gray_shade(value, max_value, min_intensity=100):
    """
    Returns a CSS background color string for a gray shade based on the given value
    relative to the given max_value. The shade is scaled from min_intensity to 255.
    If value or max_value is NaN or max_value is 0, the function returns a default
    shade of min_intensity.
    
    Parameters
    ----------
    value : float
        The value to shade.
    max_value : float
        The maximum value to scale the shade.
    min_intensity : int, optional
        The minimum shade intensity (default is 100).
    
    Returns
    -------
    str
        A CSS background color string for the gray shade.
    """
    if math.isnan(value) or math.isnan(max_value) or max_value == 0:
        return f'background-color: rgb({min_intensity}, {min_intensity}, {min_intensity})'  # Default to min_intensity if value or max_value is NaN or max_value is 0
    gray_intensity = int(min_intensity + (155 * (value / max_value)))  # Scale from min_intensity to 255
    return f'background-color: rgb({gray_intensity}, {gray_intensity}, {gray_intensity})'

def apply_color_shade(df):
    """Apply green shade to columns ending with '_counts'."""
    max_values = df.max()
    
    def style_column(col):
        return col.apply(lambda v: gray_shade(v, max_values[col.name]) if col.name.endswith('_counts') else '')
    
    styled_df = df.style.apply(style_column, axis=0)
    return styled_df
    
def process_log_file(log_file_path: str, model_name: str) -> pd.DataFrame:
    """Process a log file (csv) generated by tensorboard and return a DataFrame with the metrics of the best step (lowest val_loss) as columns and the model name as an additional column.

    Args:
        log_file_path (str): Path to the log file (csv) generated by tensorboard.
        model_name (str): Name of the model.

    Returns:
        pd.DataFrame: DataFrame with the metrics of the best step as columns and the model name as an additional column.
    """
    log_df = pd.read_csv(log_file_path).reset_index(drop=True)
    min_val_loss_index = log_df.loc[log_df['tag'] == 'val_loss', 'value'].idxmin()
    best_step = log_df.loc[min_val_loss_index, 'step']
    best_step_metrics = log_df.loc[log_df['step'] == best_step][['tag', 'value']]
    test_metrics = log_df.loc[log_df['tag'].str.startswith('test_'), ['tag', 'value']]
    metrics_df = pd.concat([best_step_metrics, test_metrics], ignore_index=True)
    metrics_df.drop_duplicates(inplace=True)
    metrics_df.set_index('tag', inplace=True)
    metrics_df = metrics_df.T
    metrics_df['model'] = model_name
    metrics_df.drop_duplicates(inplace=True)
    metrics_df = metrics_df.loc[:, ~metrics_df.columns.duplicated()]
    return metrics_df

def get_latest_files(selected_model: str, selected_model_path_csvs: Path, path_models: str):
    """
    Get the latest files in the selected model path.

    Parameters
    ----------
    selected_model : str
        Name of the model to get the latest files for.
    selected_model_path_csvs : Path
        Path to the directory containing the CSV files.
    path_models : str
        Path to the directory containing the model directories.

    Returns
    -------
    latest_csv : Path
        Path to the latest CSV file.
    log_file : Path
        Path to the log file containing the metrics for the latest model.
    """
    latest_files = sorted(selected_model_path_csvs.glob("*.csv"), key=lambda x: x.stat().st_mtime, reverse=True)
    if not latest_files:
        st.error(f"No csv files found in {selected_model_path_csvs}/")
    latest_csv = latest_files[0]

    selected_model_path_original = Path(path_models) / selected_model
    log_file = selected_model_path_original / "version_0" / "tbparse_scalars.csv"

    if not log_file.exists():
        from tbparse import SummaryReader

        with st.spinner('tbparse_scalars.csv not found. Creating it...'):
            log_dir = selected_model_path_original / "version_0"
            reader = SummaryReader(log_dir)
            df = reader.scalars
            df.to_csv(log_file, index=False)

    return latest_csv, log_file

def create_difference_columns(df: pd.DataFrame, class_names: list) -> pd.DataFrame:
    """
    Create new columns in the dataframe df that contain the differences between
    the model and scl counts for each class in class_names.

    Parameters
    ----------
    df : pd.DataFrame
        The dataframe to modify. This dataframe is expected to have columns
        with names like "model_CLASSNAME_counts" and "scl_CLASSNAME_counts"
        where CLASSNAME is one of the class names in the class_names list.
    class_names : list
        List of class names for which to create the difference columns.

    Returns
    -------
    The modified dataframe with the new columns. The new columns will be named
    "diff_CLASSNAME" where CLASSNAME is one of the class names in the
    class_names list.

    Notes
    -----
    This function assumes that the dataframe df has columns with names like
    "model_CLASSNAME_counts" and "scl_CLASSNAME_counts" where CLASSNAME is one
    of the class names in the class_names list. These columns should contain
    the counts of the model and scl, respectively, for the given class. The
    function will then create new columns with the differences between the
    model and scl counts for each class. If the columns do not exist in the
    dataframe, the function will print a warning message and do nothing.
    """

    # Iterate over all class names
    for class_name in class_names:
        # Construct the column names for the model and scl counts
        model_col = f"model_{class_name}_counts"
        scl_col = f"scl_{class_name}_counts"

        # Check if the columns exist in the dataframe
        if model_col in df.columns and scl_col in df.columns:
            # Construct the column name for the difference
            diff_col = f"diff_{class_name}"

            # Calculate the difference between the model and scl counts
            df[diff_col] = df[model_col] - df[scl_col]
        else:
            # Print a warning message if the columns do not exist
            print(f"Columns {model_col} and/or {scl_col} not found in DataFrame.")
    
    # Return the modified dataframe
    return df

def make_comparison_metric(test_iou: float, val_iou: float, diff_std: float, all_diff_stds: list) -> float:
    """
    This function takes a list of metrics and calculates a score from 0 to 1
    It rewards models with high test_iou, val_iou and low diff_std
    """
    
    # Normalize the diff_std to be between 0 and 1
    diff_std = (diff_std - min(all_diff_stds)) / (max(all_diff_stds) - min(all_diff_stds))
    
    # Calculate the score
    score = (test_iou + val_iou) * (1 - diff_std)
    
    return score


# Find all directories in path_models that have a file named
# "test_results.json" inside the "version_0" subdirectory. This is
# used to filter out directories that are not valid models.
path_models = '/vitodata/vegteam_vol2/models/cloudsen/80m'
path_csvs = '/data/users/Public/kalfasyan/share/outputs/csvs'
model_dirs = list(Path(path_models).glob("*"))
models = [model_dir for model_dir in model_dirs if (model_dir / "version_0/test_results.json").exists()]

model_to_names = {model: model.name for model in models}
inv_model_to_names = {v: k for k, v in model_to_names.items()}

def main():
    # Let's make 2 tabs: one for the model matrix and one for all models' matrix
    tab1, tab2 = st.tabs(['Model matrix', 'All models matrix'])

    with tab1:
        # Select which model to show matrix for
        selected_model = st.selectbox('Select model to show matrix', model_to_names.values(), index=0)

        # Construct the path to the CSV file and the log file
        selected_model_path_csvs = Path(path_csvs) / selected_model
        selected_model_path_original = Path(path_models) / selected_model

        # Get the latest CSV file and log file for the selected model
        latest_csv, log_file = get_latest_files(selected_model, selected_model_path_csvs, path_models)

        # If the log file exists, read it and display the metrics
        if log_file.exists():
            df_metrics = process_log_file(log_file, selected_model)
            st.dataframe(df_metrics)

        # Read the CSV file with pandas
        df_csv = pd.read_csv(latest_csv)
        df_csv['model'] = selected_model

        # Filter out any column starting with 'iou'
        df_csv = df_csv.loc[:, ~df_csv.columns.str.startswith('iou')]

        # Warn if there are NaN values
        if df_csv.isnull().values.any():
            st.warning(f"Columns with NaN values: {df_csv.columns[df_csv.isnull().any()].tolist()}")
            # Fill NaN values with 0
            df_csv.fillna(0, inplace=True)

        # Normalize each column that ends with _counts by dividing by the max possible (1372x1372)
        counts_columns = df_csv.columns[df_csv.columns.str.endswith('_counts')]
        df_csv.loc[:, counts_columns] = df_csv.loc[:, counts_columns].astype('float64')#.div(1372*1372).astype('float64')

        # Display the dataframe
        st.dataframe(apply_color_shade(df_csv))

    with tab2:
        st.write("All models matrix")
        # In this tab we will display the merged dataframe+dataframe_metrics for all models
        # We will read the csv dataframe (calculated during inference; contains counts of model/scl)
        # and the log dataframe (calculated during training; contains metrics)
        # and merge them on the model name. We will also add a column with the average of the values
        # in the diff columns
        # with st.spinner('Loading data...'):
        df_all = pd.DataFrame()
        with st.spinner('Loading data...'):
            for selected_model in tqdm(models, desc="Processing models", total=len(models)):
                try:
                    # st.write(f"Processing {selected_model.name}...")
                    selected_model_path_csvs = Path(path_csvs) / selected_model.name
                    latest_csv, log_file = get_latest_files(selected_model.name, selected_model_path_csvs, path_models)

                    # Read the csv dataframe (calculated during inference; contains counts of model/scl)
                    df_csv = pd.read_csv(latest_csv)
                    df_csv['model'] = selected_model.name
                    # filter out any column starting with 'iou'
                    df_csv = df_csv.loc[:, ~df_csv.columns.str.startswith('iou')]
                    # warn if there are NaN values
                    if df_csv.isnull().values.any():
                        st.warning(f"Columns with NaN values: {df_csv.columns[df_csv.isnull().any()].tolist()}")
                        # fill NaN values with 0
                        df_csv.fillna(0, inplace=True)

                    # get columns that end with _counts 
                    counts_columns = df_csv.columns[df_csv.columns.str.endswith('_counts')]
                    df_counts = df_csv.loc[:, counts_columns]
                    df_counts = create_difference_columns(df_csv.copy(), ['surface','clouds','thin_clouds','shadows'])
                    # let's take the avg of diff_surface, diff_clouds, diff_thin_clouds, diff_shadows as another df_metrics_diff
                    df_counts_diff = df_counts[['diff_surface','diff_clouds','diff_thin_clouds','diff_shadows']].mean(axis=0)
                    df_counts_diff['model'] = selected_model.name
                    df_counts_diff = df_counts_diff.to_frame().T
                    df_counts_diff = df_counts_diff.loc[:,~df_counts_diff.columns.duplicated()]
                    # Let's add a column with the average of the values in the diff columns
                    df_counts_diff['diff_std'] = df_counts_diff[['diff_surface','diff_clouds','diff_thin_clouds','diff_shadows']].std(axis=1)

                    # Read the log dataframe (calculated during training; contains metrics)
                    df_metrics = process_log_file(log_file, selected_model.name)

                    # assert the model column is the same in both dataframes before merging
                    assert df_metrics['model'].values[0] == df_counts_diff['model'].values[0], f"Model name mismatch: {df_metrics['model'].values[0]} != {df_counts_diff['model'].values[0]}"

                    # Merge df_metrics_diff with df_metrics on the model name
                    df_metrics_merged = pd.merge(df_metrics, df_counts_diff, on='model')
                    df_metrics_merged = df_metrics_merged.loc[:,~df_metrics_merged.columns.duplicated()]

                    df_all = pd.concat([df_all, df_metrics_merged], ignore_index=True)
                except Exception as e:
                    st.error(f"Error processing {selected_model.name}: {e}")
                    continue

            # # Display the dataframe
            df_all.drop('train_loss_step', axis=1, inplace=True)
            # drop any column that starts with "DeviceStats"
            df_all = df_all.loc[:, ~df_all.columns.str.startswith('DeviceStats')]
            # add column with comparison metric
            all_diff_stds = df_all['diff_std'].tolist()
            df_all['comparison_metric'] = df_all.apply(lambda x: make_comparison_metric(x['test_iou_score'], x['val_iou_score'], x['diff_std'], all_diff_stds), axis=1)
            df_all.set_index('model', inplace=True)
            st.dataframe(df_all)
            
            # Check which models are not in the dataframe
            missing_models = set(model_to_names.values()) - set(df_all.index.unique())
            if missing_models:
                st.warning(f"Missing models: {missing_models}")

    
if __name__ == '__main__':
    main()