from cropclass.features import (tsteps, to_db, to_pwr,
                                divide_10000)
import numpy as np


def compute_n_tsteps(start_month: int, end_month: int,
                     steps_per_month: int = 3) -> int:
    """Method to compute the amount of tsteps needed to cover
    period from start_month to end_month

    Args:
        start_month (int): start month
        end_month (int): end_month
        steps_per_month (int, optional): then amount of tsteps
            in one month (defaults to 3 = dekadal resolution)

    Returns:
        int: the required amount of tsteps
    """

    if start_month < end_month:
        nr_months = end_month - start_month + 1
    else:
        # start month from previous year
        nr_months = end_month + (12 - start_month + 1)

    return nr_months * steps_per_month


def get_preprocessing_settings():
    return {
        'S2':
            {
                'dtype': np.float32,
                'pre_func': divide_10000,
                'bands': ["B02", "B03", "B04", "B05", "B06",
                          "B07", "B08", "B11", "B12"],
                'composite': {
                    'freq': 'dekad',
                    'start': None,
                    'end': None
                },
                'interpolate': True,
                'mask': {
                    'scl_mask_values': [0, 1, 3, 8, 9, 10],
                    'erode_r': 3,
                    'dilate_r': 13,
                    'multitemporal': True
                }
            },
        'S1':
            {
                'dtype': np.float32,
                'bands': ['VV', 'VH'],
                'pre_func': to_pwr,
                'post_func': to_db,
                'composite': {
                    'freq': 'dekad',
                    'mode': 'mean',
                    'start': None,
                    'end': None
                },
                'interpolate': True
            },
        'METEO':
            {
                'dtype': np.float32,
                'bands': ['temperature_mean'],
                'composite': {
                    'freq': 'dekad',
                    'mode': 'mean',
                    'start': None,
                    'end': None
                },
                'interpolate': False  # Should not be needed
            },
    }


def get_openeo_preprocessing_settings():

    from cropclass.classification import get_preproc_settings

    settings = get_preproc_settings()
    # add composite settings
    for sensor in settings.keys():
        if sensor == 'S2':
            settings[sensor]['composite'] = {
                'freq': 'dekad',
                'start': None,
                'end': None}
        else:
            settings[sensor]['composite'] = {
                'freq': 'dekad',
                'mode': 'mean',
                'start': None,
                'end': None
            }

    return settings


def get_feature_settings(n_tsteps):

    return {
        'S2':
        {
            "tsteps": {
                "function": tsteps,
                "parameters": {
                    'n_steps': n_tsteps,
                },
                "names": [f'ts{i}' for i in range(n_tsteps)]
            },
        },
        'S1':
        {
            "tsteps": {
                "function": tsteps,
                "parameters": {
                    'n_steps': n_tsteps,
                },
                "names": [f'ts{i}' for i in range(n_tsteps)]
            }
        },
        'METEO':
        {
            "tsteps": {
                "function": tsteps,
                "parameters": {
                    'n_steps': n_tsteps,
                },
                "names": [f'ts{i}' for i in range(n_tsteps)]
            }
        }
    }
