"""
Script with evaluation metrics to determine the perforamce
of the method
"""

import numpy as np
import pandas as pd
from sklearn.metrics import r2_score


def precision(pred, obs):
    """
    Returns:
        Precision (=unbiased RMSE) between the observation and reference dataset
    Args:
        x
        y
    """
    A = np.nanmean(pred - obs)
    one_over_n1 = 1. / (np.size(pred) - 1)
    epsilon = pred - obs
    if one_over_n1 * np.nansum((epsilon - A)**2) >= 0:
        prec = np.sqrt(one_over_n1 * np.nansum((epsilon - A)**2))
    else:
        prec = np.nan
    return prec



def RMSE(pred, obs):
    return np.sqrt(((pred-obs)**2).mean())


def MAE(pred, obs):
    return np.round(np.mean(np.abs(pred-obs)), 2)


def get_val_metrics(pred, obs, index):
    seasons = {
        'DJF': [12, 1, 2],
        'MAM': [3, 4, 5],
        'JJA': [6, 7, 8],
        'SON': [9, 10, 11],
        'all': [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]
    }
    df_pred_obs = pd.DataFrame({'date': pd.to_datetime(index),
                                'pred': pred, 'obs': obs})
    val_results = {}
    for season, months in seasons.items():
        # Filter the data for the specific months of the season
        season_mask = df_pred_obs.date.dt.month.isin(months)
        obs_season = df_pred_obs[season_mask].obs
        pred_season = df_pred_obs[season_mask].pred
        RMSE_out = np.round(RMSE(pred_season, obs_season), 2)
        MAE_out = np.round(MAE(pred_season, obs_season), 2)
        R2_out = np.round(r2_score(obs_season, pred_season), 2)
        Prec_out = np.round(precision(pred_season, obs_season), 2)
        val_results[season] = {'RMSE': RMSE_out,
                               'MAE': MAE_out,
                               'PRECISION': Prec_out,
                               'R2_out': R2_out}
    return val_results